package com.example.controldecolonias.ui.navigation

import androidx.compose.foundation.layout.padding
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.filled.*
import androidx.compose.material3.Icon
import androidx.compose.material3.NavigationBar
import androidx.compose.material3.NavigationBarItem
import androidx.compose.material3.Scaffold
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.runtime.remember
import androidx.compose.ui.Modifier
import androidx.navigation.NavDestination.Companion.hierarchy
import androidx.navigation.NavGraph.Companion.findStartDestination
import androidx.navigation.NavType
import androidx.navigation.compose.NavHost
import androidx.navigation.compose.composable
import androidx.navigation.compose.currentBackStackEntryAsState
import androidx.navigation.compose.rememberNavController
import androidx.navigation.navArgument
import com.example.controldecolonias.ColoniaViewModel
import com.example.controldecolonias.ui.screens.colonias.*
import com.example.controldecolonias.ui.screens.herramientas.PantallaHerramientas
import com.example.controldecolonias.ui.screens.nidos.*
import com.example.controldecolonias.ui.screens.dashboard.PantallaPrincipalDashboard
import com.example.controldecolonias.ui.screens.mapa.PantallaMapaNidos
import com.example.controldecolonias.ui.screens.tareas.PantallaPrincipalTareas

sealed class Pantalla(val ruta: String, val label: String, val icon: androidx.compose.ui.graphics.vector.ImageVector) {
    object Dashboard : Pantalla("dashboard", "Dashboard", Icons.Default.Dashboard)
    object Colonias : Pantalla("colonias", "Colonias", Icons.Default.Home)
    object Nidos : Pantalla("nidos", "Nidos", Icons.Default.LocationOn)
    object Tareas : Pantalla("tareas", "Tareas", Icons.Default.Checklist)
}

object Rutas {
    // MODIFICADO: Añadido el parámetro 'idYaExiste'
    const val ANADIR_EDITAR_COLONIA = "anadirEditarColonia?coloniaId={coloniaId}&scannedId={scannedId}&coloniaJson={coloniaJson}&idYaExiste={idYaExiste}"
    const val DETALLE_COLONIA = "detalleColonia/{coloniaId}"
    const val ANADIR_EDITAR_NIDO = "anadirEditarNido?nidoId={nidoId}"
    const val DETALLE_NIDO = "detalleNido/{nidoId}"
    const val HERRAMIENTAS = "herramientas"
    const val MAPA_NIDOS = "mapaNidos"
}

@Composable
fun ControlDeColoniasApp(viewModel: ColoniaViewModel) {
    val navController = rememberNavController()
    val ninjaModeActivado by viewModel.ninjaModeActivado.collectAsState()

    val pantallas = remember(ninjaModeActivado) {
        if (ninjaModeActivado) {
            listOf(Pantalla.Dashboard, Pantalla.Colonias, Pantalla.Tareas)
        } else {
            listOf(Pantalla.Dashboard, Pantalla.Colonias, Pantalla.Nidos, Pantalla.Tareas)
        }
    }

    Scaffold(
        bottomBar = {
            NavigationBar {
                val navBackStackEntry by navController.currentBackStackEntryAsState()
                val currentDestination = navBackStackEntry?.destination
                pantallas.forEach { pantalla ->
                    NavigationBarItem(
                        icon = { Icon(pantalla.icon, contentDescription = null) },
                        label = { Text(pantalla.label) },
                        selected = currentDestination?.hierarchy?.any { it.route == pantalla.ruta } == true,
                        onClick = {
                            navController.navigate(pantalla.ruta) {
                                popUpTo(navController.graph.findStartDestination().id) {
                                    saveState = true
                                }
                                launchSingleTop = true
                                restoreState = true
                            }
                        }
                    )
                }
            }
        }
    ) { innerPadding ->
        NavHost(navController, startDestination = Pantalla.Dashboard.ruta, Modifier.padding(innerPadding)) {
            composable(Pantalla.Colonias.ruta) {
                PantallaPrincipalColonias(navController = navController, viewModel = viewModel)
            }
            composable(Pantalla.Nidos.ruta) {
                PantallaPrincipalNidos(navController = navController, viewModel = viewModel)
            }
            composable(Pantalla.Dashboard.ruta) {
                PantallaPrincipalDashboard(navController = navController, viewModel = viewModel)
            }
            composable(Pantalla.Tareas.ruta) {
                PantallaPrincipalTareas(viewModel = viewModel)
            }
            composable(Rutas.MAPA_NIDOS) {
                PantallaMapaNidos(viewModel = viewModel)
            }
            composable(
                route = Rutas.ANADIR_EDITAR_COLONIA,
                arguments = listOf(
                    navArgument("coloniaId") { type = NavType.StringType; nullable = true },
                    navArgument("scannedId") { type = NavType.StringType; nullable = true },
                    navArgument("coloniaJson") { type = NavType.StringType; nullable = true },
                    // MODIFICADO: Definido el nuevo argumento
                    navArgument("idYaExiste") { type = NavType.BoolType; defaultValue = false }
                )
            ) { backStackEntry ->
                val coloniaId = backStackEntry.arguments?.getString("coloniaId")
                val scannedId = backStackEntry.arguments?.getString("scannedId")
                val coloniaJson = backStackEntry.arguments?.getString("coloniaJson")
                // MODIFICADO: Leído el nuevo argumento
                val idYaExiste = backStackEntry.arguments?.getBoolean("idYaExiste") ?: false
                PantallaAnadirEditarColonia(
                    coloniaId = coloniaId,
                    scannedId = scannedId,
                    coloniaJson = coloniaJson,
                    idYaExiste = idYaExiste, // Pasado a la pantalla
                    viewModel = viewModel,
                    onNavigateBack = { navController.popBackStack() },
                    onGuardarColonia = { colonia ->
                        viewModel.insertar(colonia)
                    }
                )
            }
            composable(route = Rutas.DETALLE_COLONIA) { backStackEntry ->
                val coloniaId = backStackEntry.arguments?.getString("coloniaId")
                if (coloniaId != null) {
                    PantallaDetalleColonia(
                        coloniaId = coloniaId, viewModel = viewModel,
                        onNavigateBack = { navController.popBackStack() },
                        onNavigateToEditar = { id -> navController.navigate("anadirEditarColonia?coloniaId=$id") }
                    )
                }
            }
            composable(route = Rutas.ANADIR_EDITAR_NIDO) { backStackEntry ->
                val nidoId = backStackEntry.arguments?.getString("nidoId")
                PantallaAnadirEditarNido(
                    nidoId = nidoId, viewModel = viewModel,
                    onNavigateBack = { navController.popBackStack() },
                    onGuardarNido = { nido -> viewModel.insertarNido(nido); navController.popBackStack() }
                )
            }
            composable(route = Rutas.DETALLE_NIDO) { backStackEntry ->
                val nidoId = backStackEntry.arguments?.getString("nidoId")
                if (nidoId != null) {
                    PantallaDetalleNido(
                        nidoId = nidoId, viewModel = viewModel,
                        onNavigateBack = { navController.popBackStack() },
                        onNavigateToEditar = { id -> navController.navigate("anadirEditarNido?nidoId=$id") }
                    )
                }
            }
            composable(Rutas.HERRAMIENTAS) {
                PantallaHerramientas(
                    viewModel = viewModel,
                    onNavigateBack = { navController.popBackStack() }
                )
            }
        }
    }
}