package com.example.controldecolonias

import androidx.room.*
import kotlinx.coroutines.flow.Flow

@Dao
interface ColoniaDao {
    // --- Colonias ---
    @Query("SELECT * FROM colonias_table ORDER BY fecha DESC")
    fun obtenerTodasLasColonias(): Flow<List<Colonia>>

    @Transaction
    @Query("SELECT * FROM colonias_table WHERE id = :id")
    fun obtenerColoniaFullDetail(id: String): Flow<ColoniaFullDetail?>

    @Insert(onConflict = OnConflictStrategy.REPLACE)
    suspend fun insertar(colonia: Colonia)

    @Update
    suspend fun actualizar(colonia: Colonia)

    @Delete
    suspend fun borrar(colonia: Colonia)

    @Query("SELECT EXISTS(SELECT 1 FROM colonias_table WHERE id = :id)")
    suspend fun idExiste(id: String): Boolean

    @Query("SELECT * FROM colonias_table WHERE id = :id")
    suspend fun obtenerColoniaPorIdSync(id: String): Colonia?

    // --- Notas ---
    @Insert
    suspend fun insertarNota(nota: Nota)
    @Update
    suspend fun actualizarNota(nota: Nota)
    @Delete
    suspend fun borrarNota(nota: Nota)

    // --- Nidos ---
    @Query("SELECT * FROM nidos_table ORDER BY fecha DESC")
    fun obtenerTodosLosNidos(): Flow<List<Nido>>
    @Query("SELECT * FROM nidos_table WHERE id = :id")
    fun obtenerNidoPorId(id: String): Flow<Nido?>
    @Insert(onConflict = OnConflictStrategy.REPLACE)
    suspend fun insertarNido(nido: Nido)
    @Delete
    suspend fun borrarNido(nido: Nido)

    // --- Alimentación ---
    @Query("SELECT * FROM alimentacion_table WHERE coloniaId = :coloniaId ORDER BY fecha DESC")
    fun obtenerAlimentacionPorColoniaId(coloniaId: String): Flow<List<Alimentacion>>
    @Insert
    suspend fun insertarAlimentacion(registro: Alimentacion)
    @Update
    suspend fun actualizarAlimentacion(registro: Alimentacion)
    @Delete
    suspend fun borrarAlimentacion(registro: Alimentacion)

    // --- Tareas ---
    @Query("SELECT * FROM tareas_table ORDER BY completada ASC, fechaLimite ASC")
    fun obtenerTodasLasTareas(): Flow<List<Tarea>>
    @Insert
    suspend fun insertarTarea(tarea: Tarea): Long
    @Update
    suspend fun actualizarTarea(tarea: Tarea)
    @Delete
    suspend fun borrarTarea(tarea: Tarea)

    // --- Fotos ---
    @Insert
    suspend fun insertarFoto(foto: Foto)
    @Delete
    suspend fun borrarFoto(foto: Foto)

    // --- Exportación ---
    @Query("SELECT * FROM colonias_table")
    suspend fun exportarColonias(): List<Colonia>
    @Query("SELECT * FROM notas_table")
    suspend fun exportarNotas(): List<Nota>
    @Query("SELECT * FROM nidos_table")
    suspend fun exportarNidos(): List<Nido>
    @Query("SELECT * FROM alimentacion_table")
    suspend fun exportarAlimentacion(): List<Alimentacion>
    @Query("SELECT * FROM fotos_table")
    suspend fun exportarFotos(): List<Foto>
    @Query("SELECT * FROM tareas_table")
    suspend fun exportarTareas(): List<Tarea>

    // --- Importación (Borrado Masivo) ---
    @Query("DELETE FROM fotos_table")
    suspend fun borrarTodasLasFotos()
    @Query("DELETE FROM tareas_table")
    suspend fun borrarTodasLasTareas()
    @Query("DELETE FROM notas_table")
    suspend fun borrarTodasLasNotas()
    @Query("DELETE FROM alimentacion_table")
    suspend fun borrarTodaLaAlimentacion()
    @Query("DELETE FROM colonias_table")
    suspend fun borrarTodasLasColonias()
    @Query("DELETE FROM nidos_table")
    suspend fun borrarTodosLosNidos()

    @Transaction
    suspend fun importarBackup(backup: BackupData) {
        // Borramos todo en el orden correcto para evitar conflictos de claves foráneas
        borrarTodaLaAlimentacion()
        borrarTodasLasNotas()
        borrarTodasLasFotos()
        borrarTodasLasColonias() // Ahora se pueden borrar las colonias

        borrarTodosLosNidos()
        borrarTodasLasTareas()

        // Insertamos los nuevos datos
        backup.colonias.forEach { insertar(it) }
        backup.nidos.forEach { insertarNido(it) }
        backup.notas.forEach { insertarNota(it) }
        backup.alimentacion.forEach { insertarAlimentacion(it) }
        backup.tareas.forEach { insertarTarea(it) }
        backup.fotos.forEach { insertarFoto(it) }
    }

    // --- Dashboard ---
    @Query("SELECT COUNT(*) FROM colonias_table")
    fun contarColonias(): Flow<Int>
    @Query("SELECT COUNT(*) FROM nidos_table")
    fun contarNidos(): Flow<Int>
    @Query("SELECT COUNT(*) FROM colonias_table WHERE estaEnferma = 1")
    fun contarColoniasEnfermas(): Flow<Int>

    // --- Exportación Selectiva ---
    @Query("SELECT * FROM notas_table WHERE coloniaId = :coloniaId")
    suspend fun exportarNotasDeColonia(coloniaId: String): List<Nota>
    @Query("SELECT * FROM alimentacion_table WHERE coloniaId = :coloniaId")
    suspend fun exportarAlimentacionDeColonia(coloniaId: String): List<Alimentacion>
}