"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const Errors_1 = require("../Errors");
const modulePluginResolver_1 = require("./modulePluginResolver");
/**
 * Resolves static module plugin and potentially falls back on a provided plugin if the module cannot be resolved
 *
 * @param config
 * @param fallback Plugin with `_resolverError` explaining why the module couldn't be used
 * @param projectRoot optional project root, fallback to _internal.projectRoot. Used for testing.
 */
exports.withStaticPlugin = (config, props) => {
    var _a;
    let projectRoot = props.projectRoot;
    if (!projectRoot) {
        projectRoot = (_a = config._internal) === null || _a === void 0 ? void 0 : _a.projectRoot;
        modulePluginResolver_1.assertInternalProjectRoot(projectRoot);
    }
    let [pluginResolve, pluginProps] = modulePluginResolver_1.normalizeStaticPlugin(props.plugin);
    // Ensure no one uses this property by accident.
    Errors_1.assert(!(pluginProps === null || pluginProps === void 0 ? void 0 : pluginProps._resolverError), `Plugin property '_resolverError' is a reserved property of \`withStaticPlugin\``);
    let withPlugin;
    // Function was provided, no need to resolve: [withPlugin, {}]
    if (typeof pluginResolve === 'function') {
        withPlugin = pluginResolve;
    }
    else if (typeof pluginResolve === 'string') {
        try {
            // Resolve and evaluate plugins.
            withPlugin = modulePluginResolver_1.resolveConfigPluginFunction(projectRoot, pluginResolve);
        }
        catch (error) {
            // If the static module failed to resolve, attempt to use a fallback.
            // This enables support for built-in plugins with versioned variations living in other packages.
            if (props.fallback) {
                if (!pluginProps)
                    pluginProps = {};
                // Pass this to the fallback plugin for potential warnings about needing to install a versioned package.
                pluginProps._resolverError = error;
                withPlugin = props.fallback;
            }
            else {
                // If no fallback, throw the resolution error.
                throw error;
            }
        }
    }
    else {
        throw new Errors_1.ConfigError(`Static plugin is an unexpected type: ${typeof pluginResolve}`, 'INVALID_PLUGIN');
    }
    // Execute the plugin.
    config = withPlugin(config, pluginProps);
    return config;
};
//# sourceMappingURL=withStaticPlugin.js.map