import URI from "urijs";
export { deriveKey, ready } from "./Crypto";
export * from "./Exceptions";
import { base64 } from "./Helpers";
import { CollectionAccessLevel, EncryptedCollection, EncryptedCollectionItem, SignedInvitationRead, SignedInvitationWrite, ChunkJson } from "./EncryptedModels";
export interface User {
    username: string;
    email: string;
}
export interface LoginResponseUser extends User {
    pubkey: Uint8Array;
    encryptedContent: Uint8Array;
}
export interface UserProfile {
    pubkey: Uint8Array;
}
export declare type LoginChallange = {
    challenge: Uint8Array;
    salt: Uint8Array;
    version: number;
};
export declare type LoginChallangeResponse = {
    challenge: Uint8Array;
    host: string;
};
export declare type LoginResponse = {
    token: string;
    user: LoginResponseUser;
};
export interface ListResponse<T> {
    data: T[];
    done: boolean;
}
export interface CollectionItemListResponse<T> extends ListResponse<T> {
    stoken: string;
}
export interface CollectionListResponse<T> extends CollectionItemListResponse<T> {
    removedMemberships?: RemovedCollection[];
}
export interface IteratorListResponse<T> extends ListResponse<T> {
    iterator: string;
}
export declare type CollectionMemberListResponse<T> = IteratorListResponse<T>;
export declare type CollectionInvitationListResponse<T> = IteratorListResponse<T>;
export interface RemovedCollection {
    uid: base64;
}
export interface CollectionMember {
    username: string;
    accessLevel: CollectionAccessLevel;
}
export interface AcceptedInvitation {
    encryptionKey: Uint8Array;
}
export declare enum PrefetchOption {
    Auto = "auto",
    Medium = "medium"
}
export interface ListFetchOptions {
    limit?: number;
}
export interface FetchOptions extends ListFetchOptions {
    stoken?: string | null;
    prefetch?: PrefetchOption;
}
export interface ItemFetchOptions extends FetchOptions {
    withCollection?: boolean;
}
export interface IteratorFetchOptions extends ListFetchOptions {
    iterator?: string | null;
}
export declare type MemberFetchOptions = IteratorFetchOptions;
export declare type InvitationFetchOptions = IteratorFetchOptions;
export interface RevisionsFetchOptions extends IteratorFetchOptions {
    prefetch?: PrefetchOption;
}
interface AccountOnlineData {
    serverUrl: string;
    authToken: string | null;
}
declare class BaseNetwork {
    static urlExtend(baseUrlIn: URI, segments: string[]): URI;
    apiBase: URI;
    constructor(apiBase: string);
    newCall<T = any>(segments?: string[], extra?: RequestInit, apiBaseIn?: URI): Promise<T>;
}
export declare class Authenticator extends BaseNetwork {
    constructor(apiBase: string);
    isEtebase(): Promise<boolean>;
    signup(user: User, salt: Uint8Array, loginPubkey: Uint8Array, pubkey: Uint8Array, encryptedContent: Uint8Array): Promise<LoginResponse>;
    getLoginChallenge(username: string): Promise<LoginChallange>;
    login(response: Uint8Array, signature: Uint8Array): Promise<LoginResponse>;
    logout(authToken: string): Promise<void>;
    changePassword(authToken: string, response: Uint8Array, signature: Uint8Array): Promise<void>;
    getDashboardUrl(authToken: string): Promise<string>;
}
declare class BaseManager extends BaseNetwork {
    protected etebase: AccountOnlineData;
    constructor(etebase: AccountOnlineData, segments: string[]);
    newCall<T = any>(segments?: string[], extra?: RequestInit, apiBase?: URI): Promise<T>;
    protected urlFromFetchOptions(options?: ItemFetchOptions & IteratorFetchOptions): URI;
}
export declare class CollectionManagerOnline extends BaseManager {
    constructor(etebase: AccountOnlineData);
    fetch(colUid: string, options?: FetchOptions): Promise<EncryptedCollection>;
    list(collectionTypes: Uint8Array[], options?: FetchOptions): Promise<CollectionListResponse<EncryptedCollection>>;
    create(collection: EncryptedCollection, options?: FetchOptions): Promise<{}>;
}
export declare class CollectionItemManagerOnline extends BaseManager {
    constructor(etebase: AccountOnlineData, col: EncryptedCollection);
    fetch(itemUid: string, options?: ItemFetchOptions): Promise<EncryptedCollectionItem>;
    list(options?: ItemFetchOptions): Promise<CollectionItemListResponse<EncryptedCollectionItem>>;
    itemRevisions(item: EncryptedCollectionItem, options?: RevisionsFetchOptions): Promise<IteratorListResponse<EncryptedCollectionItem>>;
    create(item: EncryptedCollectionItem): Promise<{}>;
    fetchUpdates(items: EncryptedCollectionItem[], options?: ItemFetchOptions): Promise<CollectionItemListResponse<EncryptedCollectionItem>>;
    fetchMulti(items: base64[], options?: ItemFetchOptions): Promise<CollectionItemListResponse<EncryptedCollectionItem>>;
    batch(items: EncryptedCollectionItem[], deps?: EncryptedCollectionItem[], options?: ItemFetchOptions): Promise<{}>;
    transaction(items: EncryptedCollectionItem[], deps?: EncryptedCollectionItem[], options?: ItemFetchOptions): Promise<{}>;
    chunkUpload(item: EncryptedCollectionItem, chunk: ChunkJson, options?: ItemFetchOptions): Promise<{}>;
    chunkDownload(item: EncryptedCollectionItem, chunkUid: base64, options?: ItemFetchOptions): Promise<Uint8Array>;
}
export declare class CollectionInvitationManagerOnline extends BaseManager {
    constructor(etebase: AccountOnlineData);
    listIncoming(options?: InvitationFetchOptions): Promise<CollectionInvitationListResponse<SignedInvitationRead>>;
    listOutgoing(options?: InvitationFetchOptions): Promise<CollectionInvitationListResponse<SignedInvitationRead>>;
    accept(invitation: SignedInvitationRead, collectionType: Uint8Array, encryptionKey: Uint8Array): Promise<{}>;
    reject(invitation: SignedInvitationRead): Promise<{}>;
    fetchUserProfile(username: string): Promise<UserProfile>;
    invite(invitation: SignedInvitationWrite): Promise<{}>;
    disinvite(invitation: SignedInvitationRead): Promise<{}>;
}
export declare class CollectionMemberManagerOnline extends BaseManager {
    constructor(etebase: AccountOnlineData, col: EncryptedCollection);
    list(options?: MemberFetchOptions): Promise<CollectionMemberListResponse<CollectionMember>>;
    remove(username: string): Promise<{}>;
    leave(): Promise<{}>;
    modifyAccessLevel(username: string, accessLevel: CollectionAccessLevel): Promise<{}>;
}
