"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const estraverse_1 = require("estraverse");
const esrecurse = require("esrecurse");
const reference_1 = require("./reference");
const variable_1 = require("./variable");
const patternVisitor_1 = require("./patternVisitor");
const definition_1 = require("./definition");
const assert = require("assert");
const importManager_1 = require("./importManager");
const exportManager_1 = require("./exportManager");
function traverseIdentifierInPattern(options, rootPattern, referencer, callback) {
    const visitor = new patternVisitor_1.PatternVisitor(options, rootPattern, callback);
    visitor.visit(rootPattern);
    if (referencer !== null && referencer !== undefined) {
        visitor.rightHandNodes.forEach(referencer.visit, referencer);
    }
}
class Importer extends esrecurse.Visitor {
    constructor(declaration, referencer) {
        super(undefined, referencer.options);
        this.declaration = declaration;
        this.referencer = referencer;
    }
    get moduleScope() {
        return this.referencer.currentScope;
    }
    visitImport(id, specifier) {
        this.referencer.visitPattern(id, undefined, pattern => {
            this.referencer.currentScope.__define(pattern, new definition_1.Definition(variable_1.VariableType.ImportBinding, pattern, specifier, this.declaration));
        });
    }
    ImportNamespaceSpecifier(node) {
        const local = node.local;
        this.visitImport(local, node);
        const importId = new importManager_1.ImportIdentifierInfo(local.name, local.name, this.declaration.source.value, importManager_1.ImportType.Namespace);
        this.moduleScope.importManager.addImportId(importId);
    }
    ImportDefaultSpecifier(node) {
        const local = node.local;
        this.visitImport(local, node);
        const importId = new importManager_1.ImportIdentifierInfo(local.name, "default", this.declaration.source.value, importManager_1.ImportType.Default);
        this.moduleScope.importManager.addImportId(importId);
    }
    ImportSpecifier(node) {
        const local = node.local;
        this.visitImport(local, node);
        const importId = new importManager_1.ImportIdentifierInfo(local.name, node.imported.name, this.declaration.source.value, importManager_1.ImportType.Identifier);
        this.moduleScope.importManager.addImportId(importId);
    }
}
class Referencer extends esrecurse.Visitor {
    constructor(options, scopeManager) {
        super(undefined, options);
        this.options = options;
        this.scopeManager = scopeManager;
        this.parent = null;
        this.isInnerMethodDefinition = false;
        this.exportingSource = null;
        this.isExportingFromLocal = false;
    }
    get currentScope() {
        return this.scopeManager.__currentScope;
    }
    close(node) {
        while (this.currentScope &&
            node === this.currentScope.block) {
            this.scopeManager.__currentScope = this.currentScope.__close(this.scopeManager);
        }
    }
    pushInnerMethodDefinition(isInnerMethodDefinition) {
        const previous = this.isInnerMethodDefinition;
        this.isInnerMethodDefinition = isInnerMethodDefinition;
        return previous;
    }
    popInnerMethodDefinition(isInnerMethodDefinition) {
        this.isInnerMethodDefinition = isInnerMethodDefinition;
    }
    materializeTDZScope(node, iterationNode) {
        this.scopeManager.__nestTDZScope(node);
        this.visitVariableDeclaration(this.currentScope, variable_1.VariableType.TDZ, iterationNode.left, 0, true);
    }
    materializeIterationScope(node) {
        const letOrConstDecl = node.left;
        this.scopeManager.__nestForScope(node);
        this.visitVariableDeclaration(this.currentScope, variable_1.VariableType.Variable, letOrConstDecl, 0);
        this.visitPattern(letOrConstDecl.declarations[0].id, undefined, pattern => {
            this.currentScope.__referencing(pattern, reference_1.Reference.WRITE, node.right, undefined, true, true);
        });
    }
    referencingDefaultValue(pattern, assignments, maybeImplicitGlobal, init) {
        const scope = this.currentScope;
        assignments.forEach(assignment => {
            scope.__referencing(pattern, reference_1.Reference.WRITE, assignment.right, maybeImplicitGlobal, pattern !== assignment.left, init);
        });
    }
    visitPattern(node, options = { processRightHandNodes: false }, callback) {
        traverseIdentifierInPattern(this.options, node, options.processRightHandNodes ? this : null, callback);
    }
    visitFunction(node) {
        let i, iz;
        if (node.type === estraverse_1.Syntax.FunctionDeclaration) {
            this.currentScope.__define(node.id, new definition_1.Definition(variable_1.VariableType.FunctionName, node.id, node));
        }
        if (node.type === estraverse_1.Syntax.FunctionExpression && node.id) {
            this.scopeManager.__nestFunctionExpressionNameScope(node);
        }
        this.scopeManager.__nestFunctionScope(node, this.isInnerMethodDefinition);
        const that = this;
        function visitPatternCallback(pattern, info) {
            that.currentScope.__define(pattern, new definition_1.ParameterDefinition(pattern, node, i, info.rest));
            that.referencingDefaultValue(pattern, info.assignments, undefined, true);
        }
        for (i = 0, iz = node.params.length; i < iz; ++i) {
            this.visitPattern(node.params[i], { processRightHandNodes: true }, visitPatternCallback);
        }
        if (node.body) {
            if (node.body.type === estraverse_1.Syntax.BlockStatement) {
                this.visitChildren(node.body);
            }
            else {
                this.visit(node.body);
            }
        }
        this.close(node);
    }
    visitClass(node) {
        if (node.type === estraverse_1.Syntax.ClassDeclaration) {
            this.currentScope.__define(node.id, new definition_1.Definition(variable_1.VariableType.ClassName, node.id, node));
        }
        this.visit(node.superClass);
        this.scopeManager.__nestClassScope(node);
        if (node.id) {
            this.currentScope.__define(node.id, new definition_1.Definition(variable_1.VariableType.ClassName, node.id, node));
        }
        this.visit(node.body);
        this.close(node);
    }
    visitProperty(node) {
        let previous;
        if (node.computed) {
            this.visit(node.key);
        }
        const isMethodDefinition = node.type === estraverse_1.Syntax.MethodDefinition;
        if (isMethodDefinition) {
            previous = this.pushInnerMethodDefinition(true);
        }
        this.visit(node.value);
        if (isMethodDefinition) {
            this.popInnerMethodDefinition(previous);
        }
    }
    visitForIn(node) {
        if (node.left.type === estraverse_1.Syntax.VariableDeclaration &&
            node.left.kind !== "var") {
            this.materializeTDZScope(node.right, node);
            this.visit(node.right);
            this.close(node.right);
            this.materializeIterationScope(node);
            this.visit(node.body);
            this.close(node);
        }
        else {
            if (node.left.type === estraverse_1.Syntax.VariableDeclaration) {
                this.visit(node.left);
                this.visitPattern(node.left.declarations[0].id, undefined, pattern => {
                    this.currentScope.__referencing(pattern, reference_1.Reference.WRITE, node.right, undefined, true, true);
                });
            }
            else {
                this.visitPattern(node.left, { processRightHandNodes: true }, (pattern, info) => {
                    let maybeImplicitGlobal;
                    if (!this.currentScope.isStrict) {
                        maybeImplicitGlobal = {
                            pattern,
                            node,
                        };
                    }
                    this.referencingDefaultValue(pattern, info.assignments, maybeImplicitGlobal, false);
                    this.currentScope.__referencing(pattern, reference_1.Reference.WRITE, node.right, maybeImplicitGlobal, true, false);
                });
            }
            this.visit(node.right);
            this.visit(node.body);
        }
    }
    visitVariableDeclaration(variableTargetScope, type, node, index, fromTDZ) {
        const decl = node.declarations[index];
        const init = decl.init;
        this.visitPattern(decl.id, { processRightHandNodes: !fromTDZ }, (pattern, info) => {
            variableTargetScope.__define(pattern, new definition_1.Definition(type, pattern, decl, node, index, node.kind));
            if (!fromTDZ) {
                this.referencingDefaultValue(pattern, info.assignments, undefined, true);
            }
            if (init) {
                this.currentScope.__referencing(pattern, reference_1.Reference.WRITE, init, undefined, !info.topLevel, true);
            }
        });
    }
    AssignmentExpression(node) {
        if (patternVisitor_1.PatternVisitor.isPattern(node.left)) {
            if (node.operator === "=") {
                this.visitPattern(node.left, { processRightHandNodes: true }, (pattern, info) => {
                    let maybeImplicitGlobal;
                    if (!this.currentScope.isStrict) {
                        maybeImplicitGlobal = {
                            pattern,
                            node,
                        };
                    }
                    this.referencingDefaultValue(pattern, info.assignments, maybeImplicitGlobal, false);
                    this.currentScope.__referencing(pattern, reference_1.Reference.WRITE, node.right, maybeImplicitGlobal, !info.topLevel, false);
                });
            }
            else {
                this.currentScope.__referencing(node.left, reference_1.Reference.RW, node.right);
            }
        }
        else {
            this.visit(node.left);
        }
        this.visit(node.right);
    }
    CatchClause(node) {
        this.scopeManager.__nestCatchScope(node);
        this.visitPattern(node.param, { processRightHandNodes: true }, (pattern, info) => {
            this.currentScope.__define(pattern, new definition_1.Definition(variable_1.VariableType.CatchClause, node.param, node));
            this.referencingDefaultValue(pattern, info.assignments, undefined, true);
        });
        this.visit(node.body);
        this.close(node);
    }
    Program(node) {
        this.scopeManager.__nestGlobalScope(node);
        if (this.scopeManager.__isNodejsScope()) {
            this.currentScope.isStrict = false;
            this.scopeManager.__nestFunctionScope(node, false);
        }
        if (this.scopeManager.__isES6() &&
            this.scopeManager.isModule()) {
            this.scopeManager.__nestModuleScope(node);
        }
        if (this.scopeManager.isStrictModeSupported() &&
            this.scopeManager.isImpliedStrict()) {
            this.currentScope.isStrict = true;
        }
        this.visitChildren(node);
        this.close(node);
    }
    Identifier(node) {
        this.currentScope.__referencing(node, undefined, undefined, undefined, undefined, undefined, this.isExportingFromLocal);
    }
    UpdateExpression(node) {
        if (patternVisitor_1.PatternVisitor.isPattern(node.argument)) {
            this.currentScope.__referencing(node.argument, reference_1.Reference.RW);
        }
        else {
            this.visitChildren(node);
        }
    }
    MemberExpression(node) {
        this.visit(node.object);
        if (node.computed) {
            this.visit(node.property);
        }
    }
    Property(node) {
        this.visitProperty(node);
    }
    MethodDefinition(node) {
        this.visitProperty(node);
    }
    BreakStatement() { }
    ContinueStatement() { }
    LabeledStatement(node) {
        this.visit(node.body);
    }
    ForStatement(node) {
        if (node.init &&
            node.init.type === estraverse_1.Syntax.VariableDeclaration &&
            node.init.kind !== "var") {
            this.scopeManager.__nestForScope(node);
        }
        this.visitChildren(node);
        this.close(node);
    }
    ClassExpression(node) {
        this.visitClass(node);
    }
    ClassDeclaration(node) {
        this.visitClass(node);
    }
    CallExpression(node) {
        if (!this.scopeManager.__ignoreEval() &&
            node.callee.type === estraverse_1.Syntax.Identifier &&
            node.callee.name === "eval") {
            this.currentScope.variableScope.__detectEval();
        }
        this.visitChildren(node);
    }
    BlockStatement(node) {
        if (this.scopeManager.__isES6()) {
            this.scopeManager.__nestBlockScope(node);
        }
        this.visitChildren(node);
        this.close(node);
    }
    ThisExpression() {
        this.currentScope.variableScope.__detectThis();
    }
    WithStatement(node) {
        this.visit(node.object);
        this.scopeManager.__nestWithScope(node);
        this.visit(node.body);
        this.close(node);
    }
    VariableDeclaration(node) {
        const variableTargetScope = node.kind === "var"
            ? this.currentScope.variableScope
            : this.currentScope;
        for (let i = 0, iz = node.declarations.length; i < iz; ++i) {
            const decl = node.declarations[i];
            this.visitVariableDeclaration(variableTargetScope, variable_1.VariableType.Variable, node, i);
            if (decl.init) {
                this.visit(decl.init);
            }
        }
    }
    SwitchStatement(node) {
        this.visit(node.discriminant);
        if (this.scopeManager.__isES6()) {
            this.scopeManager.__nestSwitchScope(node);
        }
        for (let i = 0, iz = node.cases.length; i < iz; ++i) {
            this.visit(node.cases[i]);
        }
        this.close(node);
    }
    FunctionDeclaration(node) {
        this.visitFunction(node);
    }
    FunctionExpression(node) {
        this.visitFunction(node);
    }
    ForOfStatement(node) {
        this.visitForIn(node);
    }
    ForInStatement(node) {
        this.visitForIn(node);
    }
    ArrowFunctionExpression(node) {
        this.visitFunction(node);
    }
    ImportDeclaration(node) {
        assert(this.scopeManager.__isES6() &&
            this.scopeManager.isModule(), "ImportDeclaration should appear when the mode is ES6 and in the module context.");
        const importer = new Importer(node, this);
        importer.visit(node);
    }
    visitExportDeclaration(node) {
        if (node.source) {
            this.exportingSource = node.source.value;
        }
        else {
            this.exportingSource = null;
        }
        if (node.declaration) {
            this.visit(node.declaration);
            return;
        }
        this.visitChildren(node);
    }
    ExportNamedDeclaration(node) {
        const moduleScope = this.currentScope;
        const previous = moduleScope.isExportingNamedDeclaration;
        moduleScope.isExportingNamedDeclaration = true;
        this.visitExportDeclaration(node);
        moduleScope.isExportingNamedDeclaration = previous;
    }
    ExportDefaultDeclaration(node) {
        const currentScope = this.currentScope;
        if (currentScope.type !== "module") {
            throw new Error("use export in a non module scope");
        }
        let localExportVar;
        if (node.declaration.type === "Identifier") {
            localExportVar = {
                type: exportManager_1.ExportVariableType.Local,
                exportName: "default",
                localName: node.declaration.name,
                node: node.declaration,
            };
            this.isExportingFromLocal = true;
            this.visit(node.declaration);
            this.isExportingFromLocal = false;
        }
        else {
            localExportVar = {
                type: exportManager_1.ExportVariableType.Local,
                exportName: "default",
                localName: null,
                node: node.declaration,
            };
            this.visit(node.declaration);
        }
        currentScope.exportManager.addLocalExportVariable(localExportVar);
    }
    ExportAllDeclaration(node) {
        const currentScope = this.currentScope;
        currentScope.exportManager.addExternalVariable({
            type: exportManager_1.ExportVariableType.External,
            moduleName: node.source.value,
            moduleType: exportManager_1.ExternalType.All,
        });
    }
    ExportSpecifier(node) {
        const local = node.local;
        const currentScope = this.currentScope;
        if (this.exportingSource) {
            currentScope.exportManager.addExternalVariable({
                type: exportManager_1.ExportVariableType.External,
                moduleName: this.exportingSource,
                moduleType: exportManager_1.ExternalType.Identifier,
                names: {
                    exportName: node.exported.name,
                    sourceName: node.local.name,
                },
            });
        }
        else {
            this.isExportingFromLocal = true;
            this.visit(local);
            currentScope.exportManager.addLocalExportVariable({
                type: exportManager_1.ExportVariableType.Local,
                exportName: node.exported.name,
                localName: node.local.name,
                node: node.local,
            });
            this.isExportingFromLocal = false;
        }
    }
    MetaProperty() {
    }
}
exports.Referencer = Referencer;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicmVmZXJlbmNlci5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uL3NyYy9yZWZlcmVuY2VyLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7O0FBQUEsMkNBQW9DO0FBQ3BDLHVDQUF1QztBQUN2QywyQ0FBd0Q7QUFDeEQseUNBQTBDO0FBQzFDLHFEQUkwQjtBQUMxQiw2Q0FBK0Q7QUFDL0QsaUNBQWlDO0FBSWpDLG1EQUd5QjtBQUN6QixtREFFeUI7QUFFekIscUNBQ0UsT0FBZ0MsRUFDaEMsV0FBd0IsRUFDeEIsVUFBNkIsRUFDN0IsUUFBZ0M7SUFHaEMsTUFBTSxPQUFPLEdBQUcsSUFBSSwrQkFBYyxDQUNoQyxPQUFPLEVBQ1AsV0FBVyxFQUNYLFFBQVEsQ0FDVCxDQUFDO0lBRUYsT0FBTyxDQUFDLEtBQUssQ0FBQyxXQUFXLENBQUMsQ0FBQztJQUczQixJQUFJLFVBQVUsS0FBSyxJQUFJLElBQUksVUFBVSxLQUFLLFNBQVMsRUFBRTtRQUNuRCxPQUFPLENBQUMsY0FBYyxDQUFDLE9BQU8sQ0FDNUIsVUFBVSxDQUFDLEtBQUssRUFDaEIsVUFBVSxDQUNYLENBQUM7S0FDSDtBQUNILENBQUM7QUFhRCxjQUFlLFNBQVEsU0FBUyxDQUFDLE9BQU87SUFDdEMsWUFDa0IsV0FBcUMsRUFDckMsVUFBc0I7UUFFdEMsS0FBSyxDQUFDLFNBQVMsRUFBRSxVQUFVLENBQUMsT0FBTyxDQUFDLENBQUM7UUFIckIsZ0JBQVcsR0FBWCxXQUFXLENBQTBCO1FBQ3JDLGVBQVUsR0FBVixVQUFVLENBQVk7SUFHeEMsQ0FBQztJQUVELElBQUksV0FBVztRQUNiLE9BQU8sSUFBSSxDQUFDLFVBQVUsQ0FBQyxZQUEyQixDQUFDO0lBQ3JELENBQUM7SUFFTSxXQUFXLENBQ2hCLEVBQXFCLEVBQ3JCLFNBQThCO1FBRTlCLElBQUksQ0FBQyxVQUFVLENBQUMsWUFBWSxDQUFDLEVBQUUsRUFBRSxTQUFTLEVBQUUsT0FBTyxDQUFDLEVBQUU7WUFDcEQsSUFBSSxDQUFDLFVBQVUsQ0FBQyxZQUFhLENBQUMsUUFBUSxDQUNwQyxPQUFPLEVBQ1AsSUFBSSx1QkFBVSxDQUNaLHVCQUFZLENBQUMsYUFBYSxFQUMxQixPQUFPLEVBQ1AsU0FBUyxFQUNULElBQUksQ0FBQyxXQUFXLENBQ2pCLENBQ0YsQ0FBQztRQUNKLENBQUMsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVNLHdCQUF3QixDQUM3QixJQUFxQztRQUVyQyxNQUFNLEtBQUssR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDO1FBRXpCLElBQUksQ0FBQyxXQUFXLENBQUMsS0FBSyxFQUFFLElBQUksQ0FBQyxDQUFDO1FBQzlCLE1BQU0sUUFBUSxHQUFHLElBQUksb0NBQW9CLENBQ3ZDLEtBQUssQ0FBQyxJQUFJLEVBQ1YsS0FBSyxDQUFDLElBQUksRUFDVixJQUFJLENBQUMsV0FBVyxDQUFDLE1BQU0sQ0FBQyxLQUFlLEVBQ3ZDLDBCQUFVLENBQUMsU0FBUyxDQUNyQixDQUFDO1FBQ0YsSUFBSSxDQUFDLFdBQVcsQ0FBQyxhQUFhLENBQUMsV0FBVyxDQUFDLFFBQVEsQ0FBQyxDQUFDO0lBQ3ZELENBQUM7SUFFTSxzQkFBc0IsQ0FDM0IsSUFBbUM7UUFFbkMsTUFBTSxLQUFLLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQztRQUV6QixJQUFJLENBQUMsV0FBVyxDQUFDLEtBQUssRUFBRSxJQUFJLENBQUMsQ0FBQztRQUM5QixNQUFNLFFBQVEsR0FBRyxJQUFJLG9DQUFvQixDQUN2QyxLQUFLLENBQUMsSUFBSSxFQUNWLFNBQVMsRUFDVCxJQUFJLENBQUMsV0FBVyxDQUFDLE1BQU0sQ0FBQyxLQUFlLEVBQ3ZDLDBCQUFVLENBQUMsT0FBTyxDQUNuQixDQUFDO1FBQ0YsSUFBSSxDQUFDLFdBQVcsQ0FBQyxhQUFhLENBQUMsV0FBVyxDQUFDLFFBQVEsQ0FBQyxDQUFDO0lBQ3ZELENBQUM7SUFFTSxlQUFlLENBQUMsSUFBNEI7UUFDakQsTUFBTSxLQUFLLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQztRQUV6QixJQUFJLENBQUMsV0FBVyxDQUFDLEtBQUssRUFBRSxJQUFJLENBQUMsQ0FBQztRQUM5QixNQUFNLFFBQVEsR0FBRyxJQUFJLG9DQUFvQixDQUN2QyxLQUFLLENBQUMsSUFBSSxFQUNWLElBQUksQ0FBQyxRQUFRLENBQUMsSUFBSSxFQUNsQixJQUFJLENBQUMsV0FBVyxDQUFDLE1BQU0sQ0FBQyxLQUFlLEVBQ3ZDLDBCQUFVLENBQUMsVUFBVSxDQUN0QixDQUFDO1FBQ0YsSUFBSSxDQUFDLFdBQVcsQ0FBQyxhQUFhLENBQUMsV0FBVyxDQUFDLFFBQVEsQ0FBQyxDQUFDO0lBQ3ZELENBQUM7Q0FDRjtBQUdELGdCQUF3QixTQUFRLFNBQVMsQ0FBQyxPQUFPO0lBTS9DLFlBQ2tCLE9BQWdDLEVBQ2hDLFlBQTBCO1FBRTFDLEtBQUssQ0FBQyxTQUFTLEVBQUUsT0FBTyxDQUFDLENBQUM7UUFIVixZQUFPLEdBQVAsT0FBTyxDQUF5QjtRQUNoQyxpQkFBWSxHQUFaLFlBQVksQ0FBYztRQVByQyxXQUFNLEdBQXNCLElBQUksQ0FBQztRQUNqQyw0QkFBdUIsR0FBWSxLQUFLLENBQUM7UUFDekMsb0JBQWUsR0FBa0IsSUFBSSxDQUFDO1FBQ3RDLHlCQUFvQixHQUFZLEtBQUssQ0FBQztJQU83QyxDQUFDO0lBRUQsSUFBSSxZQUFZO1FBQ2QsT0FBTyxJQUFJLENBQUMsWUFBWSxDQUFDLGNBQWMsQ0FBQztJQUMxQyxDQUFDO0lBRU0sS0FBSyxDQUFDLElBQWlCO1FBQzVCLE9BQ0UsSUFBSSxDQUFDLFlBQVk7WUFDakIsSUFBSSxLQUFLLElBQUksQ0FBQyxZQUFZLENBQUMsS0FBSyxFQUNoQztZQUNBLElBQUksQ0FBQyxZQUFZLENBQUMsY0FBYyxHQUFHLElBQUksQ0FBQyxZQUFZLENBQUMsT0FBTyxDQUMxRCxJQUFJLENBQUMsWUFBWSxDQUNsQixDQUFDO1NBQ0g7SUFDSCxDQUFDO0lBRU0seUJBQXlCLENBQzlCLHVCQUFnQztRQUVoQyxNQUFNLFFBQVEsR0FBRyxJQUFJLENBQUMsdUJBQXVCLENBQUM7UUFFOUMsSUFBSSxDQUFDLHVCQUF1QixHQUFHLHVCQUF1QixDQUFDO1FBQ3ZELE9BQU8sUUFBUSxDQUFDO0lBQ2xCLENBQUM7SUFFTSx3QkFBd0IsQ0FDN0IsdUJBQWdDO1FBRWhDLElBQUksQ0FBQyx1QkFBdUIsR0FBRyx1QkFBdUIsQ0FBQztJQUN6RCxDQUFDO0lBRU0sbUJBQW1CLENBQ3hCLElBQWlCLEVBQ2pCLGFBRXlCO1FBTXpCLElBQUksQ0FBQyxZQUFZLENBQUMsY0FBYyxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQ3ZDLElBQUksQ0FBQyx3QkFBd0IsQ0FDM0IsSUFBSSxDQUFDLFlBQWEsRUFDbEIsdUJBQVksQ0FBQyxHQUFHLEVBQ2hCLGFBQWEsQ0FBQyxJQUFrQyxFQUNoRCxDQUFDLEVBQ0QsSUFBSSxDQUNMLENBQUM7SUFDSixDQUFDO0lBRU0seUJBQXlCLENBQzlCLElBQW1EO1FBR25ELE1BQU0sY0FBYyxHQUFHLElBQUksQ0FBQyxJQUFrQyxDQUFDO1FBRS9ELElBQUksQ0FBQyxZQUFZLENBQUMsY0FBYyxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQ3ZDLElBQUksQ0FBQyx3QkFBd0IsQ0FDM0IsSUFBSSxDQUFDLFlBQWEsRUFDbEIsdUJBQVksQ0FBQyxRQUFRLEVBQ3JCLGNBQWMsRUFDZCxDQUFDLENBQ0YsQ0FBQztRQUNGLElBQUksQ0FBQyxZQUFZLENBQ2YsY0FBYyxDQUFDLFlBQVksQ0FBQyxDQUFDLENBQUMsQ0FBQyxFQUFFLEVBQ2pDLFNBQVMsRUFDVCxPQUFPLENBQUMsRUFBRTtZQUNSLElBQUksQ0FBQyxZQUFhLENBQUMsYUFBYSxDQUM5QixPQUFPLEVBQ1AscUJBQVMsQ0FBQyxLQUFLLEVBQ2YsSUFBSSxDQUFDLEtBQUssRUFDVixTQUFTLEVBQ1QsSUFBSSxFQUNKLElBQUksQ0FDTCxDQUFDO1FBQ0osQ0FBQyxDQUNGLENBQUM7SUFDSixDQUFDO0lBRU0sdUJBQXVCLENBQzVCLE9BQXVCLEVBQ3ZCLFdBQTZCLEVBQzdCLG1CQUFvQyxFQUNwQyxJQUFjO1FBRWQsTUFBTSxLQUFLLEdBQUcsSUFBSSxDQUFDLFlBQWEsQ0FBQztRQUVqQyxXQUFXLENBQUMsT0FBTyxDQUFDLFVBQVUsQ0FBQyxFQUFFO1lBQy9CLEtBQUssQ0FBQyxhQUFhLENBQ2pCLE9BQU8sRUFDUCxxQkFBUyxDQUFDLEtBQUssRUFDZixVQUFVLENBQUMsS0FBSyxFQUNoQixtQkFBbUIsRUFDbkIsT0FBTyxLQUFLLFVBQVUsQ0FBQyxJQUFJLEVBQzNCLElBQUksQ0FDTCxDQUFDO1FBQ0osQ0FBQyxDQUFDLENBQUM7SUFDTCxDQUFDO0lBRU0sWUFBWSxDQUNqQixJQUFvQixFQUNwQixPQUFPLEdBQUcsRUFBRSxxQkFBcUIsRUFBRSxLQUFLLEVBQUUsRUFDMUMsUUFBZ0M7UUFFaEMsMkJBQTJCLENBQ3pCLElBQUksQ0FBQyxPQUFPLEVBQ1osSUFBSSxFQUNKLE9BQU8sQ0FBQyxxQkFBcUIsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxJQUFJLEVBQzNDLFFBQVEsQ0FDVCxDQUFDO0lBQ0osQ0FBQztJQUVNLGFBQWEsQ0FBQyxJQUFxQjtRQUN4QyxJQUFJLENBQVMsRUFBRSxFQUFVLENBQUM7UUFRMUIsSUFBSSxJQUFJLENBQUMsSUFBSSxLQUFLLG1CQUFNLENBQUMsbUJBQW1CLEVBQUU7WUFFNUMsSUFBSSxDQUFDLFlBQWEsQ0FBQyxRQUFRLENBQ3pCLElBQUksQ0FBQyxFQUFHLEVBQ1IsSUFBSSx1QkFBVSxDQUNaLHVCQUFZLENBQUMsWUFBWSxFQUN6QixJQUFJLENBQUMsRUFBRyxFQUNSLElBQUksQ0FDTCxDQUNGLENBQUM7U0FDSDtRQUlELElBQUksSUFBSSxDQUFDLElBQUksS0FBSyxtQkFBTSxDQUFDLGtCQUFrQixJQUFJLElBQUksQ0FBQyxFQUFFLEVBQUU7WUFDdEQsSUFBSSxDQUFDLFlBQVksQ0FBQyxpQ0FBaUMsQ0FBQyxJQUFJLENBQUMsQ0FBQztTQUMzRDtRQUdELElBQUksQ0FBQyxZQUFZLENBQUMsbUJBQW1CLENBQ25DLElBQUksRUFDSixJQUFJLENBQUMsdUJBQXVCLENBQzdCLENBQUM7UUFFRixNQUFNLElBQUksR0FBRyxJQUFJLENBQUM7UUFRbEIsOEJBQ0UsT0FBMEIsRUFDMUIsSUFBUztZQUVULElBQUksQ0FBQyxZQUFhLENBQUMsUUFBUSxDQUN6QixPQUFPLEVBQ1AsSUFBSSxnQ0FBbUIsQ0FBQyxPQUFPLEVBQUUsSUFBSSxFQUFFLENBQUMsRUFBRSxJQUFJLENBQUMsSUFBSSxDQUFDLENBQ3JELENBQUM7WUFFRixJQUFJLENBQUMsdUJBQXVCLENBQzFCLE9BQU8sRUFDUCxJQUFJLENBQUMsV0FBVyxFQUNoQixTQUFTLEVBQ1QsSUFBSSxDQUNMLENBQUM7UUFDSixDQUFDO1FBR0QsS0FBSyxDQUFDLEdBQUcsQ0FBQyxFQUFFLEVBQUUsR0FBRyxJQUFJLENBQUMsTUFBTSxDQUFDLE1BQU0sRUFBRSxDQUFDLEdBQUcsRUFBRSxFQUFFLEVBQUUsQ0FBQyxFQUFFO1lBQ2hELElBQUksQ0FBQyxZQUFZLENBQ2YsSUFBSSxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsRUFDZCxFQUFFLHFCQUFxQixFQUFFLElBQUksRUFBRSxFQUMvQixvQkFBb0IsQ0FDckIsQ0FBQztTQUNIO1FBSUQsSUFBSSxJQUFJLENBQUMsSUFBSSxFQUFFO1lBRWIsSUFBSSxJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksS0FBSyxtQkFBTSxDQUFDLGNBQWMsRUFBRTtnQkFDNUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUM7YUFDL0I7aUJBQU07Z0JBQ0wsSUFBSSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUM7YUFDdkI7U0FDRjtRQUVELElBQUksQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLENBQUM7SUFDbkIsQ0FBQztJQUVNLFVBQVUsQ0FBQyxJQUFrQjtRQUNsQyxJQUFJLElBQUksQ0FBQyxJQUFJLEtBQUssbUJBQU0sQ0FBQyxnQkFBZ0IsRUFBRTtZQUN6QyxJQUFJLENBQUMsWUFBYSxDQUFDLFFBQVEsQ0FDekIsSUFBSSxDQUFDLEVBQUcsRUFDUixJQUFJLHVCQUFVLENBQUMsdUJBQVksQ0FBQyxTQUFTLEVBQUUsSUFBSSxDQUFDLEVBQUUsRUFBRSxJQUFJLENBQUMsQ0FDdEQsQ0FBQztTQUNIO1FBR0QsSUFBSSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsVUFBVyxDQUFDLENBQUM7UUFFN0IsSUFBSSxDQUFDLFlBQVksQ0FBQyxnQkFBZ0IsQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUV6QyxJQUFJLElBQUksQ0FBQyxFQUFFLEVBQUU7WUFDWCxJQUFJLENBQUMsWUFBYSxDQUFDLFFBQVEsQ0FDekIsSUFBSSxDQUFDLEVBQUUsRUFDUCxJQUFJLHVCQUFVLENBQUMsdUJBQVksQ0FBQyxTQUFTLEVBQUUsSUFBSSxDQUFDLEVBQUUsRUFBRSxJQUFJLENBQUMsQ0FDdEQsQ0FBQztTQUNIO1FBQ0QsSUFBSSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUM7UUFFdEIsSUFBSSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsQ0FBQztJQUNuQixDQUFDO0lBRU0sYUFBYSxDQUFDLElBQVM7UUFDNUIsSUFBSSxRQUFpQixDQUFDO1FBRXRCLElBQUksSUFBSSxDQUFDLFFBQVEsRUFBRTtZQUNqQixJQUFJLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQztTQUN0QjtRQUVELE1BQU0sa0JBQWtCLEdBQ3RCLElBQUksQ0FBQyxJQUFJLEtBQUssbUJBQU0sQ0FBQyxnQkFBZ0IsQ0FBQztRQUV4QyxJQUFJLGtCQUFrQixFQUFFO1lBQ3RCLFFBQVEsR0FBRyxJQUFJLENBQUMseUJBQXlCLENBQUMsSUFBSSxDQUFDLENBQUM7U0FDakQ7UUFDRCxJQUFJLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQztRQUN2QixJQUFJLGtCQUFrQixFQUFFO1lBQ3RCLElBQUksQ0FBQyx3QkFBd0IsQ0FBQyxRQUFTLENBQUMsQ0FBQztTQUMxQztJQUNILENBQUM7SUFFTSxVQUFVLENBQ2YsSUFBbUQ7UUFFbkQsSUFDRSxJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksS0FBSyxtQkFBTSxDQUFDLG1CQUFtQjtZQUM3QyxJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksS0FBSyxLQUFLLEVBQ3hCO1lBQ0EsSUFBSSxDQUFDLG1CQUFtQixDQUFDLElBQUksQ0FBQyxLQUFLLEVBQUUsSUFBSSxDQUFDLENBQUM7WUFDM0MsSUFBSSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLENBQUM7WUFDdkIsSUFBSSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLENBQUM7WUFFdkIsSUFBSSxDQUFDLHlCQUF5QixDQUFDLElBQUksQ0FBQyxDQUFDO1lBQ3JDLElBQUksQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDO1lBQ3RCLElBQUksQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLENBQUM7U0FDbEI7YUFBTTtZQUNMLElBQUksSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLEtBQUssbUJBQU0sQ0FBQyxtQkFBbUIsRUFBRTtnQkFDakQsSUFBSSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUM7Z0JBQ3RCLElBQUksQ0FBQyxZQUFZLENBQ2YsSUFBSSxDQUFDLElBQUksQ0FBQyxZQUFZLENBQUMsQ0FBQyxDQUFDLENBQUMsRUFBRSxFQUM1QixTQUFTLEVBQ1QsT0FBTyxDQUFDLEVBQUU7b0JBQ1IsSUFBSSxDQUFDLFlBQWEsQ0FBQyxhQUFhLENBQzlCLE9BQU8sRUFDUCxxQkFBUyxDQUFDLEtBQUssRUFDZixJQUFJLENBQUMsS0FBSyxFQUNWLFNBQVMsRUFDVCxJQUFJLEVBQ0osSUFBSSxDQUNMLENBQUM7Z0JBQ0osQ0FBQyxDQUNGLENBQUM7YUFDSDtpQkFBTTtnQkFDTCxJQUFJLENBQUMsWUFBWSxDQUNmLElBQUksQ0FBQyxJQUFzQixFQUMzQixFQUFFLHFCQUFxQixFQUFFLElBQUksRUFBRSxFQUMvQixDQUFDLE9BQTBCLEVBQUUsSUFBUyxFQUFFLEVBQUU7b0JBQ3hDLElBQUksbUJBQStDLENBQUM7b0JBRXBELElBQUksQ0FBQyxJQUFJLENBQUMsWUFBYSxDQUFDLFFBQVEsRUFBRTt3QkFDaEMsbUJBQW1CLEdBQUc7NEJBQ3BCLE9BQU87NEJBQ1AsSUFBSTt5QkFDTCxDQUFDO3FCQUNIO29CQUNELElBQUksQ0FBQyx1QkFBdUIsQ0FDMUIsT0FBTyxFQUNQLElBQUksQ0FBQyxXQUFXLEVBQ2hCLG1CQUFtQixFQUNuQixLQUFLLENBQ04sQ0FBQztvQkFDRixJQUFJLENBQUMsWUFBYSxDQUFDLGFBQWEsQ0FDOUIsT0FBTyxFQUNQLHFCQUFTLENBQUMsS0FBSyxFQUNmLElBQUksQ0FBQyxLQUFLLEVBQ1YsbUJBQW1CLEVBQ25CLElBQUksRUFDSixLQUFLLENBQ04sQ0FBQztnQkFDSixDQUFDLENBQ0YsQ0FBQzthQUNIO1lBQ0QsSUFBSSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLENBQUM7WUFDdkIsSUFBSSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUM7U0FDdkI7SUFDSCxDQUFDO0lBRU0sd0JBQXdCLENBQzdCLG1CQUEwQixFQUMxQixJQUFrQixFQUNsQixJQUFnQyxFQUNoQyxLQUFhLEVBQ2IsT0FBaUI7UUFHakIsTUFBTSxJQUFJLEdBQUcsSUFBSSxDQUFDLFlBQVksQ0FBQyxLQUFLLENBQUMsQ0FBQztRQUN0QyxNQUFNLElBQUksR0FBRyxJQUFJLENBQUMsSUFBSSxDQUFDO1FBRXZCLElBQUksQ0FBQyxZQUFZLENBQ2YsSUFBSSxDQUFDLEVBQUUsRUFDUCxFQUFFLHFCQUFxQixFQUFFLENBQUMsT0FBTyxFQUFFLEVBQ25DLENBQUMsT0FBTyxFQUFFLElBQUksRUFBRSxFQUFFO1lBQ2hCLG1CQUFtQixDQUFDLFFBQVEsQ0FDMUIsT0FBTyxFQUNQLElBQUksdUJBQVUsQ0FDWixJQUFJLEVBQ0osT0FBTyxFQUNQLElBQUksRUFDSixJQUFJLEVBQ0osS0FBSyxFQUNMLElBQUksQ0FBQyxJQUFJLENBQ1YsQ0FDRixDQUFDO1lBRUYsSUFBSSxDQUFDLE9BQU8sRUFBRTtnQkFDWixJQUFJLENBQUMsdUJBQXVCLENBQzFCLE9BQU8sRUFDUCxJQUFJLENBQUMsV0FBVyxFQUNoQixTQUFTLEVBQ1QsSUFBSSxDQUNMLENBQUM7YUFDSDtZQUNELElBQUksSUFBSSxFQUFFO2dCQUNSLElBQUksQ0FBQyxZQUFhLENBQUMsYUFBYSxDQUM5QixPQUFPLEVBQ1AscUJBQVMsQ0FBQyxLQUFLLEVBQ2YsSUFBSSxFQUNKLFNBQVMsRUFDVCxDQUFDLElBQUksQ0FBQyxRQUFRLEVBQ2QsSUFBSSxDQUNMLENBQUM7YUFDSDtRQUNILENBQUMsQ0FDRixDQUFDO0lBQ0osQ0FBQztJQUVNLG9CQUFvQixDQUN6QixJQUFpQztRQUVqQyxJQUFJLCtCQUFjLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsRUFBRTtZQUN2QyxJQUFJLElBQUksQ0FBQyxRQUFRLEtBQUssR0FBRyxFQUFFO2dCQUN6QixJQUFJLENBQUMsWUFBWSxDQUNmLElBQUksQ0FBQyxJQUFJLEVBQ1QsRUFBRSxxQkFBcUIsRUFBRSxJQUFJLEVBQUUsRUFDL0IsQ0FBQyxPQUFPLEVBQUUsSUFBSSxFQUFFLEVBQUU7b0JBQ2hCLElBQUksbUJBQW1CLENBQUM7b0JBRXhCLElBQUksQ0FBQyxJQUFJLENBQUMsWUFBYSxDQUFDLFFBQVEsRUFBRTt3QkFDaEMsbUJBQW1CLEdBQUc7NEJBQ3BCLE9BQU87NEJBQ1AsSUFBSTt5QkFDTCxDQUFDO3FCQUNIO29CQUNELElBQUksQ0FBQyx1QkFBdUIsQ0FDMUIsT0FBTyxFQUNQLElBQUksQ0FBQyxXQUFXLEVBQ2hCLG1CQUFtQixFQUNuQixLQUFLLENBQ04sQ0FBQztvQkFDRixJQUFJLENBQUMsWUFBYSxDQUFDLGFBQWEsQ0FDOUIsT0FBTyxFQUNQLHFCQUFTLENBQUMsS0FBSyxFQUNmLElBQUksQ0FBQyxLQUFLLEVBQ1YsbUJBQW1CLEVBQ25CLENBQUMsSUFBSSxDQUFDLFFBQVEsRUFDZCxLQUFLLENBQ04sQ0FBQztnQkFDSixDQUFDLENBQ0YsQ0FBQzthQUNIO2lCQUFNO2dCQUNMLElBQUksQ0FBQyxZQUFhLENBQUMsYUFBYSxDQUM5QixJQUFJLENBQUMsSUFBSSxFQUNULHFCQUFTLENBQUMsRUFBRSxFQUNaLElBQUksQ0FBQyxLQUFLLENBQ1gsQ0FBQzthQUNIO1NBQ0Y7YUFBTTtZQUNMLElBQUksQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDO1NBQ3ZCO1FBQ0QsSUFBSSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLENBQUM7SUFDekIsQ0FBQztJQUVNLFdBQVcsQ0FBQyxJQUF3QjtRQUN6QyxJQUFJLENBQUMsWUFBWSxDQUFDLGdCQUFnQixDQUFDLElBQUksQ0FBQyxDQUFDO1FBRXpDLElBQUksQ0FBQyxZQUFZLENBQ2YsSUFBSSxDQUFDLEtBQUssRUFDVixFQUFFLHFCQUFxQixFQUFFLElBQUksRUFBRSxFQUMvQixDQUFDLE9BQU8sRUFBRSxJQUFJLEVBQUUsRUFBRTtZQUNoQixJQUFJLENBQUMsWUFBYSxDQUFDLFFBQVEsQ0FDekIsT0FBTyxFQUNQLElBQUksdUJBQVUsQ0FDWix1QkFBWSxDQUFDLFdBQVcsRUFDeEIsSUFBSSxDQUFDLEtBQUssRUFDVixJQUFJLENBQ0wsQ0FDRixDQUFDO1lBQ0YsSUFBSSxDQUFDLHVCQUF1QixDQUMxQixPQUFPLEVBQ1AsSUFBSSxDQUFDLFdBQVcsRUFDaEIsU0FBUyxFQUNULElBQUksQ0FDTCxDQUFDO1FBQ0osQ0FBQyxDQUNGLENBQUM7UUFDRixJQUFJLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUV0QixJQUFJLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxDQUFDO0lBQ25CLENBQUM7SUFFTSxPQUFPLENBQUMsSUFBb0I7UUFDakMsSUFBSSxDQUFDLFlBQVksQ0FBQyxpQkFBaUIsQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUUxQyxJQUFJLElBQUksQ0FBQyxZQUFZLENBQUMsZUFBZSxFQUFFLEVBQUU7WUFFdkMsSUFBSSxDQUFDLFlBQWEsQ0FBQyxRQUFRLEdBQUcsS0FBSyxDQUFDO1lBQ3BDLElBQUksQ0FBQyxZQUFZLENBQUMsbUJBQW1CLENBQUMsSUFBSSxFQUFFLEtBQUssQ0FBQyxDQUFDO1NBQ3BEO1FBRUQsSUFDRSxJQUFJLENBQUMsWUFBWSxDQUFDLE9BQU8sRUFBRTtZQUMzQixJQUFJLENBQUMsWUFBWSxDQUFDLFFBQVEsRUFBRSxFQUM1QjtZQUNBLElBQUksQ0FBQyxZQUFZLENBQUMsaUJBQWlCLENBQUMsSUFBSSxDQUFDLENBQUM7U0FDM0M7UUFFRCxJQUNFLElBQUksQ0FBQyxZQUFZLENBQUMscUJBQXFCLEVBQUU7WUFDekMsSUFBSSxDQUFDLFlBQVksQ0FBQyxlQUFlLEVBQUUsRUFDbkM7WUFDQSxJQUFJLENBQUMsWUFBYSxDQUFDLFFBQVEsR0FBRyxJQUFJLENBQUM7U0FDcEM7UUFFRCxJQUFJLENBQUMsYUFBYSxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQ3pCLElBQUksQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLENBQUM7SUFDbkIsQ0FBQztJQUVNLFVBQVUsQ0FBQyxJQUF1QjtRQUN2QyxJQUFJLENBQUMsWUFBYSxDQUFDLGFBQWEsQ0FDOUIsSUFBSSxFQUNKLFNBQVMsRUFDVCxTQUFTLEVBQ1QsU0FBUyxFQUNULFNBQVMsRUFDVCxTQUFTLEVBQ1QsSUFBSSxDQUFDLG9CQUFvQixDQUMxQixDQUFDO0lBQ0osQ0FBQztJQUVNLGdCQUFnQixDQUFDLElBQTZCO1FBQ25ELElBQUksK0JBQWMsQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxFQUFFO1lBQzNDLElBQUksQ0FBQyxZQUFhLENBQUMsYUFBYSxDQUM5QixJQUFJLENBQUMsUUFBUSxFQUNiLHFCQUFTLENBQUMsRUFBRSxDQUNiLENBQUM7U0FDSDthQUFNO1lBQ0wsSUFBSSxDQUFDLGFBQWEsQ0FBQyxJQUFJLENBQUMsQ0FBQztTQUMxQjtJQUNILENBQUM7SUFFTSxnQkFBZ0IsQ0FBQyxJQUE2QjtRQUNuRCxJQUFJLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsQ0FBQztRQUN4QixJQUFJLElBQUksQ0FBQyxRQUFRLEVBQUU7WUFDakIsSUFBSSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLENBQUM7U0FDM0I7SUFDSCxDQUFDO0lBRU0sUUFBUSxDQUFDLElBQXFCO1FBQ25DLElBQUksQ0FBQyxhQUFhLENBQUMsSUFBSSxDQUFDLENBQUM7SUFDM0IsQ0FBQztJQUVNLGdCQUFnQixDQUFDLElBQTZCO1FBQ25ELElBQUksQ0FBQyxhQUFhLENBQUMsSUFBSSxDQUFDLENBQUM7SUFDM0IsQ0FBQztJQUVNLGNBQWMsS0FBSSxDQUFDO0lBRW5CLGlCQUFpQixLQUFJLENBQUM7SUFFdEIsZ0JBQWdCLENBQUMsSUFBNkI7UUFDbkQsSUFBSSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUM7SUFDeEIsQ0FBQztJQUVNLFlBQVksQ0FBQyxJQUF5QjtRQUszQyxJQUNFLElBQUksQ0FBQyxJQUFJO1lBQ1QsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLEtBQUssbUJBQU0sQ0FBQyxtQkFBbUI7WUFDN0MsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLEtBQUssS0FBSyxFQUN4QjtZQUNBLElBQUksQ0FBQyxZQUFZLENBQUMsY0FBYyxDQUFDLElBQUksQ0FBQyxDQUFDO1NBQ3hDO1FBRUQsSUFBSSxDQUFDLGFBQWEsQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUV6QixJQUFJLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxDQUFDO0lBQ25CLENBQUM7SUFFTSxlQUFlLENBQUMsSUFBNEI7UUFDakQsSUFBSSxDQUFDLFVBQVUsQ0FBQyxJQUFJLENBQUMsQ0FBQztJQUN4QixDQUFDO0lBRU0sZ0JBQWdCLENBQUMsSUFBNkI7UUFDbkQsSUFBSSxDQUFDLFVBQVUsQ0FBQyxJQUFJLENBQUMsQ0FBQztJQUN4QixDQUFDO0lBRU0sY0FBYyxDQUFDLElBQTJCO1FBRS9DLElBQ0UsQ0FBQyxJQUFJLENBQUMsWUFBWSxDQUFDLFlBQVksRUFBRTtZQUNqQyxJQUFJLENBQUMsTUFBTSxDQUFDLElBQUksS0FBSyxtQkFBTSxDQUFDLFVBQVU7WUFDdEMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxJQUFJLEtBQUssTUFBTSxFQUMzQjtZQUdBLElBQUksQ0FBQyxZQUFhLENBQUMsYUFBYSxDQUFDLFlBQVksRUFBRSxDQUFDO1NBQ2pEO1FBQ0QsSUFBSSxDQUFDLGFBQWEsQ0FBQyxJQUFJLENBQUMsQ0FBQztJQUMzQixDQUFDO0lBRU0sY0FBYyxDQUFDLElBQTJCO1FBQy9DLElBQUksSUFBSSxDQUFDLFlBQVksQ0FBQyxPQUFPLEVBQUUsRUFBRTtZQUMvQixJQUFJLENBQUMsWUFBWSxDQUFDLGdCQUFnQixDQUFDLElBQUksQ0FBQyxDQUFDO1NBQzFDO1FBRUQsSUFBSSxDQUFDLGFBQWEsQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUV6QixJQUFJLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxDQUFDO0lBQ25CLENBQUM7SUFFTSxjQUFjO1FBQ25CLElBQUksQ0FBQyxZQUFhLENBQUMsYUFBYSxDQUFDLFlBQVksRUFBRSxDQUFDO0lBQ2xELENBQUM7SUFFTSxhQUFhLENBQUMsSUFBMEI7UUFDN0MsSUFBSSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLENBQUM7UUFHeEIsSUFBSSxDQUFDLFlBQVksQ0FBQyxlQUFlLENBQUMsSUFBSSxDQUFDLENBQUM7UUFFeEMsSUFBSSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUM7UUFFdEIsSUFBSSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsQ0FBQztJQUNuQixDQUFDO0lBRU0sbUJBQW1CLENBQUMsSUFBZ0M7UUFDekQsTUFBTSxtQkFBbUIsR0FDdkIsSUFBSSxDQUFDLElBQUksS0FBSyxLQUFLO1lBQ2pCLENBQUMsQ0FBQyxJQUFJLENBQUMsWUFBYSxDQUFDLGFBQWE7WUFDbEMsQ0FBQyxDQUFDLElBQUksQ0FBQyxZQUFhLENBQUM7UUFFekIsS0FDRSxJQUFJLENBQUMsR0FBRyxDQUFDLEVBQUUsRUFBRSxHQUFHLElBQUksQ0FBQyxZQUFZLENBQUMsTUFBTSxFQUN4QyxDQUFDLEdBQUcsRUFBRSxFQUNOLEVBQUUsQ0FBQyxFQUNIO1lBQ0EsTUFBTSxJQUFJLEdBQUcsSUFBSSxDQUFDLFlBQVksQ0FBQyxDQUFDLENBQUMsQ0FBQztZQUVsQyxJQUFJLENBQUMsd0JBQXdCLENBQzNCLG1CQUFtQixFQUNuQix1QkFBWSxDQUFDLFFBQVEsRUFDckIsSUFBSSxFQUNKLENBQUMsQ0FDRixDQUFDO1lBQ0YsSUFBSSxJQUFJLENBQUMsSUFBSSxFQUFFO2dCQUNiLElBQUksQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDO2FBQ3ZCO1NBQ0Y7SUFDSCxDQUFDO0lBR00sZUFBZSxDQUFDLElBQTRCO1FBQ2pELElBQUksQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLFlBQVksQ0FBQyxDQUFDO1FBRTlCLElBQUksSUFBSSxDQUFDLFlBQVksQ0FBQyxPQUFPLEVBQUUsRUFBRTtZQUMvQixJQUFJLENBQUMsWUFBWSxDQUFDLGlCQUFpQixDQUFDLElBQUksQ0FBQyxDQUFDO1NBQzNDO1FBRUQsS0FBSyxJQUFJLENBQUMsR0FBRyxDQUFDLEVBQUUsRUFBRSxHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsTUFBTSxFQUFFLENBQUMsR0FBRyxFQUFFLEVBQUUsRUFBRSxDQUFDLEVBQUU7WUFDbkQsSUFBSSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7U0FDM0I7UUFFRCxJQUFJLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxDQUFDO0lBQ25CLENBQUM7SUFFTSxtQkFBbUIsQ0FBQyxJQUFnQztRQUN6RCxJQUFJLENBQUMsYUFBYSxDQUFDLElBQUksQ0FBQyxDQUFDO0lBQzNCLENBQUM7SUFFTSxrQkFBa0IsQ0FBQyxJQUErQjtRQUN2RCxJQUFJLENBQUMsYUFBYSxDQUFDLElBQUksQ0FBQyxDQUFDO0lBQzNCLENBQUM7SUFFTSxjQUFjLENBQUMsSUFBMkI7UUFDL0MsSUFBSSxDQUFDLFVBQVUsQ0FBQyxJQUFJLENBQUMsQ0FBQztJQUN4QixDQUFDO0lBRU0sY0FBYyxDQUFDLElBQTJCO1FBQy9DLElBQUksQ0FBQyxVQUFVLENBQUMsSUFBSSxDQUFDLENBQUM7SUFDeEIsQ0FBQztJQUVNLHVCQUF1QixDQUM1QixJQUFvQztRQUVwQyxJQUFJLENBQUMsYUFBYSxDQUFDLElBQUksQ0FBQyxDQUFDO0lBQzNCLENBQUM7SUFFTSxpQkFBaUIsQ0FBQyxJQUE4QjtRQUNyRCxNQUFNLENBQ0osSUFBSSxDQUFDLFlBQVksQ0FBQyxPQUFPLEVBQUU7WUFDekIsSUFBSSxDQUFDLFlBQVksQ0FBQyxRQUFRLEVBQUUsRUFDOUIsaUZBQWlGLENBQ2xGLENBQUM7UUFFRixNQUFNLFFBQVEsR0FBRyxJQUFJLFFBQVEsQ0FBQyxJQUFJLEVBQUUsSUFBSSxDQUFDLENBQUM7UUFFMUMsUUFBUSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsQ0FBQztJQUN2QixDQUFDO0lBRU0sc0JBQXNCLENBQzNCLElBQW1DO1FBRW5DLElBQUksSUFBSSxDQUFDLE1BQU0sRUFBRTtZQUNmLElBQUksQ0FBQyxlQUFlLEdBQUcsSUFBSSxDQUFDLE1BQU0sQ0FBQyxLQUFlLENBQUM7U0FDcEQ7YUFBTTtZQUNMLElBQUksQ0FBQyxlQUFlLEdBQUcsSUFBSSxDQUFDO1NBQzdCO1FBQ0QsSUFBSSxJQUFJLENBQUMsV0FBVyxFQUFFO1lBQ3BCLElBQUksQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLFdBQVcsQ0FBQyxDQUFDO1lBQzdCLE9BQU87U0FDUjtRQUVELElBQUksQ0FBQyxhQUFhLENBQUMsSUFBSSxDQUFDLENBQUM7SUFDM0IsQ0FBQztJQUVNLHNCQUFzQixDQUMzQixJQUFtQztRQUVuQyxNQUFNLFdBQVcsR0FBRyxJQUFJLENBQUMsWUFBMkIsQ0FBQztRQUNyRCxNQUFNLFFBQVEsR0FBRyxXQUFXLENBQUMsMkJBQTJCLENBQUM7UUFDekQsV0FBVyxDQUFDLDJCQUEyQixHQUFHLElBQUksQ0FBQztRQUMvQyxJQUFJLENBQUMsc0JBQXNCLENBQUMsSUFBSSxDQUFDLENBQUM7UUFDbEMsV0FBVyxDQUFDLDJCQUEyQixHQUFHLFFBQVEsQ0FBQztJQUNyRCxDQUFDO0lBRU0sd0JBQXdCLENBQzdCLElBQXFDO1FBRXJDLE1BQU0sWUFBWSxHQUFHLElBQUksQ0FBQyxZQUEyQixDQUFDO1FBQ3RELElBQUksWUFBWSxDQUFDLElBQUksS0FBSyxRQUFRLEVBQUU7WUFDbEMsTUFBTSxJQUFJLEtBQUssQ0FBQyxrQ0FBa0MsQ0FBQyxDQUFDO1NBQ3JEO1FBRUQsSUFBSSxjQUFtQyxDQUFDO1FBQ3hDLElBQUksSUFBSSxDQUFDLFdBQVcsQ0FBQyxJQUFJLEtBQUssWUFBWSxFQUFFO1lBQzFDLGNBQWMsR0FBRztnQkFDZixJQUFJLEVBQUUsa0NBQWtCLENBQUMsS0FBSztnQkFDOUIsVUFBVSxFQUFFLFNBQVM7Z0JBQ3JCLFNBQVMsRUFBRSxJQUFJLENBQUMsV0FBVyxDQUFDLElBQUk7Z0JBQ2hDLElBQUksRUFBRSxJQUFJLENBQUMsV0FBVzthQUN2QixDQUFDO1lBQ0YsSUFBSSxDQUFDLG9CQUFvQixHQUFHLElBQUksQ0FBQztZQUNqQyxJQUFJLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxXQUFXLENBQUMsQ0FBQztZQUM3QixJQUFJLENBQUMsb0JBQW9CLEdBQUcsS0FBSyxDQUFDO1NBQ25DO2FBQU07WUFDTCxjQUFjLEdBQUc7Z0JBQ2YsSUFBSSxFQUFFLGtDQUFrQixDQUFDLEtBQUs7Z0JBQzlCLFVBQVUsRUFBRSxTQUFTO2dCQUNyQixTQUFTLEVBQUUsSUFBSTtnQkFDZixJQUFJLEVBQUUsSUFBSSxDQUFDLFdBQVc7YUFDdkIsQ0FBQztZQUNGLElBQUksQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLFdBQVcsQ0FBQyxDQUFDO1NBQzlCO1FBRUQsWUFBWSxDQUFDLGFBQWEsQ0FBQyxzQkFBc0IsQ0FBQyxjQUFjLENBQUMsQ0FBQztJQUNwRSxDQUFDO0lBRU0sb0JBQW9CLENBQ3pCLElBQWlDO1FBRWpDLE1BQU0sWUFBWSxHQUFHLElBQUksQ0FBQyxZQUEyQixDQUFDO1FBQ3RELFlBQVksQ0FBQyxhQUFhLENBQUMsbUJBQW1CLENBQUM7WUFDN0MsSUFBSSxFQUFFLGtDQUFrQixDQUFDLFFBQVE7WUFDakMsVUFBVSxFQUFFLElBQUksQ0FBQyxNQUFNLENBQUMsS0FBZTtZQUN2QyxVQUFVLEVBQUUsNEJBQVksQ0FBQyxHQUFHO1NBQzdCLENBQUMsQ0FBQztJQUNMLENBQUM7SUFFTSxlQUFlLENBQUMsSUFBNEI7UUFDakQsTUFBTSxLQUFLLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQztRQUN6QixNQUFNLFlBQVksR0FBRyxJQUFJLENBQUMsWUFBMkIsQ0FBQztRQUN0RCxJQUFJLElBQUksQ0FBQyxlQUFlLEVBQUU7WUFDeEIsWUFBWSxDQUFDLGFBQWEsQ0FBQyxtQkFBbUIsQ0FBQztnQkFDN0MsSUFBSSxFQUFFLGtDQUFrQixDQUFDLFFBQVE7Z0JBQ2pDLFVBQVUsRUFBRSxJQUFJLENBQUMsZUFBZTtnQkFDaEMsVUFBVSxFQUFFLDRCQUFZLENBQUMsVUFBVTtnQkFDbkMsS0FBSyxFQUFFO29CQUNMLFVBQVUsRUFBRSxJQUFJLENBQUMsUUFBUSxDQUFDLElBQUk7b0JBQzlCLFVBQVUsRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLElBQUk7aUJBQzVCO2FBQ0YsQ0FBQyxDQUFDO1NBQ0o7YUFBTTtZQUNMLElBQUksQ0FBQyxvQkFBb0IsR0FBRyxJQUFJLENBQUM7WUFDakMsSUFBSSxDQUFDLEtBQUssQ0FBQyxLQUFLLENBQUMsQ0FBQztZQUNsQixZQUFZLENBQUMsYUFBYSxDQUFDLHNCQUFzQixDQUFDO2dCQUNoRCxJQUFJLEVBQUUsa0NBQWtCLENBQUMsS0FBSztnQkFDOUIsVUFBVSxFQUFFLElBQUksQ0FBQyxRQUFRLENBQUMsSUFBSTtnQkFDOUIsU0FBUyxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsSUFBSTtnQkFDMUIsSUFBSSxFQUFFLElBQUksQ0FBQyxLQUFLO2FBQ2pCLENBQUMsQ0FBQztZQUNILElBQUksQ0FBQyxvQkFBb0IsR0FBRyxLQUFLLENBQUM7U0FDbkM7SUFDSCxDQUFDO0lBRU0sWUFBWTtJQUduQixDQUFDO0NBQ0Y7QUE3dUJELGdDQTZ1QkMifQ==