"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const estraverse_1 = require("estraverse");
const esrecurse = require("esrecurse");
function getLast(xs) {
    return xs[xs.length - 1] || null;
}
class PatternVisitor extends esrecurse.Visitor {
    static isPattern(node) {
        const nodeType = node.type;
        return (nodeType === estraverse_1.Syntax.Identifier ||
            nodeType === estraverse_1.Syntax.ObjectPattern ||
            nodeType === estraverse_1.Syntax.ArrayPattern ||
            nodeType === estraverse_1.Syntax.SpreadElement ||
            nodeType === estraverse_1.Syntax.RestElement ||
            nodeType === estraverse_1.Syntax.AssignmentPattern);
    }
    constructor(options, rootPattern, callback) {
        super(undefined, options);
        this.rootPattern = rootPattern;
        this.callback = callback;
        this.assignments = [];
        this.rightHandNodes = [];
        this.restElements = [];
    }
    Identifier(pattern) {
        const lastRestElement = getLast(this.restElements);
        this.callback(pattern, {
            topLevel: pattern === this.rootPattern,
            rest: lastRestElement !== null &&
                lastRestElement !== undefined &&
                lastRestElement.argument === pattern,
            assignments: this.assignments,
        });
    }
    Property(property) {
        if (property.computed) {
            this.rightHandNodes.push(property.key);
        }
        this.visit(property.value);
    }
    ArrayPattern(pattern) {
        for (let i = 0, iz = pattern.elements.length; i < iz; ++i) {
            const element = pattern.elements[i];
            this.visit(element);
        }
    }
    AssignmentPattern(pattern) {
        this.assignments.push(pattern);
        this.visit(pattern.left);
        this.rightHandNodes.push(pattern.right);
        this.assignments.pop();
    }
    RestElement(pattern) {
        this.restElements.push(pattern);
        this.visit(pattern.argument);
        this.restElements.pop();
    }
    MemberExpression(node) {
        if (node.computed) {
            this.rightHandNodes.push(node.property);
        }
        this.rightHandNodes.push(node.object);
    }
    SpreadElement(node) {
        this.visit(node.argument);
    }
    ArrayExpression(node) {
        node.elements.forEach(this.visit, this);
    }
    AssignmentExpression(node) {
        this.assignments.push(node);
        this.visit(node.left);
        this.rightHandNodes.push(node.right);
        this.assignments.pop();
    }
    CallExpression(node) {
        node.arguments.forEach(a => {
            this.rightHandNodes.push(a);
        });
        this.visit(node.callee);
    }
}
exports.PatternVisitor = PatternVisitor;
//# sourceMappingURL=data:application/json;base64,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