"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const _1 = require(".");
const estraverse = require("estraverse");
class VariableVirtualScope {
    constructor(contentType, variable, isChildrenDependent = true) {
        this.contentType = contentType;
        this.variable = variable;
        this.isChildrenDependent = isChildrenDependent;
        this.children = [];
        if (contentType === _1.VScopeContentType.PureFunctionCall) {
            const { declarator } = this;
            this.pureRange = [declarator.start, declarator.end];
        }
    }
    get type() {
        return _1.VirtualScopeType.Variable;
    }
    get declarator() {
        const def = this.variable.defs[0];
        const node = def.node;
        if (typeof node === "undefined") {
            throw new TypeError("node.init is undefined");
        }
        return node;
    }
    get isImport() {
        return this.contentType === _1.VScopeContentType.Import;
    }
    findAllReferencesToVirtualScope(visitedSet, scopeManager, virtualScopeMap) {
        if (!this.isChildrenDependent)
            return;
        switch (this.contentType) {
            case _1.VScopeContentType.ClassDeclaration:
            case _1.VScopeContentType.FunctionDeclaration:
                this.traverseDeclarationScope(visitedSet, scopeManager, virtualScopeMap);
                break;
            case _1.VScopeContentType.ArrowFunction:
            case _1.VScopeContentType.FunctionExpression:
            case _1.VScopeContentType.ClassExpression:
                this.traverseExpressionScope(visitedSet, scopeManager, virtualScopeMap);
                break;
            case _1.VScopeContentType.PureFunctionCall:
                this.traversePureDeclarator(visitedSet, this.declarator, scopeManager, virtualScopeMap);
                break;
            case _1.VScopeContentType.NormalFunctionCall:
                break;
            case _1.VScopeContentType.Undefined:
                break;
        }
    }
    getModuleScope(scopeManager) {
        return scopeManager.scopes[1];
    }
    traverseDeclarationScope(visitedSet, scopeManager, virtualScopeMap) {
        const def = this.variable.defs[0];
        const scopes = scopeManager.__nodeToScope.get(def.node);
        const moduleScope = this.getModuleScope(scopeManager);
        this.traverseScopes(scopes, moduleScope, visitedSet, virtualScopeMap);
    }
    traverseExpressionScope(visitedSet, scopeManager, virtualScopeMap) {
        const def = this.variable.defs[0];
        const node = def.node;
        const { init } = node;
        const scopes = scopeManager.__nodeToScope.get(init);
        const moduleScope = this.getModuleScope(scopeManager);
        this.traverseScopes(scopes, moduleScope, visitedSet, virtualScopeMap);
    }
    traverseScopes(scopes, moduleScope, visitedSet, virtualScopeMap) {
        scopes.forEach(scope => {
            visitedSet.add(scope);
            const traverse = (scope) => {
                scope.references.forEach(ref => {
                    if (ref.resolved &&
                        ref.resolved.scope.type === "module") {
                        const idName = ref.identifier.name;
                        this.addToVs(idName, moduleScope, virtualScopeMap);
                    }
                });
                scope.childScopes.forEach(traverse);
            };
            traverse(scope);
        });
    }
    traversePureDeclarator(visitedSet, validatorDeclarator, scopeManager, virtualScopeMap) {
        const moduleScope = this.getModuleScope(scopeManager);
        const nodeContains = (node1, node2) => {
            return node2.start >= node1.start && node2.end <= node1.end;
        };
        moduleScope.childScopes.forEach(scope => {
            const block = scope.block;
            if (nodeContains(validatorDeclarator, block)) {
                visitedSet.add(scope);
            }
        });
        estraverse.traverse(validatorDeclarator, {
            enter: (node) => {
                if (node.type === "Identifier") {
                    const idName = node.name;
                    this.addToVs(idName, moduleScope, virtualScopeMap);
                }
            },
        });
    }
    addToVs(idName, moduleScope, virtualScopeMap) {
        const refVar = moduleScope.set.get(idName);
        if (typeof refVar !== "undefined") {
            const virtualScope = virtualScopeMap.get(refVar);
            this.children.push(virtualScope);
        }
    }
}
exports.VariableVirtualScope = VariableVirtualScope;
//# sourceMappingURL=data:application/json;base64,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