import * as React from 'react';
import { View, StyleSheet } from 'react-native';
import { withTheme } from '../../core/theming';
import { RadioButtonContext } from './RadioButtonGroup';
import { handlePress } from './utils';
import TouchableRipple from '../TouchableRipple/TouchableRipple';
import RadioButton from './RadioButton';
import Text from '../Typography/Text';

/**
 * RadioButton.Item allows you to press the whole row (item) instead of only the RadioButton.
 *
 * <div class="screenshots">
 *   <figure>
 *     <img class="medium" src="screenshots/radio-item.ios.png" />
 *     <figcaption>Pressed</figcaption>
 *   </figure>
 * </div>
 *
 * ## Usage
 * ```js
 * import * as React from 'react';
 * import { RadioButton } from 'react-native-paper';
 *
 * const MyComponent = () => {
 *   const [value, setValue] = React.useState('first');
 *
 *   return (
 *     <RadioButton.Group onValueChange={value => setValue(value)} value={value}>
 *       <RadioButton.Item label="First item" value="first" />
 *       <RadioButton.Item label="Second item" value="second" />
 *     </RadioButton.Group>
 *   );
 * };
 *
 * export default MyComponent;
 *```
 */
const RadioButtonItem = ({
  value,
  label,
  style,
  labelStyle,
  onPress,
  disabled,
  color,
  uncheckedColor,
  status,
  theme: {
    colors
  },
  accessibilityLabel
}) => /*#__PURE__*/React.createElement(RadioButtonContext.Consumer, null, context => {
  return /*#__PURE__*/React.createElement(TouchableRipple, {
    onPress: disabled ? undefined : () => handlePress({
      onPress: onPress,
      onValueChange: context === null || context === void 0 ? void 0 : context.onValueChange,
      value
    }),
    accessibilityLabel: accessibilityLabel
  }, /*#__PURE__*/React.createElement(View, {
    style: [styles.container, style],
    pointerEvents: "none"
  }, /*#__PURE__*/React.createElement(Text, {
    style: [styles.label, {
      color: colors.text
    }, labelStyle]
  }, label), /*#__PURE__*/React.createElement(RadioButton, {
    value: value,
    disabled: disabled,
    status: status,
    color: color,
    uncheckedColor: uncheckedColor
  })));
});

RadioButtonItem.displayName = 'RadioButton.Item';
export default withTheme(RadioButtonItem); // @component-docs ignore-next-line

export { RadioButtonItem };
const styles = StyleSheet.create({
  container: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingVertical: 8,
    paddingHorizontal: 16
  },
  label: {
    fontSize: 16
  }
});
//# sourceMappingURL=RadioButtonItem.js.map