"use strict";
// ExtensionCodec to handle MessagePack extensions
Object.defineProperty(exports, "__esModule", { value: true });
exports.ExtensionCodec = void 0;
const ExtData_1 = require("./ExtData");
const timestamp_1 = require("./timestamp");
let ExtensionCodec = /** @class */ (() => {
    class ExtensionCodec {
        constructor() {
            // built-in extensions
            this.builtInEncoders = [];
            this.builtInDecoders = [];
            // custom extensions
            this.encoders = [];
            this.decoders = [];
            this.register(timestamp_1.timestampExtension);
        }
        register({ type, encode, decode, }) {
            if (type >= 0) {
                // custom extensions
                this.encoders[type] = encode;
                this.decoders[type] = decode;
            }
            else {
                // built-in extensions
                const index = 1 + type;
                this.builtInEncoders[index] = encode;
                this.builtInDecoders[index] = decode;
            }
        }
        tryToEncode(object, context) {
            // built-in extensions
            for (let i = 0; i < this.builtInEncoders.length; i++) {
                const encoder = this.builtInEncoders[i];
                if (encoder != null) {
                    const data = encoder(object, context);
                    if (data != null) {
                        const type = -1 - i;
                        return new ExtData_1.ExtData(type, data);
                    }
                }
            }
            // custom extensions
            for (let i = 0; i < this.encoders.length; i++) {
                const encoder = this.encoders[i];
                if (encoder != null) {
                    const data = encoder(object, context);
                    if (data != null) {
                        const type = i;
                        return new ExtData_1.ExtData(type, data);
                    }
                }
            }
            if (object instanceof ExtData_1.ExtData) {
                // to keep ExtData as is
                return object;
            }
            return null;
        }
        decode(data, type, context) {
            const decoder = type < 0 ? this.builtInDecoders[-1 - type] : this.decoders[type];
            if (decoder) {
                return decoder(data, type, context);
            }
            else {
                // decode() does not fail, returns ExtData instead.
                return new ExtData_1.ExtData(type, data);
            }
        }
    }
    ExtensionCodec.defaultCodec = new ExtensionCodec();
    return ExtensionCodec;
})();
exports.ExtensionCodec = ExtensionCodec;
//# sourceMappingURL=ExtensionCodec.js.map