"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const WarningAggregator_1 = require("../WarningAggregator");
function getScheme(config) {
    if (Array.isArray(config.scheme)) {
        const validate = (value) => typeof value === 'string';
        return config.scheme.filter(validate);
    }
    else if (typeof config.scheme === 'string') {
        return [config.scheme];
    }
    return [];
}
exports.getScheme = getScheme;
async function setScheme(config, manifestDocument) {
    var _a, _b;
    const scheme = [
        ...getScheme(config),
        // @ts-ignore: TODO: android.scheme is an unreleased -- harder to add to turtle v1.
        ...getScheme((_a = config.android) !== null && _a !== void 0 ? _a : {}),
    ];
    // Add the package name to the list of schemes for easier Google auth and parity with Turtle v1.
    if ((_b = config.android) === null || _b === void 0 ? void 0 : _b['package']) {
        scheme.push(config.android['package']);
    }
    if (scheme.length === 0) {
        return manifestDocument;
    }
    if (!ensureManifestHasValidIntentFilter(manifestDocument)) {
        WarningAggregator_1.addWarningAndroid('scheme', `Cannot add schemes because the provided manifest does not have a valid Activity with \`android:launchMode="singleTask"\`.\nThis guide can help you get setup properly https://expo.fyi/setup-android-uri-scheme`);
        return manifestDocument;
    }
    // Get the current schemes and remove them.
    const currentSchemes = getSchemesFromManifest(manifestDocument);
    for (const uri of currentSchemes) {
        manifestDocument = removeScheme(uri, manifestDocument);
    }
    // Now add all the new schemes.
    for (const uri of scheme) {
        manifestDocument = appendScheme(uri, manifestDocument);
    }
    return manifestDocument;
}
exports.setScheme = setScheme;
function isValidRedirectIntentFilter({ actions, categories, schemes }) {
    return (actions.includes('android.intent.action.VIEW') &&
        !categories.includes('android.intent.category.LAUNCHER'));
}
function propertiesFromIntentFilter(intentFilter) {
    var _a, _b, _c, _d, _e, _f;
    const actions = (_b = (_a = intentFilter === null || intentFilter === void 0 ? void 0 : intentFilter.action) === null || _a === void 0 ? void 0 : _a.map((data) => { var _a; return (_a = data === null || data === void 0 ? void 0 : data['$']) === null || _a === void 0 ? void 0 : _a['android:name']; })) !== null && _b !== void 0 ? _b : [];
    const categories = (_d = (_c = intentFilter === null || intentFilter === void 0 ? void 0 : intentFilter.category) === null || _c === void 0 ? void 0 : _c.map((data) => { var _a; return (_a = data === null || data === void 0 ? void 0 : data['$']) === null || _a === void 0 ? void 0 : _a['android:name']; })) !== null && _d !== void 0 ? _d : [];
    const schemes = (_f = (_e = intentFilter === null || intentFilter === void 0 ? void 0 : intentFilter.data) === null || _e === void 0 ? void 0 : _e.map((data) => { var _a; return (_a = data === null || data === void 0 ? void 0 : data['$']) === null || _a === void 0 ? void 0 : _a['android:scheme']; })) !== null && _f !== void 0 ? _f : [];
    return {
        schemes,
        actions,
        categories,
    };
}
function getSingleTaskIntentFilters(manifestDocument) {
    if (!Array.isArray(manifestDocument.manifest.application))
        return [];
    let outputSchemes = [];
    for (const application of manifestDocument.manifest.application) {
        const { activity } = application;
        // @ts-ignore
        const activities = Array.isArray(activity) ? activity : [activity];
        const singleTaskActivities = activities.filter(activity => { var _a; return ((_a = activity === null || activity === void 0 ? void 0 : activity['$']) === null || _a === void 0 ? void 0 : _a['android:launchMode']) === 'singleTask'; });
        for (const activity of singleTaskActivities) {
            const intentFilters = activity['intent-filter'];
            outputSchemes = outputSchemes.concat(intentFilters);
        }
    }
    return outputSchemes;
}
function getSchemesFromManifest(manifestDocument) {
    const outputSchemes = [];
    const singleTaskIntentFilters = getSingleTaskIntentFilters(manifestDocument);
    for (const intentFilter of singleTaskIntentFilters) {
        const properties = propertiesFromIntentFilter(intentFilter);
        if (isValidRedirectIntentFilter(properties)) {
            outputSchemes.push(properties);
        }
    }
    return outputSchemes.reduce((prev, { schemes }) => [...prev, ...schemes], []);
}
exports.getSchemesFromManifest = getSchemesFromManifest;
function ensureManifestHasValidIntentFilter(manifestDocument) {
    var _a;
    if (!Array.isArray(manifestDocument.manifest.application)) {
        return false;
    }
    for (const application of manifestDocument.manifest.application) {
        for (const activity of application.activity || []) {
            if (((_a = activity === null || activity === void 0 ? void 0 : activity['$']) === null || _a === void 0 ? void 0 : _a['android:launchMode']) === 'singleTask') {
                for (const intentFilter of activity['intent-filter'] || []) {
                    // Parse valid intent filters...
                    const properties = propertiesFromIntentFilter(intentFilter);
                    if (isValidRedirectIntentFilter(properties)) {
                        return true;
                    }
                }
                if (!activity['intent-filter']) {
                    activity['intent-filter'] = [];
                }
                activity['intent-filter'].push({
                    action: [{ $: { 'android:name': 'android.intent.action.VIEW' } }],
                    category: [
                        { $: { 'android:name': 'android.intent.category.DEFAULT' } },
                        { $: { 'android:name': 'android.intent.category.BROWSABLE' } },
                    ],
                });
                return true;
            }
        }
    }
    return false;
}
exports.ensureManifestHasValidIntentFilter = ensureManifestHasValidIntentFilter;
function hasScheme(scheme, manifestDocument) {
    const schemes = getSchemesFromManifest(manifestDocument);
    return schemes.includes(scheme);
}
exports.hasScheme = hasScheme;
function appendScheme(scheme, manifestDocument) {
    var _a;
    if (!Array.isArray(manifestDocument.manifest.application)) {
        return manifestDocument;
    }
    for (const application of manifestDocument.manifest.application) {
        for (const activity of application.activity || []) {
            if (((_a = activity === null || activity === void 0 ? void 0 : activity['$']) === null || _a === void 0 ? void 0 : _a['android:launchMode']) === 'singleTask') {
                for (const intentFilter of activity['intent-filter'] || []) {
                    const properties = propertiesFromIntentFilter(intentFilter);
                    if (isValidRedirectIntentFilter(properties)) {
                        if (!intentFilter.data)
                            intentFilter.data = [];
                        intentFilter.data.push({
                            $: { 'android:scheme': scheme },
                        });
                    }
                }
                break;
            }
        }
    }
    return manifestDocument;
}
exports.appendScheme = appendScheme;
function removeScheme(scheme, manifestDocument) {
    var _a, _b, _c, _d;
    if (!Array.isArray(manifestDocument.manifest.application)) {
        return manifestDocument;
    }
    for (const application of manifestDocument.manifest.application) {
        for (const activity of application.activity || []) {
            if (((_a = activity === null || activity === void 0 ? void 0 : activity['$']) === null || _a === void 0 ? void 0 : _a['android:launchMode']) === 'singleTask') {
                for (const intentFilter of activity['intent-filter'] || []) {
                    // Parse valid intent filters...
                    const properties = propertiesFromIntentFilter(intentFilter);
                    if (isValidRedirectIntentFilter(properties)) {
                        for (const dataKey in (intentFilter === null || intentFilter === void 0 ? void 0 : intentFilter.data) || []) {
                            const data = (_b = intentFilter.data) === null || _b === void 0 ? void 0 : _b[dataKey];
                            if (((_c = data === null || data === void 0 ? void 0 : data['$']) === null || _c === void 0 ? void 0 : _c['android:scheme']) === scheme) {
                                (_d = intentFilter.data) === null || _d === void 0 ? true : delete _d[dataKey];
                            }
                        }
                    }
                }
                break;
            }
        }
    }
    return manifestDocument;
}
exports.removeScheme = removeScheme;
//# sourceMappingURL=Scheme.js.map