"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    Object.defineProperty(o, k2, { enumerable: true, get: function() { return m[k]; } });
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __exportStar = (this && this.__exportStar) || function(m, exports) {
    for (var p in m) if (p !== "default" && !exports.hasOwnProperty(p)) __createBinding(exports, m, p);
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.CollectionMemberManagerOnline = exports.CollectionInvitationManagerOnline = exports.CollectionItemManagerOnline = exports.CollectionManagerOnline = exports.Authenticator = exports.PrefetchOption = void 0;
const Request_1 = __importDefault(require("./Request"));
const urijs_1 = __importDefault(require("urijs"));
var Crypto_1 = require("./Crypto");
Object.defineProperty(exports, "deriveKey", { enumerable: true, get: function () { return Crypto_1.deriveKey; } });
Object.defineProperty(exports, "ready", { enumerable: true, get: function () { return Crypto_1.ready; } });
const Exceptions_1 = require("./Exceptions");
__exportStar(require("./Exceptions"), exports);
const Helpers_1 = require("./Helpers");
const EncryptedModels_1 = require("./EncryptedModels");
var PrefetchOption;
(function (PrefetchOption) {
    PrefetchOption["Auto"] = "auto";
    PrefetchOption["Medium"] = "medium";
})(PrefetchOption = exports.PrefetchOption || (exports.PrefetchOption = {}));
class BaseNetwork {
    constructor(apiBase) {
        this.apiBase = urijs_1.default(apiBase).normalize();
    }
    static urlExtend(baseUrlIn, segments) {
        const baseUrl = baseUrlIn.clone();
        for (const segment of segments) {
            baseUrl.segment(segment);
        }
        baseUrl.segment("");
        return baseUrl.normalize();
    }
    async newCall(segments = [], extra = {}, apiBaseIn = this.apiBase) {
        const apiBase = BaseNetwork.urlExtend(apiBaseIn, segments);
        extra = {
            ...extra,
            headers: {
                Accept: "application/msgpack",
                ...extra.headers,
            },
        };
        let response;
        try {
            response = await Request_1.default(apiBase.toString(), extra);
        }
        catch (e) {
            throw new Exceptions_1.NetworkError(e.message);
        }
        const body = response.body;
        let data;
        let strError = undefined;
        try {
            data = Helpers_1.msgpackDecode(body);
        }
        catch (e) {
            data = new Uint8Array(body);
            try {
                strError = Helpers_1.toString(data);
            }
            catch (_a) {
                // Ignore
            }
        }
        if (response.ok) {
            return data;
        }
        else {
            const content = data.detail || data.non_field_errors || strError;
            switch (response.status) {
                case 401: throw new Exceptions_1.UnauthorizedError(content, data);
                case 403: throw new Exceptions_1.PermissionDeniedError(content);
                case 404: throw new Exceptions_1.NotFoundError(content);
                case 409: throw new Exceptions_1.ConflictError(content);
                case 502:
                case 503:
                case 504:
                    throw new Exceptions_1.TemporaryServerError(response.status, content, data);
                default: {
                    if ((response.status >= 500) && (response.status <= 599)) {
                        throw new Exceptions_1.ServerError(response.status, content, data);
                    }
                    else {
                        throw new Exceptions_1.HttpError(response.status, content, data);
                    }
                }
            }
        }
    }
}
class Authenticator extends BaseNetwork {
    constructor(apiBase) {
        super(apiBase);
        this.apiBase = BaseNetwork.urlExtend(this.apiBase, ["api", "v1", "authentication"]);
    }
    async isEtebase() {
        try {
            await this.newCall(["is_etebase"]);
            return true;
        }
        catch (e) {
            if (e instanceof Exceptions_1.NotFoundError) {
                return false;
            }
            throw e;
        }
    }
    async signup(user, salt, loginPubkey, pubkey, encryptedContent) {
        user = {
            username: user.username,
            email: user.email,
        };
        const extra = {
            method: "post",
            headers: {
                "Content-Type": "application/msgpack",
            },
            body: Helpers_1.msgpackEncode({
                user,
                salt: salt,
                loginPubkey: loginPubkey,
                pubkey: pubkey,
                encryptedContent: encryptedContent,
            }),
        };
        return this.newCall(["signup"], extra);
    }
    getLoginChallenge(username) {
        const extra = {
            method: "post",
            headers: {
                "Content-Type": "application/msgpack",
            },
            body: Helpers_1.msgpackEncode({ username }),
        };
        return this.newCall(["login_challenge"], extra);
    }
    login(response, signature) {
        const extra = {
            method: "post",
            headers: {
                "Content-Type": "application/msgpack",
            },
            body: Helpers_1.msgpackEncode({
                response: response,
                signature: signature,
            }),
        };
        return this.newCall(["login"], extra);
    }
    logout(authToken) {
        const extra = {
            method: "post",
            headers: {
                "Content-Type": "application/msgpack",
                "Authorization": "Token " + authToken,
            },
        };
        return this.newCall(["logout"], extra);
    }
    async changePassword(authToken, response, signature) {
        const extra = {
            method: "post",
            headers: {
                "Content-Type": "application/msgpack",
                "Authorization": "Token " + authToken,
            },
            body: Helpers_1.msgpackEncode({
                response: response,
                signature: signature,
            }),
        };
        await this.newCall(["change_password"], extra);
    }
    async getDashboardUrl(authToken) {
        const extra = {
            method: "post",
            headers: {
                "Content-Type": "application/msgpack",
                "Authorization": "Token " + authToken,
            },
        };
        const ret = await this.newCall(["dashboard_url"], extra);
        return ret.url;
    }
}
exports.Authenticator = Authenticator;
class BaseManager extends BaseNetwork {
    constructor(etebase, segments) {
        super(etebase.serverUrl);
        this.etebase = etebase;
        this.apiBase = BaseNetwork.urlExtend(this.apiBase, ["api", "v1"].concat(segments));
    }
    newCall(segments = [], extra = {}, apiBase = this.apiBase) {
        extra = {
            ...extra,
            headers: {
                "Content-Type": "application/msgpack",
                "Authorization": "Token " + this.etebase.authToken,
                ...extra.headers,
            },
        };
        return super.newCall(segments, extra, apiBase);
    }
    urlFromFetchOptions(options) {
        if (!options) {
            return this.apiBase;
        }
        const { stoken, prefetch, limit, withCollection, iterator } = options;
        return this.apiBase.clone().search({
            stoken: (stoken !== null) ? stoken : undefined,
            iterator: (iterator !== null) ? iterator : undefined,
            limit: (limit && (limit > 0)) ? limit : undefined,
            withCollection: withCollection,
            prefetch,
        });
    }
}
class CollectionManagerOnline extends BaseManager {
    constructor(etebase) {
        super(etebase, ["collection"]);
    }
    async fetch(colUid, options) {
        const apiBase = this.urlFromFetchOptions(options);
        const json = await this.newCall([colUid], undefined, apiBase);
        return EncryptedModels_1.EncryptedCollection.deserialize(json);
    }
    async list(collectionTypes, options) {
        const apiBase = this.urlFromFetchOptions(options);
        const extra = {
            method: "post",
            body: Helpers_1.msgpackEncode({ collectionTypes }),
        };
        const json = await this.newCall(["list_multi"], extra, apiBase);
        return {
            ...json,
            data: json.data.map((val) => EncryptedModels_1.EncryptedCollection.deserialize(val)),
        };
    }
    create(collection, options) {
        const apiBase = this.urlFromFetchOptions(options);
        const extra = {
            method: "post",
            body: Helpers_1.msgpackEncode(collection.serialize()),
        };
        return this.newCall(undefined, extra, apiBase);
    }
}
exports.CollectionManagerOnline = CollectionManagerOnline;
class CollectionItemManagerOnline extends BaseManager {
    constructor(etebase, col) {
        super(etebase, ["collection", col.uid, "item"]);
    }
    async fetch(itemUid, options) {
        const apiBase = this.urlFromFetchOptions(options);
        const json = await this.newCall([itemUid], undefined, apiBase);
        return EncryptedModels_1.EncryptedCollectionItem.deserialize(json);
    }
    async list(options) {
        const apiBase = this.urlFromFetchOptions(options);
        const json = await this.newCall(undefined, undefined, apiBase);
        return {
            ...json,
            data: json.data.map((val) => EncryptedModels_1.EncryptedCollectionItem.deserialize(val)),
        };
    }
    async itemRevisions(item, options) {
        const apiBase = this.urlFromFetchOptions(options);
        const { uid, encryptionKey, version } = item.serialize();
        const json = await this.newCall([item.uid, "revision"], undefined, apiBase);
        return {
            ...json,
            data: json.data.map((val) => EncryptedModels_1.EncryptedCollectionItem.deserialize({
                uid,
                encryptionKey,
                version,
                etag: val.uid,
                content: val,
            })),
        };
    }
    create(item) {
        const extra = {
            method: "post",
            body: Helpers_1.msgpackEncode(item.serialize()),
        };
        return this.newCall(undefined, extra);
    }
    async fetchUpdates(items, options) {
        const apiBase = this.urlFromFetchOptions(options);
        // We only use stoken if available
        const wantEtag = !(options === null || options === void 0 ? void 0 : options.stoken);
        const extra = {
            method: "post",
            body: Helpers_1.msgpackEncode(items === null || items === void 0 ? void 0 : items.map((x) => ({ uid: x.uid, etag: ((wantEtag) ? x.lastEtag : undefined) }))),
        };
        const json = await this.newCall(["fetch_updates"], extra, apiBase);
        const data = json.data;
        return {
            ...json,
            data: data.map((val) => EncryptedModels_1.EncryptedCollectionItem.deserialize(val)),
        };
    }
    batch(items, deps, options) {
        const apiBase = this.urlFromFetchOptions(options);
        const extra = {
            method: "post",
            body: Helpers_1.msgpackEncode({
                items: items.map((x) => x.serialize()),
                deps: deps === null || deps === void 0 ? void 0 : deps.map((x) => ({ uid: x.uid, etag: x.lastEtag })),
            }),
        };
        return this.newCall(["batch"], extra, apiBase);
    }
    transaction(items, deps, options) {
        const apiBase = this.urlFromFetchOptions(options);
        const extra = {
            method: "post",
            body: Helpers_1.msgpackEncode({
                items: items.map((x) => x.serialize()),
                deps: deps === null || deps === void 0 ? void 0 : deps.map((x) => ({ uid: x.uid, etag: x.lastEtag })),
            }),
        };
        return this.newCall(["transaction"], extra, apiBase);
    }
    chunkUpload(item, chunk, options) {
        const apiBase = this.urlFromFetchOptions(options);
        const [chunkUid, chunkContent] = chunk;
        if (chunkContent === undefined) {
            throw new Exceptions_1.ProgrammingError("Tried uploading a missing chunk.");
        }
        const extra = {
            method: "put",
            headers: {
                "Content-Type": "application/octet-stream",
            },
            body: chunkContent,
        };
        return this.newCall([item.uid, "chunk", chunkUid], extra, apiBase);
    }
    chunkDownload(item, chunkUid, options) {
        const apiBase = this.urlFromFetchOptions(options);
        return this.newCall([item.uid, "chunk", chunkUid, "download"], undefined, apiBase);
    }
}
exports.CollectionItemManagerOnline = CollectionItemManagerOnline;
class CollectionInvitationManagerOnline extends BaseManager {
    constructor(etebase) {
        super(etebase, ["invitation"]);
    }
    async listIncoming(options) {
        const apiBase = this.urlFromFetchOptions(options);
        const json = await this.newCall(["incoming"], undefined, apiBase);
        return {
            ...json,
            data: json.data.map((val) => val),
        };
    }
    async listOutgoing(options) {
        const apiBase = this.urlFromFetchOptions(options);
        const json = await this.newCall(["outgoing"], undefined, apiBase);
        return {
            ...json,
            data: json.data.map((val) => val),
        };
    }
    async accept(invitation, collectionType, encryptionKey) {
        const extra = {
            method: "post",
            body: Helpers_1.msgpackEncode({
                collectionType,
                encryptionKey,
            }),
        };
        return this.newCall(["incoming", invitation.uid, "accept"], extra);
    }
    async reject(invitation) {
        const extra = {
            method: "delete",
        };
        return this.newCall(["incoming", invitation.uid], extra);
    }
    async fetchUserProfile(username) {
        const apiBase = this.apiBase.clone().search({
            username: username,
        });
        return this.newCall(["outgoing", "fetch_user_profile"], undefined, apiBase);
    }
    async invite(invitation) {
        const extra = {
            method: "post",
            body: Helpers_1.msgpackEncode(invitation),
        };
        return this.newCall(["outgoing"], extra);
    }
    async disinvite(invitation) {
        const extra = {
            method: "delete",
        };
        return this.newCall(["outgoing", invitation.uid], extra);
    }
}
exports.CollectionInvitationManagerOnline = CollectionInvitationManagerOnline;
class CollectionMemberManagerOnline extends BaseManager {
    constructor(etebase, col) {
        super(etebase, ["collection", col.uid, "member"]);
    }
    async list(options) {
        const apiBase = this.urlFromFetchOptions(options);
        return this.newCall(undefined, undefined, apiBase);
    }
    async remove(username) {
        const extra = {
            method: "delete",
        };
        return this.newCall([username], extra);
    }
    async leave() {
        const extra = {
            method: "post",
        };
        return this.newCall(["leave"], extra);
    }
    async modifyAccessLevel(username, accessLevel) {
        const extra = {
            method: "patch",
            body: Helpers_1.msgpackEncode({
                accessLevel,
            }),
        };
        return this.newCall([username], extra);
    }
}
exports.CollectionMemberManagerOnline = CollectionMemberManagerOnline;
//# sourceMappingURL=OnlineManagers.js.map