"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const xml2js_1 = require("xml2js");
const Manifest_1 = require("./Manifest");
const Resources_1 = require("./Resources");
const Strings_1 = require("./Strings");
const XML_1 = require("./XML");
const facebookSchemeActivity = (scheme) => `
<activity
    android:name="com.facebook.CustomTabActivity"
    android:exported="true">
    <intent-filter>
        <action android:name="android.intent.action.VIEW" />
        <category android:name="android.intent.category.DEFAULT" />
        <category android:name="android.intent.category.BROWSABLE" />
        <data android:scheme="${scheme}" />
    </intent-filter>
</activity>
`;
function getFacebookScheme(config) {
    var _a;
    return (_a = config.facebookScheme) !== null && _a !== void 0 ? _a : null;
}
exports.getFacebookScheme = getFacebookScheme;
function getFacebookAppId(config) {
    var _a;
    return (_a = config.facebookAppId) !== null && _a !== void 0 ? _a : null;
}
exports.getFacebookAppId = getFacebookAppId;
function getFacebookDisplayName(config) {
    var _a;
    return (_a = config.facebookDisplayName) !== null && _a !== void 0 ? _a : null;
}
exports.getFacebookDisplayName = getFacebookDisplayName;
function getFacebookAutoInitEnabled(config) {
    var _a;
    return (_a = config.facebookAutoInitEnabled) !== null && _a !== void 0 ? _a : null;
}
exports.getFacebookAutoInitEnabled = getFacebookAutoInitEnabled;
function getFacebookAutoLogAppEvents(config) {
    var _a;
    return (_a = config.facebookAutoLogAppEventsEnabled) !== null && _a !== void 0 ? _a : null;
}
exports.getFacebookAutoLogAppEvents = getFacebookAutoLogAppEvents;
function getFacebookAdvertiserIDCollection(config) {
    var _a;
    return (_a = config.facebookAdvertiserIDCollectionEnabled) !== null && _a !== void 0 ? _a : null;
}
exports.getFacebookAdvertiserIDCollection = getFacebookAdvertiserIDCollection;
function removeFacebookCustomTabActivities(mainApplication) {
    // Remove all Facebook CustomTabActivities first
    if ('activity' in mainApplication) {
        mainApplication['activity'] = mainApplication['activity'].filter((activity) => {
            var _a;
            return ((_a = activity['$']) === null || _a === void 0 ? void 0 : _a['android:name']) !== 'com.facebook.CustomTabActivity';
        });
    }
}
async function ensureFacebookActivityAsync({ mainApplication, scheme, }) {
    const facebookSchemeActivityXML = facebookSchemeActivity(scheme);
    const parser = new xml2js_1.Parser();
    const facebookSchemeActivityJSON = await parser.parseStringPromise(facebookSchemeActivityXML);
    //TODO: don't write if facebook scheme activity is already present
    if ('activity' in mainApplication) {
        mainApplication['activity'] = mainApplication['activity'].concat(facebookSchemeActivityJSON['activity']);
    }
    else {
        mainApplication['activity'] = facebookSchemeActivityJSON['activity'];
    }
}
async function setFacebookAppIdString(config, projectDirectory) {
    const appId = getFacebookAppId(config);
    const stringsPath = await Strings_1.getProjectStringsXMLPathAsync(projectDirectory);
    if (!stringsPath) {
        throw new Error(`There was a problem setting your Facebook App ID in ${stringsPath}.`);
    }
    let stringsJSON = await Resources_1.readResourcesXMLAsync({ path: stringsPath });
    if (appId) {
        const stringItemToAdd = [{ _: appId, $: { name: 'facebook_app_id' } }];
        stringsJSON = Strings_1.setStringItem(stringItemToAdd, stringsJSON);
    }
    else {
        stringsJSON = Strings_1.removeStringItem('facebook_app_id', stringsJSON);
    }
    try {
        await XML_1.writeXMLAsync({ path: stringsPath, xml: stringsJSON });
    }
    catch (e) {
        throw new Error(`Error setting facebookAppId. Cannot write strings.xml to ${stringsPath}.`);
    }
    return true;
}
exports.setFacebookAppIdString = setFacebookAppIdString;
async function setFacebookConfig(config, manifestDocument) {
    const scheme = getFacebookScheme(config);
    const appId = getFacebookAppId(config);
    const displayName = getFacebookDisplayName(config);
    const autoInitEnabled = getFacebookAutoInitEnabled(config);
    const autoLogAppEvents = getFacebookAutoLogAppEvents(config);
    const advertiserIdCollection = getFacebookAdvertiserIDCollection(config);
    let mainApplication = Manifest_1.getMainApplication(manifestDocument);
    removeFacebookCustomTabActivities(mainApplication);
    if (scheme) {
        ensureFacebookActivityAsync({ scheme, mainApplication });
    }
    if (appId) {
        mainApplication = Manifest_1.addMetaDataItemToMainApplication(mainApplication, 'com.facebook.sdk.ApplicationId', '@string/facebook_app_id' // The corresponding string is set in setFacebookAppIdString
        );
    }
    else {
        mainApplication = removeMetaDataItemFromMainApplication(mainApplication, 'com.facebook.sdk.ApplicationId');
    }
    if (displayName) {
        mainApplication = Manifest_1.addMetaDataItemToMainApplication(mainApplication, 'com.facebook.sdk.ApplicationName', displayName);
    }
    else {
        mainApplication = removeMetaDataItemFromMainApplication(mainApplication, 'com.facebook.sdk.ApplicationName');
    }
    if (autoInitEnabled !== null) {
        mainApplication = Manifest_1.addMetaDataItemToMainApplication(mainApplication, 'com.facebook.sdk.AutoInitEnabled', autoInitEnabled ? 'true' : 'false');
    }
    else {
        mainApplication = removeMetaDataItemFromMainApplication(mainApplication, 'com.facebook.sdk.AutoInitEnabled');
    }
    if (autoLogAppEvents !== null) {
        mainApplication = Manifest_1.addMetaDataItemToMainApplication(mainApplication, 'com.facebook.sdk.AutoLogAppEventsEnabled', autoLogAppEvents ? 'true' : 'false');
    }
    else {
        mainApplication = removeMetaDataItemFromMainApplication(mainApplication, 'com.facebook.sdk.AutoLogAppEventsEnabled');
    }
    if (advertiserIdCollection !== null) {
        mainApplication = Manifest_1.addMetaDataItemToMainApplication(mainApplication, 'com.facebook.sdk.AdvertiserIDCollectionEnabled', advertiserIdCollection ? 'true' : 'false');
    }
    else {
        mainApplication = removeMetaDataItemFromMainApplication(mainApplication, 'com.facebook.sdk.AdvertiserIDCollectionEnabled');
    }
    return manifestDocument;
}
exports.setFacebookConfig = setFacebookConfig;
// TODO: Use Manifest version after https://github.com/expo/expo-cli/pull/2587 lands
function removeMetaDataItemFromMainApplication(mainApplication, itemName) {
    if (mainApplication.hasOwnProperty('meta-data')) {
        const index = mainApplication['meta-data'].findIndex((e) => e['$']['android:name'] === itemName);
        if (index > -1) {
            mainApplication['meta-data'].splice(index, 1);
        }
    }
    return mainApplication;
}
//# sourceMappingURL=Facebook.js.map