declare type Method = "GET" | "POST" | "DELETE" | "PUT";
declare type Headers = {
    [name: string]: string;
};
interface Request<RequestBody = unknown> extends FetchRequestInit<RequestBody> {
    method: Method;
    url: string;
    timeoutMs?: number;
}
interface CallbackResponse {
    statusCode: number;
    headers: Headers;
    body: string;
}
declare type SuccessCallback = (response: CallbackResponse) => void;
declare type ErrorCallback = (err: Error) => void;
interface ResponseHandler {
    onSuccess: SuccessCallback;
    onError: ErrorCallback;
}
interface NetworkTransport {
    fetch<RequestBody>(request: Request<RequestBody>): Promise<FetchResponse>;
    fetchWithCallbacks<RequestBody>(request: Request<RequestBody>, handler: ResponseHandler): void;
}
declare type AbortSignal = unknown;
/** A controller object that allows you to abort one or more DOM requests as and when desired. */
declare type AbortController = {
    /**
     * The prototype of an instance is the class.
     */
    prototype: AbortController;
    /**
     * Constructs an AbortController.
     */
    new (): AbortController;
    /**
     * Returns the AbortSignal object associated with this object.
     */
    readonly signal: AbortSignal;
    /**
     * Invoking this method will set this object's AbortSignal's aborted flag and signal to any observers that the associated activity is to be aborted.
     */
    abort(): void;
};
declare type FetchRequestInfo = FetchRequest | string;
declare type FetchHeadersInit = FetchHeaders | string[][] | Record<string, string>;
declare type FetchRequestCredentials = "include" | "omit" | "same-origin";
declare type FetchRequestMode = "cors" | "navigate" | "no-cors" | "same-origin";
declare type FetchReadableStream = unknown;
interface FetchBody {
    readonly body: FetchReadableStream | null;
    readonly bodyUsed: boolean;
    arrayBuffer(): Promise<ArrayBuffer>;
    blob(): Promise<unknown>;
    json<ResponseBody = unknown>(): Promise<ResponseBody>;
    text(): Promise<string>;
}
interface FetchHeaders {
    append(name: string, value: string): void;
    delete(name: string): void;
    get(name: string): string | null;
    has(name: string): boolean;
    set(name: string, value: string): void;
    forEach(callback: (value: string, name: string) => void): void;
}
/** This Fetch API interface represents a resource request. */
interface FetchRequest extends FetchBody {
    /**
     * Returns a Headers object consisting of the headers associated with request. Note that headers added in the network layer by the user agent will not be accounted for in this object, e.g., the "Host" header.
     */
    readonly headers: FetchHeaders;
    /**
     * Returns a boolean indicating whether or not request can outlive the global in which it was created.
     */
    readonly keepalive: boolean;
    /**
     * Returns request's HTTP method, which is "GET" by default.
     */
    readonly method: string;
    /**
     * Returns the signal associated with request, which is an AbortSignal object indicating whether or not request has been aborted, and its abort event handler.
     */
    readonly signal: AbortSignal;
    /**
     * Returns the URL of request as a string.
     */
    readonly url: string;
    clone(): FetchRequest;
}
interface FetchResponse extends FetchBody {
    readonly headers: FetchHeaders;
    readonly ok: boolean;
    readonly redirected: boolean;
    readonly status: number;
    readonly statusText: string;
    readonly type: unknown;
    readonly url: string;
    clone(): FetchResponse;
}
interface FetchRequestInit<RequestBody = unknown> {
    /**
     * A BodyInit object or null to set request's body.
     */
    body?: RequestBody;
    /**
     * A string indicating whether credentials will be sent with the request always, never, or only when sent to a same-origin URL. Sets request's credentials.
     */
    credentials?: FetchRequestCredentials;
    /**
     * A Headers object, an object literal, or an array of two-item arrays to set request's headers.
     */
    headers?: FetchHeadersInit;
    /**
     * A cryptographic hash of the resource to be fetched by request. Sets request's integrity.
     */
    integrity?: string;
    /**
     * A boolean to set request's keepalive.
     */
    keepalive?: boolean;
    /**
     * A string to set request's method.
     */
    method?: string;
    /**
     * A string to indicate whether the request will use CORS, or will be restricted to same-origin URLs. Sets request's mode.
     */
    mode?: FetchRequestMode;
    /**
     * An AbortSignal to set request's signal.
     */
    signal?: AbortSignal | null;
}
declare type Fetch = (input: FetchRequestInfo, init?: FetchRequestInit) => Promise<FetchResponse>;

declare class DefaultNetworkTransport implements NetworkTransport {
    static fetch: Fetch;
    static AbortController: AbortController;
    static extraFetchOptions: Record<string, unknown> | undefined;
    static DEFAULT_HEADERS: {
        "Content-Type": string;
    };
    constructor();
    fetchWithCallbacks<RequestBody = unknown>(request: Request<RequestBody>, handler: ResponseHandler): void;
    fetch<RequestBody = unknown>(request: Request<RequestBody>): Promise<FetchResponse>;
    private createTimeoutSignal;
}

export { AbortController, CallbackResponse, DefaultNetworkTransport, ErrorCallback, Fetch, FetchHeaders, FetchRequest, FetchResponse, Headers, Method, NetworkTransport, Request, ResponseHandler, SuccessCallback };
