package com.dozingcatsoftware.dodge.model;

/** Represents an object moving on a Field. A Sprite contains a current position, target position, speed,
 * and color. The default implementation moves in a straight line toward the target position.
 * @author brian
 *
 */

public class Sprite {
	
	Vec2 position;
	Vec2 targetPosition;
	double speed;
	int[] color;
	
	/** Returns the distance between the current position and target position.
	 */
	public double distanceToTarget() {
		return position.distanceTo(targetPosition);
	}
	
	/** Returns the squared distance between the current position and target position. This can be
	 * used to avoid square root operations.
	 */
	public double squaredDistanceToTarget() {
		return position.squaredDistanceTo(targetPosition);
	}

	/** Updates the position by moving closer to the target position.
	 * @param dt time interval since the last update
	 */
	public void tick(double dt) {
		if (position==null || targetPosition==null) return;
		double dist = speed * dt;
		double distToTarget = this.distanceToTarget();
		if (dist>=distToTarget) {
			this.position = this.targetPosition;
		}
		else {
			// "correct" way treating position as immutable object
			/*
			Vec2 dir = this.targetPosition.subtract(this.position).normalize();
			this.position = this.position.add(dir.multiply(dist));
			*/
			// fast way updating position fields and not creating new objects
			double xnorm = (targetPosition.x - position.x) / distToTarget;
			double ynorm = (targetPosition.y - position.y) / distToTarget;
			position.x += dist*xnorm;
			position.y += dist*ynorm;
		}
	}
	
	/** Clamps the x and y positions to minimum and maximum values. Subclasses that want to stay inside
	 * specific ranges can call this method.
	 */
	public void enforceBounds(double xmin, double ymin, double xmax, double ymax) {
		if (position.x < xmin) position.x = xmin;
		if (position.x > xmax) position.x = xmax;
		if (position.y < ymin) position.y = ymin;
		if (position.y > ymax) position.y = ymax;
	}
	
	
	// autogenerated
	public Vec2 getPosition() {
		return position;
	}
	public void setPosition(Vec2 position) {
		this.position = position;
	}
	public Vec2 getTargetPosition() {
		return targetPosition;
	}
	public void setTargetPosition(Vec2 targetPosition) {
		this.targetPosition = targetPosition;
	}
	public double getSpeed() {
		return speed;
	}
	public void setSpeed(double speed) {
		this.speed = speed;
	}
	public int[] getColor() {
		return color;
	}
	public void setColor(int[] color) {
		this.color = color;
	}
}
