package com.dozingcatsoftware.dodge;

import java.io.File;
import java.util.HashMap;
import java.util.Map;

import com.dozingcatsoftware.dodge.model.Field;
import com.dozingcatsoftware.dodge.model.LevelManager;

import android.app.Activity;
import android.content.Context;
import android.content.Intent;
import android.content.SharedPreferences;
import android.graphics.Bitmap;
import android.net.Uri;
import android.os.Bundle;
import android.os.Handler;
import android.os.Message;
import android.preference.PreferenceManager;
import android.view.Display;
import android.view.KeyEvent;
import android.view.Menu;
import android.view.MenuItem;
import android.view.View;
import android.view.Window;
import android.view.WindowManager;
import android.widget.Button;
import android.widget.TextView;

public class DodgeMain extends Activity implements Field.Delegate {
	
	Field field;
	LevelManager levelManager;
	
	FieldView fieldView;
	View menuView;
	View pausedMenuView;
	TextView levelText;
	TextView livesText;
	TextView statusText;
	TextView bestLevelText;
	TextView bestFreePlayLevelText;
	Button continueFreePlayButton;
	View bestFreePlayLevelView;
	View bestLevelView;
	MenuItem endGameMenuItem;
	MenuItem selectBackgroundImageMenuItem;
	MenuItem preferencesMenuItem;
	
	private static final int ACTIVITY_PREFERENCES = 1;
	
	Handler messageHandler;
		
	int lives = 9;
	
    /** Called when the activity is first created. */
    @Override
    public void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        requestWindowFeature(Window.FEATURE_NO_TITLE);
        setContentView(R.layout.main);
        
        messageHandler = new Handler() {
            public void handleMessage(Message m) {
                processMessage(m);
            }
        };
        
        levelManager = new LevelManager();
        
        field = new Field();
        field.setDelegate(this);
        field.setLevelManager(levelManager);
        field.setMaxBullets(levelManager.numberOfBulletsForCurrentLevel());
        
        levelText = (TextView)findViewById(R.id.levelText);
        livesText = (TextView)findViewById(R.id.livesText);
        statusText = (TextView)findViewById(R.id.statusText);
        
        // uncomment to clear high scores
        /*
        setBestLevel(true, 0);
        setBestLevel(false, 0);
        */
        
        bestLevelText = (TextView)findViewById(R.id.bestLevelText);
        bestFreePlayLevelText = (TextView)findViewById(R.id.bestFreePlayLevelText);
        continueFreePlayButton = (Button)findViewById(R.id.continueFreePlayButton);
        bestLevelView = findViewById(R.id.bestLevelView);
        bestFreePlayLevelView = findViewById(R.id.bestFreePlayLevelView);
        
        Button newGameButton = (Button)findViewById(R.id.newGameButton);
        newGameButton.setOnClickListener(new View.OnClickListener() {
            public void onClick(View v) {
                doNewGame();
            }
        });
        
        Button freePlayButton = (Button)findViewById(R.id.freePlayButton);
        freePlayButton.setOnClickListener(new View.OnClickListener() {
            public void onClick(View v) {
                doFreePlay(1);
            }
        });
        
        continueFreePlayButton.setOnClickListener(new View.OnClickListener() {
            public void onClick(View v) {
                doFreePlay(bestLevel(true));
            }
        });

        
        Button aboutButton = (Button)findViewById(R.id.aboutButton);
        aboutButton.setOnClickListener(new View.OnClickListener() {
            public void onClick(View v) {
                doAbout();
            }
        });

        Button preferencesButton = findViewById(R.id.preferencesButton);
        preferencesButton.setOnClickListener(new View.OnClickListener() {
        	public void onClick(View v) {
        		gotoPreferences();
			}
		});

        Button continueGameButton = findViewById(R.id.continueGameButton);
        continueGameButton.setOnClickListener(new View.OnClickListener() {
			public void onClick(View v) {
				resumeGame();
			}
		});

		Button endGameButton = findViewById(R.id.endGameButton);
		endGameButton.setOnClickListener(new View.OnClickListener() {
			public void onClick(View v) {
				doGameOver();
			}
		});

		menuView = findViewById(R.id.menuView);
        pausedMenuView = findViewById(R.id.pausedMenuView);
        showMenu();

        fieldView = (FieldView)findViewById(R.id.fieldView);
        fieldView.setField(field);
        fieldView.setMessageHandler(messageHandler);
        
        updateFromPreferences();        
    }
    
    @Override
    public void onPause() {
    	super.onPause();
    	fieldView.stop();
    }
    
    @Override
    public void onResume() {
    	super.onResume();
    	if (isGameInProgress()) {
    		pauseGame();
            // We can't draw immediately because the FieldView won't be able to lock its canvas.
            // Drawing after a short delay seems to be fine.
            messageHandler.postDelayed(new Runnable() {
                public void run() {
                    fieldView.drawField();
                }
            }, 50);
		}
    	else {
			fieldView.start();
		}
    }

	@Override public boolean onKeyDown(int keyCode, KeyEvent event) {
		// When a game is in progress, pause rather than exit when the back button is pressed.
		// This prevents accidentally quitting the game.
		if (keyCode == KeyEvent.KEYCODE_BACK) {
			if (isGameInProgress() && pausedMenuView.getVisibility() != View.VISIBLE) {
				pauseGame();
				return true;
			}
		}
		return super.onKeyDown(keyCode, event);
	}

    /** Called when preferences activity completes, updates background image and bullet flash settings.
     */
    void updateFromPreferences() {
    	SharedPreferences prefs = PreferenceManager.getDefaultSharedPreferences(getBaseContext());
    	boolean flash = prefs.getBoolean(DodgePreferences.FLASHING_COLORS_KEY, false);
    	fieldView.setFlashingBullets(flash);
    	
    	boolean tilt = prefs.getBoolean(DodgePreferences.TILT_CONTROL_KEY, true);
    	fieldView.setTiltControlEnabled(tilt);
    	
    	boolean showFPS = prefs.getBoolean(DodgePreferences.SHOW_FPS_KEY, false);
    	fieldView.setShowFPS(showFPS);

    	Bitmap backgroundBitmap = null;
    	if (prefs.getBoolean(DodgePreferences.USE_BACKGROUND_KEY,false)) {
            try {
                File backgroundImageFile = getFileStreamPath(DodgePreferences.BACKGROUND_IMAGE_FILENAME);
                if (backgroundImageFile.isFile()) {
                    // Load image scaled to the screen size (the FieldView size would be better but it's not available in onCreate)
                    WindowManager windowManager = (WindowManager)this.getSystemService(Context.WINDOW_SERVICE);
                    Display display = windowManager.getDefaultDisplay();

                    backgroundBitmap = AndroidUtils.scaledBitmapFromFileWithMinimumSize(backgroundImageFile, display.getWidth(), display.getHeight());
                }
            }
            catch(Throwable ex) {
                // we shouldn't get out of memory errors, but it's possible with weird images
                android.util.Log.i("DodgeMain", "Failed to read background image", ex);
                backgroundBitmap = null;
            }
    	}
		fieldView.setBackgroundBitmap(backgroundBitmap);
    }

    private void gotoPreferences() {
		Intent settingsActivity = new Intent(getBaseContext(), DodgePreferences.class);
		startActivityForResult(settingsActivity, ACTIVITY_PREFERENCES);
	}
    
    @Override
    public boolean onCreateOptionsMenu(Menu menu) {
    	endGameMenuItem = menu.add(R.string.end_game);
    	preferencesMenuItem = menu.add(R.string.preferences);
    	return true;
    }
    
    public boolean onOptionsItemSelected(MenuItem item) {
    	if (item==endGameMenuItem) {
    		doGameOver();
    	}
    	else if (item==preferencesMenuItem) {
    		gotoPreferences();
    	}
    	return true;
    }
    
    protected void onActivityResult(int requestCode, int resultCode, Intent intent) { 
        super.onActivityResult(requestCode, resultCode, intent); 

        switch(requestCode) { 
            case ACTIVITY_PREFERENCES:
            	updateFromPreferences();
            	break;
        }
    }
    
    boolean inFreePlay() {
    	return (lives<0);
    }
    
    // methods to store and retrieve the highest normal and free play levels reached, using SharedPreferences
    int bestLevel(boolean freePlay) {
    	String key = (freePlay) ? "BestFreePlayLevel" : "BestLevel";
    	return getPreferences(MODE_PRIVATE).getInt(key, 0);
    }
    void setBestLevel(boolean freePlay, int val) {
       	String key = (freePlay) ? "BestFreePlayLevel" : "BestLevel";
       	SharedPreferences.Editor editor = getPreferences(MODE_PRIVATE).edit();
       	editor.putInt(key, val);
       	editor.commit();
    }
    
    void recordCurrentLevel() {
    	if (levelManager.getCurrentLevel() > bestLevel(inFreePlay())) {
    		setBestLevel(inFreePlay(), levelManager.getCurrentLevel());
    	}
    }

    void updateBestLevelFields() {
    	int bestNormal = bestLevel(false);
    	bestLevelText.setText((bestNormal>1) ? String.valueOf(bestNormal) : getString(R.string.score_none));
    	
    	int bestFree = bestLevel(true);
    	bestFreePlayLevelText.setText((bestFree>1) ? String.valueOf(bestFree) : getString(R.string.score_none));
		continueFreePlayButton.setEnabled(bestFree>1);
    	menuView.forceLayout();
    }
    

    void processMessage(Message m) {
    	String action = m.getData().getString("event");
    	if ("goal".equals(action)) {
    		levelManager.setCurrentLevel(1 + levelManager.getCurrentLevel());
    		recordCurrentLevel();
        	synchronized(field) {
        		field.setMaxBullets(levelManager.numberOfBulletsForCurrentLevel());
        	}
        	updateScore();
    	}
    	else if ("death".equals(action)) {
    		if (lives>0) lives--;
    		synchronized(field) {
        		if (lives==0) {
        			field.removeDodger();
        			doGameOver();
        		}
        		else {
        			fieldView.startDeathAnimation(field.getDodger().getPosition());
        			field.createDodger();
        		}
    		}
    		updateScore();
    	}
    }

    boolean isGameInProgress() {
    	return field.getDodger() != null;
	}

	void pauseGame() {
    	fieldView.stop();
    	showMenu();
	}

	void resumeGame() {
    	fieldView.start();
    	hideMenu();
	}

    void showMenu() {
    	if (isGameInProgress()) {
    		field.stop();
    		menuView.setVisibility(View.GONE);
    		pausedMenuView.setVisibility(View.VISIBLE);
    		pausedMenuView.requestFocus();
		}
    	else {
			updateBestLevelFields();
			menuView.setVisibility(View.VISIBLE);
			pausedMenuView.setVisibility(View.GONE);
			menuView.requestFocus();
		}
	}

	void hideMenu() {
		menuView.setVisibility(View.GONE);
		pausedMenuView.setVisibility(View.GONE);
	}
    
    void updateScore() {
		levelText.setText(getString(R.string.level_prefix) + " " + levelManager.getCurrentLevel());
		livesText.setText(getString(R.string.lives_prefix) + " " + ((lives>=0) ? ""+lives : getString(R.string.free_play_lives)));
    }
    
    void doGameOver() {
    	field.removeDodger();
    	fieldView.start();
    	statusText.setText(getString(R.string.game_over_message));
    	showMenu();
    }
    
    void startGameAtLevelWithLives(int startLevel, int numLives) {
    	levelManager.setCurrentLevel(startLevel);
    	field.setMaxBullets(levelManager.numberOfBulletsForCurrentLevel());
    	field.createDodger();
    	menuView.setVisibility(View.INVISIBLE);
    	lives = numLives;
    	updateScore();
    }
    
    void doNewGame() {
    	startGameAtLevelWithLives(1, 9);
    }

    void doFreePlay(int startLevel) {
    	startGameAtLevelWithLives(startLevel, -1);
    }
    
    void doAbout() {
    	Intent aboutIntent = new Intent(this, DodgeAbout.class);
    	this.startActivity(aboutIntent);
    }

    void sendMessage(Map params) {
    	Bundle b = new Bundle();
    	for(Object key : params.keySet()) {
    		b.putString((String)key, (String)params.get(key));
    	}
    	Message m = messageHandler.obtainMessage();
    	m.setData(b);
    	messageHandler.sendMessage(m);
    }
    
    // Field.Delegate methods
    // these occur in a separate thread, so use Handler
	public void dodgerHitByBullet(Field theField) {
		Map params = new HashMap();
		params.put("event", "death");
		sendMessage(params);
	}

	public void dodgerReachedGoal(Field theField) {
		Map params = new HashMap();
		params.put("event", "goal");
		sendMessage(params);
	}
}