/*
 *  aes_key_wrap_test.cpp
 *
 *  Copyright (C) 2024-2025
 *  Terrapane Corporation
 *  All Rights Reserved
 *
 *  Author:
 *      Paul E. Jones <paulej@packetizer.com>
 *
 *  Description:
 *      This module will exercise the AES key wrap (RFC 3394) and
 *      AES key wrap with padding (RFC 5649) logic.
 *
 *  Portability Issues:
 *      None.
 */

#include <cstdint>
#include <terra/crypto/cipher/aes_key_wrap.h>
#include <terra/stf/stf.h>

using namespace Terra::Crypto::Cipher;

// Test vector in RFC 3394 Section 4.1
STF_TEST(AESKeyWrap, RFC3394_4_1)
{
    const std::uint8_t key[] =
    {
        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
        0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f
    };
    const std::uint8_t plaintext[] =
    {
        0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77,
        0x88, 0x99, 0xaa, 0xbb, 0xcc, 0xdd, 0xee, 0xff
    };
    const std::uint8_t expected_ciphertext[] =
    {
        0x1f, 0xa6, 0x8b, 0x0a, 0x81, 0x12, 0xb4, 0x47,
        0xae, 0xf3, 0x4b, 0xd8, 0xfb, 0x5a, 0x7b, 0x82,
        0x9d, 0x3e, 0x86, 0x23, 0x71, 0xd2, 0xcf, 0xe5
    };
    std::uint8_t ciphertext[sizeof(expected_ciphertext)];
    std::uint8_t plaintext_check[sizeof(plaintext)];

    // Create AESKeyWrap object using the given key and length
    AESKeyWrap aes_kw(key);

    // Perform the AES Key Wrap
    aes_kw.Wrap(plaintext, ciphertext);

    // Verify result
    STF_ASSERT_MEM_EQ(expected_ciphertext,
                      ciphertext,
                      sizeof(expected_ciphertext));

    // Perform the key unwrap operation
    STF_ASSERT_TRUE(aes_kw.Unwrap(ciphertext, plaintext_check));

    // Verify result
    STF_ASSERT_MEM_EQ(plaintext, plaintext_check, sizeof(plaintext));
}

// Test vector in RFC 3394 Section 4.2
STF_TEST(AESKeyWrap, RFC3394_4_2)
{
    const std::uint8_t key[] =
    {
        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
        0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
        0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17
    };
    const std::uint8_t plaintext[] =
    {
        0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77,
        0x88, 0x99, 0xaa, 0xbb, 0xcc, 0xdd, 0xee, 0xff
    };
    const std::uint8_t expected_ciphertext[] =
    {
        0x96, 0x77, 0x8b, 0x25, 0xae, 0x6c, 0xa4, 0x35,
        0xf9, 0x2b, 0x5b, 0x97, 0xc0, 0x50, 0xae, 0xd2,
        0x46, 0x8a, 0xb8, 0xa1, 0x7a, 0xd8, 0x4e, 0x5d
    };
    std::uint8_t ciphertext[sizeof(expected_ciphertext)];
    std::uint8_t plaintext_check[sizeof(plaintext)];

    // Create AESKeyWrap object using the given key and length
    AESKeyWrap aes_kw(key);

    // Perform the AES key wrap
    aes_kw.Wrap(plaintext, ciphertext);

    // Verify result
    STF_ASSERT_MEM_EQ(expected_ciphertext,
                      ciphertext,
                      sizeof(expected_ciphertext));

    // Perform the key unwrap operation
    STF_ASSERT_TRUE(aes_kw.Unwrap(ciphertext, plaintext_check));

    // Verify result
    STF_ASSERT_MEM_EQ(plaintext, plaintext_check, sizeof(plaintext));
}

// Test vector in RFC 3394 Section 4.3
STF_TEST(AESKeyWrap, RFC3394_4_3)
{
    const std::uint8_t key[] =
    {
        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
        0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
        0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
        0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f
    };
    const std::uint8_t plaintext[] =
    {
        0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77,
        0x88, 0x99, 0xaa, 0xbb, 0xcc, 0xdd, 0xee, 0xff
    };
    const std::uint8_t expected_ciphertext[] =
    {
        0x64, 0xe8, 0xc3, 0xf9, 0xce, 0x0f, 0x5b, 0xa2,
        0x63, 0xe9, 0x77, 0x79, 0x05, 0x81, 0x8a, 0x2a,
        0x93, 0xc8, 0x19, 0x1e, 0x7d, 0x6e, 0x8a, 0xe7
    };
    std::uint8_t ciphertext[sizeof(expected_ciphertext)];
    std::uint8_t plaintext_check[sizeof(plaintext)];

    // Create AESKeyWrap object using the given key and length
    AESKeyWrap aes_kw(key);

    // Perform the AES key wrap
    aes_kw.Wrap(plaintext, ciphertext);

    // Verify result
    STF_ASSERT_MEM_EQ(expected_ciphertext,
                      ciphertext,
                      sizeof(expected_ciphertext));

    // Perform the key unwrap operation
    STF_ASSERT_TRUE(aes_kw.Unwrap(ciphertext, plaintext_check));

    // Verify result
    STF_ASSERT_MEM_EQ(plaintext, plaintext_check, sizeof(plaintext));
}

// Test vector in RFC 3394 Section 4.4
STF_TEST(AESKeyWrap, RFC3394_4_4)
{
    const std::uint8_t key[] =
    {
        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
        0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
        0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17
    };
    const std::uint8_t plaintext[] =
    {
        0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77,
        0x88, 0x99, 0xaa, 0xbb, 0xcc, 0xdd, 0xee, 0xff,
        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07
    };
    const std::uint8_t expected_ciphertext[] =
    {
        0x03, 0x1d, 0x33, 0x26, 0x4e, 0x15, 0xd3, 0x32,
        0x68, 0xf2, 0x4e, 0xc2, 0x60, 0x74, 0x3e, 0xdc,
        0xe1, 0xc6, 0xc7, 0xdd, 0xee, 0x72, 0x5a, 0x93,
        0x6b, 0xa8, 0x14, 0x91, 0x5c, 0x67, 0x62, 0xd2
    };
    std::uint8_t ciphertext[sizeof(expected_ciphertext)];
    std::uint8_t plaintext_check[sizeof(plaintext)];

    // Create AESKeyWrap object using the given key and length
    AESKeyWrap aes_kw(key);

    // Perform the AES key wrap
    aes_kw.Wrap(plaintext, ciphertext);

    // Verify result
    STF_ASSERT_MEM_EQ(expected_ciphertext,
                      ciphertext,
                      sizeof(expected_ciphertext));

    // Perform the key unwrap operation
    STF_ASSERT_TRUE(aes_kw.Unwrap(ciphertext, plaintext_check));

    // Verify result
    STF_ASSERT_MEM_EQ(plaintext, plaintext_check, sizeof(plaintext));
}

// Test vector in RFC 3394 Section 4.5
STF_TEST(AESKeyWrap, RFC3394_4_5)
{
    const std::uint8_t key[] =
    {
        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
        0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
        0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
        0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f
    };
    const std::uint8_t plaintext[] =
    {
        0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77,
        0x88, 0x99, 0xaa, 0xbb, 0xcc, 0xdd, 0xee, 0xff,
        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07
    };
    const std::uint8_t expected_ciphertext[] =
    {
        0xa8, 0xf9, 0xbc, 0x16, 0x12, 0xc6, 0x8b, 0x3f,
        0xf6, 0xe6, 0xf4, 0xfb, 0xe3, 0x0e, 0x71, 0xe4,
        0x76, 0x9c, 0x8b, 0x80, 0xa3, 0x2c, 0xb8, 0x95,
        0x8c, 0xd5, 0xd1, 0x7d, 0x6b, 0x25, 0x4d, 0xa1
    };
    std::uint8_t ciphertext[sizeof(expected_ciphertext)];
    std::uint8_t plaintext_check[sizeof(plaintext)];

    // Create AESKeyWrap object using the given key and length
    AESKeyWrap aes_kw(key);

    // Perform the AES key wrap
    aes_kw.Wrap(plaintext, ciphertext);

    // Verify result
    STF_ASSERT_MEM_EQ(expected_ciphertext,
                      ciphertext,
                      sizeof(expected_ciphertext));

    // Perform the key unwrap operation
    STF_ASSERT_TRUE(aes_kw.Unwrap(ciphertext, plaintext_check));

    // Verify result
    STF_ASSERT_MEM_EQ(plaintext, plaintext_check, sizeof(plaintext));
}

// Test vector in RFC 3394 Section 4.6
STF_TEST(AESKeyWrap, RFC3394_4_6)
{
    const std::uint8_t key[] =
    {
        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
        0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
        0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
        0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f
    };
    const std::uint8_t plaintext[] =
    {
        0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77,
        0x88, 0x99, 0xaa, 0xbb, 0xcc, 0xdd, 0xee, 0xff,
        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
        0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f
    };
    const std::uint8_t expected_ciphertext[] =
    {
        0x28, 0xc9, 0xf4, 0x04, 0xc4, 0xb8, 0x10, 0xf4,
        0xcb, 0xcc, 0xb3, 0x5c, 0xfb, 0x87, 0xf8, 0x26,
        0x3f, 0x57, 0x86, 0xe2, 0xd8, 0x0e, 0xd3, 0x26,
        0xcb, 0xc7, 0xf0, 0xe7, 0x1a, 0x99, 0xf4, 0x3b,
        0xfb, 0x98, 0x8b, 0x9b, 0x7a, 0x02, 0xdd, 0x21
    };
    std::uint8_t ciphertext[sizeof(expected_ciphertext)];
    std::uint8_t plaintext_check[sizeof(plaintext)];

    // Create AESKeyWrap object using the given key and length
    AESKeyWrap aes_kw(key);

    // Perform the AES key wrap
    aes_kw.Wrap(plaintext, ciphertext);

    // Verify result
    STF_ASSERT_MEM_EQ(expected_ciphertext,
                      ciphertext,
                      sizeof(expected_ciphertext));

    // Perform the key unwrap operation
    STF_ASSERT_TRUE(aes_kw.Unwrap(ciphertext, plaintext_check));

    // Verify result
    STF_ASSERT_MEM_EQ(plaintext, plaintext_check, sizeof(plaintext));
}

// Test vector in RFC 5649 Section 6, 20-octet wrapped key
STF_TEST(AESKeyWrapWithPadding, RFC5649_6_20)
{
    const std::uint8_t key[] =
    {
        0x58, 0x40, 0xdf, 0x6e, 0x29, 0xb0, 0x2a, 0xf1,
        0xab, 0x49, 0x3b, 0x70, 0x5b, 0xf1, 0x6e, 0xa1,
        0xae, 0x83, 0x38, 0xf4, 0xdc, 0xc1, 0x76, 0xa8
    };
    const std::uint8_t plaintext[] =
    {
        0xc3, 0x7b, 0x7e, 0x64, 0x92, 0x58, 0x43, 0x40,
        0xbe, 0xd1, 0x22, 0x07, 0x80, 0x89, 0x41, 0x15,
        0x50, 0x68, 0xf7, 0x38
    };
    const std::uint8_t expected_ciphertext[] =
    {
        0x13, 0x8b, 0xde, 0xaa, 0x9b, 0x8f, 0xa7, 0xfc,
        0x61, 0xf9, 0x77, 0x42, 0xe7, 0x22, 0x48, 0xee,
        0x5a, 0xe6, 0xae, 0x53, 0x60, 0xd1, 0xae, 0x6a,
        0x5f, 0x54, 0xf3, 0x73, 0xfa, 0x54, 0x3b, 0x6a
    };
    std::uint8_t ciphertext[sizeof(expected_ciphertext)];
    std::size_t ciphertext_length;
    std::uint8_t plaintext_check[sizeof(expected_ciphertext) - 8];
    std::size_t plaintext_check_length;

    // Create AESKeyWrap object using the given key and length
    AESKeyWrap aes_kw(key);

    // Perform the AES key wrap
    ciphertext_length = aes_kw.WrapWithPadding(plaintext, ciphertext);

    // Verify the expected ciphertext length
    STF_ASSERT_EQ(sizeof(expected_ciphertext), ciphertext_length);

    // Verify result
    STF_ASSERT_MEM_EQ(expected_ciphertext,
                      ciphertext,
                      sizeof(expected_ciphertext));

    // Perform the key unwrap operation
    plaintext_check_length = aes_kw.UnwrapWithPadding(ciphertext,
                                                      plaintext_check);

    // Verify the expected plaintext length
    STF_ASSERT_EQ(sizeof(plaintext), plaintext_check_length);

    // Verify result
    STF_ASSERT_MEM_EQ(plaintext, plaintext_check, sizeof(plaintext));
}

// Test vector in RFC 5649 Section 6, 7-octet wrapped key
STF_TEST(AESKeyWrapWithPadding, RFC5649_6_7)
{
    const std::uint8_t key[] =
    {
        0x58, 0x40, 0xdf, 0x6e, 0x29, 0xb0, 0x2a, 0xf1,
        0xab, 0x49, 0x3b, 0x70, 0x5b, 0xf1, 0x6e, 0xa1,
        0xae, 0x83, 0x38, 0xf4, 0xdc, 0xc1, 0x76, 0xa8
    };
    const std::uint8_t plaintext[] =
    {
        0x46, 0x6f, 0x72, 0x50, 0x61, 0x73, 0x69
    };
    const std::uint8_t expected_ciphertext[] =
    {
        0xaf, 0xbe, 0xb0, 0xf0, 0x7d, 0xfb, 0xf5, 0x41,
        0x92, 0x00, 0xf2, 0xcc, 0xb5, 0x0b, 0xb2, 0x4f
    };
    std::uint8_t ciphertext[sizeof(expected_ciphertext)];
    std::size_t ciphertext_length;
    std::uint8_t plaintext_check[sizeof(expected_ciphertext) - 8];
    std::size_t plaintext_check_length;

    // Create AESKeyWrap object using the given key and length
    AESKeyWrap aes_kw(key);

    // Perform the AES key wrap
    ciphertext_length = aes_kw.WrapWithPadding(plaintext, ciphertext);

    // Verify the expected ciphertext length
    STF_ASSERT_EQ(sizeof(expected_ciphertext), ciphertext_length);

    // Verify result
    STF_ASSERT_MEM_EQ(expected_ciphertext,
                      ciphertext,
                      sizeof(expected_ciphertext));

    // Perform the key unwrap operation
    plaintext_check_length = aes_kw.UnwrapWithPadding(ciphertext,
                                                      plaintext_check);

    // Verify the expected plaintext length
    STF_ASSERT_EQ(sizeof(plaintext), plaintext_check_length);

    // Verify result
    STF_ASSERT_MEM_EQ(plaintext, plaintext_check, sizeof(plaintext));
}

// Test vector for RFC 5649, Known Test Vector using 192-bit encryption key
STF_TEST(AESKeyWrapWithPadding, Extra)
{
    const std::uint8_t key[] =
    {
        0x58, 0x40, 0xdf, 0x6e, 0x29, 0xb0, 0x2a, 0xf1,
        0xab, 0x49, 0x3b, 0x70, 0x5b, 0xf1, 0x6e, 0xa1,
        0xae, 0x83, 0x38, 0xf4, 0xdc, 0xc1, 0x76, 0xa8
    };
    const std::uint8_t plaintext[] =
    {
        0xc3, 0x7b, 0x7e, 0x64, 0x92, 0x58, 0x43, 0x40,
        0xbe, 0xd1, 0x22, 0x07, 0x80, 0x89, 0x41, 0x15,
        0xc3, 0x7b, 0x7e, 0x64, 0x92, 0x58, 0x43, 0x40,
        0xc3, 0x7b, 0x7e, 0x64, 0x92, 0x58, 0x43, 0x40,
        0xc3, 0x7b, 0x7e, 0x64, 0x92, 0x58, 0x43, 0x40,
        0x13, 0x8b, 0xde, 0xaa, 0x9b, 0x8f, 0xa7, 0xfc,
        0x61, 0xf9, 0x77, 0x42, 0xe7, 0x22, 0x48, 0xee,
        0x13, 0x8b, 0xde, 0xaa, 0x9b, 0x8f, 0xa7, 0xfc,
        0x61, 0xf9, 0x77, 0x42, 0xe7, 0x22, 0x48, 0xee,
        0x5a, 0xe6, 0xae, 0x53, 0x60, 0xd1, 0xae, 0x6a,
        0x5a, 0xe6, 0xae, 0x53, 0x60, 0xd1, 0xae, 0x6a,
        0xc3, 0x7b, 0x7e, 0x64, 0x92, 0x58, 0x43, 0x40,
        0x13, 0x8b, 0xde, 0xaa, 0x9b, 0x8f, 0xa7, 0xfc,
        0x61, 0xf9, 0x77, 0x42, 0xe7, 0x22, 0x48, 0xee,
        0x5a, 0xe6, 0xae, 0x53, 0x60, 0xd1, 0xae, 0x6a,
        0x13, 0x8b, 0xde, 0xaa, 0x9b, 0x8f, 0xa7, 0xfc,
        0x61, 0xf9, 0x77, 0x42, 0xe7, 0x22, 0x48, 0xee,
        0x5a, 0xe6, 0xae, 0x53, 0x60, 0xd1, 0xae, 0x6a,
        0xc3, 0x7b, 0x7e, 0x64, 0x92, 0x58, 0x43, 0x40,
        0x13, 0x8b, 0xde, 0xaa, 0x9b, 0x8f, 0xa7, 0xfc,
        0x61, 0xf9, 0x77, 0x42, 0xe7, 0x22, 0x48, 0xee,
        0x5a, 0xe6, 0xae, 0x53, 0x60, 0xd1, 0xae, 0x6a,
        0xc3, 0x7b, 0x7e, 0x64, 0x92, 0x58, 0x43, 0x40,
        0xbe, 0xd1, 0x22, 0x07, 0x80, 0x89, 0x41, 0x15,
        0xbe, 0xd1, 0x22, 0x07, 0x80, 0x89, 0x41, 0x15,
        0x13, 0x8b, 0xde, 0xaa, 0x9b, 0x8f, 0xa7, 0xfc,
        0x61, 0xf9, 0x77, 0x42, 0xe7, 0x22, 0x48, 0xee,
        0x5a, 0xe6, 0xae, 0x53, 0x60, 0xd1, 0xae, 0x6a,
        0xbe, 0xd1, 0x22, 0x07, 0x80, 0x89, 0x41, 0x15,
        0xbe, 0xd1, 0x22, 0x07, 0x80, 0x89, 0x41, 0x15,
        0xbe, 0xd1, 0x22, 0x07, 0x80, 0x89, 0x41, 0x15,
        0x13, 0x8b, 0xde, 0xaa, 0x9b, 0x8f, 0xa7, 0xfc,
        0x61, 0xf9, 0x77, 0x42, 0xe7, 0x22, 0x48, 0xee,
        0x5a, 0xe6, 0xae, 0x53, 0x60, 0xd1, 0xae, 0x6a,
        0xbe, 0xd1, 0x22, 0x07, 0x80, 0x89, 0x41, 0x15,
        0x50, 0x68, 0xf7
    };
    const std::uint8_t expected_ciphertext[] =
    {
        0xca, 0xbf, 0xf3, 0x9e, 0xec, 0x5f, 0xf3, 0x2e,
        0x4d, 0xc4, 0x6e, 0x81, 0xde, 0xaa, 0x98, 0xe2,
        0x31, 0xb8, 0x0a, 0xb9, 0xa2, 0x27, 0x17, 0x0b,
        0x92, 0x08, 0xad, 0x85, 0x7a, 0x05, 0x43, 0x8d,
        0x09, 0x5b, 0x5b, 0x16, 0xa1, 0x55, 0x5a, 0xf0,
        0xc6, 0x09, 0xe7, 0x3e, 0x37, 0x42, 0x0b, 0x97,
        0x3c, 0x76, 0xa0, 0x3c, 0xaa, 0xa4, 0xde, 0x6e,
        0xad, 0x63, 0xa0, 0xe2, 0x48, 0xf5, 0xdc, 0x0d,
        0x05, 0x5e, 0x5b, 0x3b, 0xd5, 0x48, 0xcd, 0x3c,
        0xe8, 0xf2, 0xbf, 0x02, 0xd9, 0xed, 0x45, 0x45,
        0x1a, 0xb6, 0xfb, 0x9b, 0x5a, 0xf7, 0x0d, 0xd4,
        0x4c, 0x39, 0x20, 0x14, 0x37, 0x65, 0xd8, 0x1d,
        0x18, 0x53, 0xed, 0x41, 0xe2, 0x88, 0xc6, 0x43,
        0x41, 0x61, 0x19, 0xcd, 0x58, 0xfb, 0x58, 0x0c,
        0x31, 0x26, 0x1a, 0x1f, 0xbd, 0x39, 0xe9, 0x5b,
        0x38, 0xe5, 0xe3, 0x65, 0x30, 0x5d, 0xcb, 0x64,
        0xa4, 0x5f, 0x60, 0x85, 0x9b, 0xa4, 0x65, 0x30,
        0x70, 0xfa, 0x6e, 0xdb, 0x7a, 0x8a, 0x81, 0xdc,
        0xbd, 0x02, 0xae, 0x4b, 0x44, 0x9b, 0xd4, 0x42,
        0xe5, 0x6b, 0xb3, 0x89, 0xb4, 0x3c, 0x66, 0x30,
        0xbd, 0x5a, 0xad, 0x92, 0x28, 0xae, 0x75, 0x51,
        0x66, 0x95, 0x27, 0x6f, 0x1b, 0x60, 0x42, 0xed,
        0x2d, 0x3e, 0x29, 0xbf, 0x2f, 0x83, 0x66, 0x64,
        0x0e, 0xc8, 0xa0, 0xac, 0x67, 0x74, 0x08, 0x8b,
        0x9b, 0xfa, 0xb6, 0xb5, 0x89, 0x5c, 0x20, 0x04,
        0x07, 0x00, 0xd9, 0x88, 0x14, 0x8c, 0xad, 0x33,
        0x3e, 0x29, 0x58, 0x15, 0xab, 0x77, 0x0f, 0xde,
        0x26, 0x46, 0x70, 0x0b, 0x22, 0xba, 0xff, 0xee,
        0xfd, 0x34, 0x7c, 0xcb, 0xec, 0xf3, 0x38, 0x03,
        0xeb, 0x10, 0xa6, 0xf4, 0x06, 0x62, 0xa2, 0x72,
        0xbe, 0x14, 0xd8, 0x52, 0x1e, 0xdf, 0xce, 0x43,
        0xd6, 0x00, 0x36, 0x51, 0x3f, 0xf0, 0xaf, 0x36,
        0x60, 0x61, 0x35, 0xb6, 0x5b, 0xb8, 0xa2, 0x8f,
        0xa5, 0x89, 0x8e, 0x08, 0x17, 0x61, 0x0e, 0x22,
        0xee, 0xa8, 0xd6, 0xd0, 0x15, 0x08, 0xc3, 0x60,
        0x78, 0x27, 0x80, 0x3b, 0x6b, 0x44, 0xa5, 0x1f,
        0x24, 0x8b, 0xd5, 0x24, 0x61, 0xc4, 0xf8, 0xdf
    };
    std::uint8_t ciphertext[sizeof(expected_ciphertext)];
    std::size_t ciphertext_length;
    std::uint8_t plaintext_check[sizeof(expected_ciphertext) - 8];
    std::size_t plaintext_check_length;

    // Create AESKeyWrap object using the given key and length
    AESKeyWrap aes_kw(key);

    // Perform the AES key wrap
    ciphertext_length = aes_kw.WrapWithPadding(plaintext, ciphertext);

    // Verify the expected ciphertext length
    STF_ASSERT_EQ(sizeof(expected_ciphertext), ciphertext_length);

    // Verify result
    STF_ASSERT_MEM_EQ(expected_ciphertext,
                      ciphertext,
                      sizeof(expected_ciphertext));

    // Perform the key unwrap operation
    plaintext_check_length = aes_kw.UnwrapWithPadding(ciphertext,
                                                      plaintext_check);

    // Verify the expected plaintext length
    STF_ASSERT_EQ(sizeof(plaintext), plaintext_check_length);

    // Verify result
    STF_ASSERT_MEM_EQ(plaintext, plaintext_check, sizeof(plaintext));
}

// Test vector for RFC 5649; single octet plaintext
STF_TEST(AESKeyWrapWithPadding, SingleOctetPlaintext)
{
    const std::uint8_t key[] =
    {
        0x25, 0x40, 0xdf, 0x6e, 0x29, 0xb0, 0x2a, 0xf1,
        0xab, 0x49, 0x3b, 0x70, 0x5b, 0xf1, 0x6e, 0xa1,
        0xae, 0x83, 0x38, 0xf4, 0xdc, 0xc1, 0x76, 0xa8
    };
    const std::uint8_t plaintext[] = { 0xc3 };
    const std::uint8_t expected_ciphertext[] =
    {
        0xb3, 0x0f, 0x2e, 0x31, 0x86, 0xf6, 0x18, 0x54,
        0x7e, 0x9a, 0x17, 0xdc, 0x8a, 0xa6, 0xcd, 0x3a
    };
    std::uint8_t ciphertext[sizeof(plaintext) + 16];
    std::size_t ciphertext_length;
    std::uint8_t plaintext_check[sizeof(ciphertext) - 8];
    std::size_t plaintext_check_length;

    // Create AESKeyWrap object using the given key and length
    AESKeyWrap aes_kw(key);

    // Perform the AES key wrap
    ciphertext_length = aes_kw.WrapWithPadding(plaintext, ciphertext);

    // Verify the expected ciphertext length
    STF_ASSERT_EQ(sizeof(expected_ciphertext), ciphertext_length);

    // Verify result
    STF_ASSERT_MEM_EQ(expected_ciphertext,
                      ciphertext,
                      sizeof(expected_ciphertext));

    // Perform the key unwrap operation
    plaintext_check_length =
        aes_kw.UnwrapWithPadding({ciphertext, ciphertext_length},
                                 plaintext_check);

    // Verify the expected plaintext length
    STF_ASSERT_EQ(sizeof(plaintext), plaintext_check_length);

    // Verify result
    STF_ASSERT_MEM_EQ(plaintext, plaintext_check, sizeof(plaintext));
}

// Test vector for RFC 5649; Eight octets of plaintext
STF_TEST(AESKeyWrapWithPadding, EightOctetPlaintext)
{
    const std::uint8_t key[] =
    {
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
    };
    const std::uint8_t plaintext[] =
    {
        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07
    };
    const std::uint8_t expected_ciphertext[] =
    {
        0x01, 0xff, 0x4c, 0x43, 0x0c, 0xdf, 0x3d, 0x2d,
        0x81, 0x5b, 0x09, 0x72, 0xb2, 0x3d, 0x7c, 0x35
    };
    std::uint8_t ciphertext[sizeof(plaintext) + 16];
    std::size_t ciphertext_length;
    std::uint8_t plaintext_check[sizeof(ciphertext) - 8];
    std::size_t plaintext_check_length;

    // Create AESKeyWrap object using the given key and length
    AESKeyWrap aes_kw(key);

    // Perform the AES key wrap
    ciphertext_length = aes_kw.WrapWithPadding(plaintext, ciphertext);

    // Verify the expected ciphertext length
    STF_ASSERT_EQ(sizeof(expected_ciphertext), ciphertext_length);

    // Verify result
    STF_ASSERT_MEM_EQ(expected_ciphertext,
                      ciphertext,
                      sizeof(expected_ciphertext));

    // Perform the key unwrap operation
    plaintext_check_length =
        aes_kw.UnwrapWithPadding({ciphertext, ciphertext_length},
                                 plaintext_check);

    // Verify the expected plaintext length
    STF_ASSERT_EQ(sizeof(plaintext), plaintext_check_length);

    // Verify result
    STF_ASSERT_MEM_EQ(plaintext, plaintext_check, sizeof(plaintext));
}

// Test vector for RFC 5649; fifteen octets of plaintext
STF_TEST(AESKeyWrapWithPadding, FifteenOctetPlaintext)
{
    const std::uint8_t key[] =
    {
        0x25, 0x40, 0xdf, 0x6e, 0x29, 0xb0, 0x2a, 0xf1,
        0xab, 0x49, 0x3b, 0x70, 0x5b, 0xf1, 0x6e, 0xa1,
        0xae, 0x83, 0x38, 0xf4, 0xdc, 0xc1, 0x76, 0xa8
    };
    const std::uint8_t plaintext[] = {
        0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77,
        0x88, 0x99, 0xaa, 0xbb, 0xcc, 0xdd, 0xee
    };
    const std::uint8_t expected_ciphertext[] =
    {
        0x5e, 0xba, 0x5a, 0xea, 0xe8, 0x3d, 0xb9, 0x9d,
        0x0d, 0x76, 0x8c, 0x7a, 0x3b, 0xf2, 0x52, 0x86,
        0xc7, 0xc4, 0x22, 0x6a, 0x27, 0xa0, 0x9d, 0x2f
    };
    std::uint8_t ciphertext[sizeof(plaintext) + 16];
    std::size_t ciphertext_length;
    std::uint8_t plaintext_check[sizeof(ciphertext) - 8];
    std::size_t plaintext_check_length;

    // Create AESKeyWrap object using the given key and length
    AESKeyWrap aes_kw(key);

    // Perform the AES key wrap
    ciphertext_length = aes_kw.WrapWithPadding(plaintext, ciphertext);

    // Verify the expected ciphertext length
    STF_ASSERT_EQ(sizeof(expected_ciphertext), ciphertext_length);

    // Verify result
    STF_ASSERT_MEM_EQ(expected_ciphertext,
                      ciphertext,
                      sizeof(expected_ciphertext));

    // Perform the key unwrap operation
    plaintext_check_length =
        aes_kw.UnwrapWithPadding({ciphertext, ciphertext_length},
                                 plaintext_check);

    // Verify the expected plaintext length
    STF_ASSERT_EQ(sizeof(plaintext), plaintext_check_length);

    // Verify result
    STF_ASSERT_MEM_EQ(plaintext, plaintext_check, sizeof(plaintext));
}

// Test vector for RFC 5649; Sixteen octets of plaintext
STF_TEST(AESKeyWrapWithPadding, SixteenOctetPlaintext)
{
    const std::uint8_t key[] =
    {
        0x25, 0x40, 0xdf, 0x6e, 0x29, 0xb0, 0x2a, 0xf1,
        0xab, 0x49, 0x3b, 0x70, 0x5b, 0xf1, 0x6e, 0xa1,
        0xae, 0x83, 0x38, 0xf4, 0xdc, 0xc1, 0x76, 0xa8
    };
    const std::uint8_t plaintext[] = {
        0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77,
        0x88, 0x99, 0xaa, 0xbb, 0xcc, 0xdd, 0xee, 0xff
    };
    const std::uint8_t expected_ciphertext[] =
    {
        0x5b, 0xe9, 0x8c, 0xdd, 0x32, 0xbd, 0xe2, 0xd3,
        0x3e, 0xf1, 0xd9, 0x5e, 0x32, 0x61, 0x57, 0x9d,
        0x4a, 0xeb, 0x08, 0xfc, 0x53, 0x8e, 0x14, 0xef
    };
    std::uint8_t ciphertext[sizeof(plaintext) + 16];
    std::size_t ciphertext_length;
    std::uint8_t plaintext_check[sizeof(ciphertext) - 8];
    std::size_t plaintext_check_length;

    // Create AESKeyWrap object using the given key and length
    AESKeyWrap aes_kw(key);

    // Perform the AES key wrap
    ciphertext_length = aes_kw.WrapWithPadding(plaintext, ciphertext);

    // Verify the expected ciphertext length
    STF_ASSERT_EQ(sizeof(expected_ciphertext), ciphertext_length);

    // Verify result
    STF_ASSERT_MEM_EQ(expected_ciphertext,
                      ciphertext,
                      sizeof(expected_ciphertext));

    // Perform the key unwrap operation
    plaintext_check_length =
        aes_kw.UnwrapWithPadding({ciphertext, ciphertext_length},
                                 plaintext_check);

    // Verify the expected plaintext length
    STF_ASSERT_EQ(sizeof(plaintext), plaintext_check_length);

    // Verify result
    STF_ASSERT_MEM_EQ(plaintext, plaintext_check, sizeof(plaintext));
}
