/*
 *  test_stream_3.cpp
 *
 *  Copyright (C) 2024
 *  Terrapane Corporation
 *  All Rights Reserved
 *
 *  Author:
 *      Paul E. Jones <paulej@packetizer.com>
 *
 *  Description:
 *      This module will test the AES Crypt Engine's ability to decrypt
 *      various known plaintext / ciphertext pairs following stream format 3.
 *
 *  Portability Issues:
 *      None.
 */

#include <vector>
#include <sstream>
#include <string>
#include <terra/aescrypt/engine/decryptor.h>
#include <terra/logger/null_ostream.h>
#include <terra/logger/logger.h>
#include <terra/stf/stf.h>
#include "../string_stream_buffer.h"

using namespace Terra::AESCrypt::Engine;
using namespace Terra::Logger;

// It is assumed that a char and uint8_t are the same size
static_assert(sizeof(char) == sizeof(std::uint8_t));

namespace
{

// Structure to hold plaintext and ciphertext pairs
struct PlaintextCiphertext
{
    const std::string plaintext;
    const std::vector<uint8_t> ciphertext;
};

} // namespace

STF_TEST(TestDecryption, Stream_Format_3)
{
    const std::u8string password = u8"Hello";
    const std::vector<PlaintextCiphertext> tests =
    {
        {
            std::string(""),
            {
                0x41, 0x45, 0x53, 0x03, 0x00, 0x00, 0x1b, 0x43,
                0x52, 0x45, 0x41, 0x54, 0x45, 0x44, 0x5f, 0x42,
                0x59, 0x00, 0x61, 0x65, 0x73, 0x63, 0x72, 0x79,
                0x70, 0x74, 0x20, 0x34, 0x2e, 0x30, 0x2e, 0x30,
                0x2e, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05,
                0x95, 0xf5, 0xb7, 0x51, 0x8f, 0xe2, 0xac, 0xc8,
                0xb4, 0x1e, 0x1d, 0x06, 0x30, 0xab, 0x9e, 0x3a,
                0x3f, 0x96, 0x2c, 0x07, 0x37, 0xbc, 0xf7, 0xe3,
                0xb7, 0x4a, 0x44, 0x54, 0x68, 0x29, 0xed, 0xe1,
                0xea, 0xed, 0x3e, 0x90, 0x4b, 0xde, 0xef, 0x45,
                0xbd, 0xc2, 0x21, 0xcc, 0x40, 0xa7, 0xda, 0x5a,
                0x55, 0x81, 0x6b, 0x18, 0x40, 0x05, 0xdd, 0x68,
                0xe7, 0x86, 0x93, 0xc4, 0x36, 0xae, 0x99, 0xc1,
                0xd6, 0xff, 0xf7, 0xe9, 0x0d, 0x46, 0x8e, 0x2c,
                0x37, 0x67, 0x63, 0x09, 0x6d, 0x19, 0xf3, 0xfb,
                0xf3, 0x77, 0x54, 0xc2, 0xbe, 0xb6, 0xcb, 0xaf,
                0x52, 0x9e, 0xb2, 0x62, 0x6f, 0x0b, 0x73, 0xc1,
                0x3f, 0xfe, 0x4d, 0x80, 0xbf, 0xfa, 0x0a, 0xa6,
                0x41, 0xec, 0x8c, 0x8b, 0x55, 0x7a, 0xc1, 0x10,
                0xa1, 0x23, 0x02, 0x4b, 0xbf, 0x7a, 0xf0, 0x48,
                0x5d, 0x1e, 0x19, 0xd4, 0x34, 0x1c, 0xe9, 0xdc,
                0x8f, 0xef, 0x30, 0x6e, 0xd5, 0x69, 0xae, 0xdd,
                0x8c, 0xd2, 0x7d, 0x9a, 0x42, 0x40, 0x99, 0x68
            }
        },
        {
            std::string("0"),
            {
                0x41, 0x45, 0x53, 0x03, 0x00, 0x00, 0x1b, 0x43,
                0x52, 0x45, 0x41, 0x54, 0x45, 0x44, 0x5f, 0x42,
                0x59, 0x00, 0x61, 0x65, 0x73, 0x63, 0x72, 0x79,
                0x70, 0x74, 0x20, 0x34, 0x2e, 0x30, 0x2e, 0x30,
                0x2e, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05,
                0xca, 0xc0, 0xb1, 0x77, 0xb8, 0x99, 0x03, 0x9b,
                0xb3, 0x26, 0xb3, 0x3c, 0x22, 0xa1, 0x8b, 0x39,
                0xf4, 0xbb, 0x30, 0x94, 0x4e, 0x08, 0x64, 0x80,
                0x74, 0x78, 0x6a, 0x57, 0x6b, 0xc4, 0xf1, 0x6c,
                0xdc, 0x59, 0x8b, 0xc1, 0xf3, 0x14, 0xb5, 0xa4,
                0xd4, 0x9b, 0x97, 0x2f, 0x30, 0x0a, 0xac, 0xd7,
                0xd8, 0xd6, 0x61, 0xe5, 0x5a, 0x73, 0xc0, 0x5c,
                0x19, 0x25, 0x80, 0x6d, 0x07, 0xdc, 0x1b, 0xfa,
                0x2f, 0x7d, 0x3c, 0x91, 0x1c, 0x0f, 0x6f, 0x61,
                0xff, 0xaf, 0x49, 0xb2, 0x51, 0xe0, 0x4b, 0x66,
                0x9a, 0x81, 0xf1, 0x47, 0x3d, 0xbc, 0xb2, 0x2d,
                0x57, 0x05, 0xf0, 0xde, 0x47, 0xbe, 0x32, 0xcf,
                0x1b, 0xee, 0x36, 0x24, 0x64, 0x0d, 0xe8, 0xda,
                0x7b, 0xec, 0x29, 0xd6, 0xbb, 0x6d, 0x78, 0xa5,
                0x0f, 0xf8, 0xfe, 0x3c, 0x70, 0xe8, 0xa7, 0x17,
                0x3e, 0x46, 0x1c, 0xfc, 0xe8, 0x08, 0x70, 0x2b,
                0xde, 0x6d, 0xde, 0x21, 0xab, 0x33, 0x5b, 0xe5,
                0xc3, 0xa1, 0x63, 0x39, 0x57, 0xb6, 0xec, 0xbe
            }
        },
        {
            std::string("012"),
            {
                0x41, 0x45, 0x53, 0x03, 0x00, 0x00, 0x1b, 0x43,
                0x52, 0x45, 0x41, 0x54, 0x45, 0x44, 0x5f, 0x42,
                0x59, 0x00, 0x61, 0x65, 0x73, 0x63, 0x72, 0x79,
                0x70, 0x74, 0x20, 0x34, 0x2e, 0x30, 0x2e, 0x30,
                0x2e, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05,
                0x61, 0xb1, 0xad, 0x59, 0x5a, 0x77, 0x25, 0x5d,
                0x1a, 0x85, 0x1a, 0x04, 0xfe, 0xd2, 0x94, 0x10,
                0x42, 0xc9, 0xa7, 0x7e, 0x24, 0xe8, 0x46, 0x6e,
                0x12, 0x43, 0xeb, 0x96, 0x7d, 0xc0, 0xfa, 0xe8,
                0xde, 0x7f, 0xfa, 0x11, 0x58, 0x08, 0xa3, 0x5c,
                0x96, 0x93, 0xf0, 0x30, 0x59, 0xa7, 0xba, 0xc9,
                0xf4, 0x88, 0xb5, 0x0f, 0x0d, 0x9b, 0xa4, 0x89,
                0xd9, 0x92, 0x6b, 0x93, 0x0e, 0x2e, 0x5f, 0x43,
                0xa7, 0x84, 0x9e, 0x2a, 0xcc, 0x63, 0xea, 0x1e,
                0xb0, 0xda, 0x6c, 0xee, 0xd4, 0x60, 0xa6, 0xf3,
                0x33, 0xad, 0x82, 0x64, 0xf8, 0x2b, 0xd9, 0xd4,
                0x36, 0x01, 0x4a, 0x71, 0x8d, 0x8a, 0x99, 0xcd,
                0xaa, 0x05, 0x41, 0x7f, 0xe6, 0x59, 0xf6, 0x0d,
                0xbb, 0x8f, 0xa1, 0xcb, 0x3b, 0x0d, 0x1d, 0x47,
                0x4f, 0x30, 0x01, 0x0b, 0x28, 0x8b, 0x1b, 0x60,
                0x20, 0xec, 0xeb, 0x15, 0x38, 0x5f, 0x2a, 0xd3,
                0xc6, 0x1b, 0xf2, 0xaa, 0x30, 0xe6, 0x88, 0x47,
                0xec, 0xf4, 0x8d, 0xae, 0x7c, 0xa5, 0xba, 0x21
            }
        },
        {
            std::string("0123456789ABCDE"),
            {
                0x41, 0x45, 0x53, 0x03, 0x00, 0x00, 0x1b, 0x43,
                0x52, 0x45, 0x41, 0x54, 0x45, 0x44, 0x5f, 0x42,
                0x59, 0x00, 0x61, 0x65, 0x73, 0x63, 0x72, 0x79,
                0x70, 0x74, 0x20, 0x34, 0x2e, 0x30, 0x2e, 0x30,
                0x2e, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05,
                0x5b, 0x0d, 0xa5, 0xbf, 0x2d, 0x63, 0x6b, 0x90,
                0x79, 0x1b, 0xf5, 0x06, 0xa9, 0x12, 0xbd, 0x74,
                0x06, 0x76, 0xfa, 0x9b, 0x0c, 0xbd, 0x67, 0x49,
                0xde, 0x44, 0xa5, 0xa3, 0xe1, 0x23, 0xa2, 0x4c,
                0xec, 0xef, 0x39, 0x3a, 0xa7, 0xbf, 0x02, 0x66,
                0x33, 0xc1, 0xf1, 0xe6, 0xf1, 0xbb, 0x86, 0xf3,
                0x1f, 0x05, 0x81, 0x69, 0x1d, 0xae, 0x57, 0x13,
                0x04, 0x0e, 0xaf, 0x78, 0x66, 0xc0, 0x5f, 0x22,
                0x2f, 0x2d, 0x44, 0xd0, 0xd1, 0x76, 0xaf, 0x04,
                0xb7, 0xe5, 0x5b, 0x96, 0x2b, 0xbe, 0x52, 0x08,
                0x3c, 0x52, 0x61, 0x34, 0x0f, 0xca, 0x4c, 0xde,
                0xbf, 0xed, 0x77, 0x59, 0x1e, 0xed, 0x8c, 0x7c,
                0xe8, 0x73, 0x25, 0x26, 0xff, 0xda, 0xf4, 0xfd,
                0x99, 0x7a, 0xec, 0x78, 0x45, 0x29, 0xdd, 0xf1,
                0x6d, 0x6b, 0x4b, 0xa3, 0xe7, 0x84, 0x8b, 0xfd,
                0x69, 0x29, 0xb9, 0xa2, 0x84, 0xe7, 0x21, 0x55,
                0xc4, 0xec, 0xd5, 0x18, 0x88, 0x5b, 0x72, 0x4b,
                0xd3, 0x60, 0x96, 0x3f, 0x0c, 0xa1, 0xe6, 0x48
            }
        },
        {
            std::string("0123456789ABCDEF"),
            {
                0x41, 0x45, 0x53, 0x03, 0x00, 0x00, 0x1b, 0x43,
                0x52, 0x45, 0x41, 0x54, 0x45, 0x44, 0x5f, 0x42,
                0x59, 0x00, 0x61, 0x65, 0x73, 0x63, 0x72, 0x79,
                0x70, 0x74, 0x20, 0x34, 0x2e, 0x30, 0x2e, 0x30,
                0x2e, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05,
                0x55, 0x9d, 0x64, 0x2d, 0x66, 0xb6, 0x65, 0x13,
                0xdf, 0x9b, 0xab, 0x97, 0x7c, 0x7b, 0xa8, 0x1d,
                0xde, 0xdc, 0x1e, 0x30, 0xf1, 0xe5, 0x0e, 0x33,
                0xc5, 0x29, 0x39, 0x26, 0xf7, 0x0b, 0x85, 0x72,
                0xc1, 0xb2, 0x02, 0x64, 0x5d, 0x45, 0xad, 0x57,
                0x7d, 0xf9, 0x88, 0x90, 0xa6, 0xe1, 0x3f, 0xa9,
                0x3c, 0x31, 0xe8, 0xc0, 0xea, 0xb0, 0xd2, 0x55,
                0x91, 0xd7, 0x23, 0x1d, 0x2b, 0x27, 0x44, 0xcd,
                0xd0, 0xb4, 0xf8, 0xd8, 0xc0, 0x8a, 0x8b, 0xa1,
                0xa7, 0x77, 0x8d, 0x67, 0x39, 0x30, 0x1c, 0xc9,
                0x82, 0x8d, 0xf6, 0xc0, 0xbd, 0xce, 0xd2, 0x3a,
                0xb1, 0xb9, 0xfb, 0xa2, 0x81, 0xa2, 0xf0, 0x4b,
                0x55, 0x5c, 0xd3, 0x88, 0xd1, 0x26, 0xdc, 0xd6,
                0x02, 0x8f, 0x20, 0x3b, 0xb3, 0x69, 0x0f, 0x30,
                0x1f, 0x68, 0xf5, 0x49, 0xaf, 0x30, 0x68, 0xee,
                0xb7, 0x26, 0x6a, 0xe9, 0xde, 0x7f, 0xee, 0x10,
                0x9f, 0xf0, 0x33, 0xc6, 0x04, 0x7d, 0x87, 0x8d,
                0xf7, 0xa1, 0x1f, 0xfe, 0x73, 0xd8, 0x07, 0xc6,
                0x7a, 0xad, 0x32, 0xaf, 0xf6, 0x8d, 0xce, 0x12,
                0xf7, 0xef, 0x95, 0xd6, 0x6c, 0x99, 0xdf, 0x9a
            }
        },
        {
            std::string("0123456789ABCDEF0"),
            {
                0x41, 0x45, 0x53, 0x03, 0x00, 0x00, 0x1b, 0x43,
                0x52, 0x45, 0x41, 0x54, 0x45, 0x44, 0x5f, 0x42,
                0x59, 0x00, 0x61, 0x65, 0x73, 0x63, 0x72, 0x79,
                0x70, 0x74, 0x20, 0x34, 0x2e, 0x30, 0x2e, 0x30,
                0x2e, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05,
                0x1e, 0x28, 0x98, 0x31, 0x91, 0x21, 0x15, 0xf5,
                0x59, 0xff, 0x0d, 0x91, 0xb0, 0x79, 0xaf, 0x59,
                0x93, 0x20, 0x76, 0x82, 0x7c, 0xc4, 0xa8, 0xbd,
                0x7d, 0x78, 0x2d, 0x37, 0xa6, 0x2d, 0x3e, 0x91,
                0x97, 0xab, 0xe1, 0xcc, 0x9a, 0x64, 0x74, 0xd4,
                0x8b, 0x29, 0x69, 0x1e, 0x65, 0x78, 0x37, 0x8e,
                0x36, 0x24, 0x15, 0xb9, 0x80, 0xd5, 0x46, 0x2f,
                0x1f, 0x17, 0x87, 0x02, 0xe8, 0x70, 0xb1, 0x5e,
                0xd7, 0x4e, 0xfe, 0x01, 0x74, 0x40, 0x8b, 0xdb,
                0x7f, 0x33, 0xf6, 0x38, 0x9b, 0x95, 0x60, 0x42,
                0x58, 0x1b, 0xd5, 0xd2, 0xd2, 0x22, 0xfb, 0x9c,
                0x99, 0xb4, 0x24, 0x41, 0x51, 0x22, 0x37, 0x79,
                0x9e, 0xd9, 0xb0, 0x0a, 0x6f, 0xbd, 0xd8, 0xef,
                0x22, 0x5d, 0x1a, 0xf6, 0x57, 0x9e, 0x8e, 0xd3,
                0x5b, 0xb7, 0x86, 0x19, 0x3f, 0x6e, 0x91, 0x50,
                0x47, 0xf9, 0x1c, 0xb8, 0x6c, 0xf8, 0x44, 0x01,
                0xd8, 0xbd, 0xee, 0x13, 0x24, 0xa3, 0x3d, 0xe3,
                0x4e, 0x13, 0xcd, 0xb4, 0x84, 0xb2, 0x39, 0x42,
                0x1d, 0x72, 0x5f, 0xdd, 0x30, 0x78, 0xb3, 0xda,
                0x48, 0x28, 0xc3, 0xd1, 0x04, 0x2a, 0xf0, 0x40
            }
        },
        {
            std::string("0123456789ABCDEF0123456789ABCDE"),
            {
                0x41, 0x45, 0x53, 0x03, 0x00, 0x00, 0x1b, 0x43,
                0x52, 0x45, 0x41, 0x54, 0x45, 0x44, 0x5f, 0x42,
                0x59, 0x00, 0x61, 0x65, 0x73, 0x63, 0x72, 0x79,
                0x70, 0x74, 0x20, 0x34, 0x2e, 0x30, 0x2e, 0x30,
                0x2e, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05,
                0x0a, 0x52, 0x42, 0x9b, 0xa6, 0x6b, 0x85, 0xbc,
                0x07, 0x5a, 0xa8, 0x19, 0xc6, 0x2e, 0x82, 0x2c,
                0x1a, 0xf8, 0xae, 0x0f, 0x59, 0x29, 0x7b, 0x52,
                0xd6, 0x93, 0xb2, 0x84, 0xea, 0x3b, 0x5f, 0x9d,
                0xe2, 0x5c, 0x5a, 0xc4, 0xf1, 0xee, 0x2c, 0x15,
                0x8e, 0x79, 0x4f, 0x2a, 0x3c, 0x24, 0x93, 0xa2,
                0xef, 0xcf, 0x9e, 0xdd, 0xc0, 0xa0, 0xc7, 0x8c,
                0x18, 0xa0, 0x48, 0xe7, 0x5b, 0x85, 0xff, 0xa2,
                0x25, 0xa5, 0x3c, 0x35, 0x08, 0xd7, 0x89, 0x30,
                0xa9, 0x43, 0x33, 0xd9, 0xa5, 0x57, 0xcf, 0x71,
                0x15, 0x87, 0x1e, 0x9d, 0xd5, 0x33, 0xc4, 0xa8,
                0xd4, 0xcc, 0x79, 0x8e, 0x91, 0x6e, 0x2a, 0x9d,
                0x08, 0xf2, 0x99, 0x39, 0xa5, 0x66, 0x24, 0xff,
                0x86, 0x98, 0x06, 0xce, 0x2d, 0x44, 0x01, 0x10,
                0x05, 0x9b, 0x68, 0xf0, 0xd7, 0x32, 0xc0, 0xfb,
                0x6d, 0xd1, 0xf0, 0x9d, 0x91, 0x28, 0xb6, 0x29,
                0xb2, 0x08, 0x48, 0x7a, 0x18, 0x78, 0xb3, 0xd9,
                0xe2, 0x12, 0x82, 0x74, 0x3b, 0x20, 0x0a, 0x54,
                0xc7, 0x38, 0xa0, 0xb6, 0xa9, 0x06, 0xe0, 0x10,
                0x53, 0xae, 0xfe, 0xe0, 0x48, 0x71, 0x08, 0x9a
            }
        },
        {
            std::string("0123456789ABCDEF0123456789ABCDEF"),
            {
                0x41, 0x45, 0x53, 0x03, 0x00, 0x00, 0x1b, 0x43,
                0x52, 0x45, 0x41, 0x54, 0x45, 0x44, 0x5f, 0x42,
                0x59, 0x00, 0x61, 0x65, 0x73, 0x63, 0x72, 0x79,
                0x70, 0x74, 0x20, 0x34, 0x2e, 0x30, 0x2e, 0x30,
                0x2e, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05,
                0x19, 0x1a, 0x15, 0x28, 0x4f, 0x27, 0x0a, 0x8c,
                0xd3, 0x33, 0x9e, 0x40, 0xbe, 0xe9, 0x70, 0xec,
                0x05, 0x43, 0x35, 0x45, 0x64, 0xbf, 0x49, 0xa6,
                0x16, 0x11, 0x53, 0x61, 0x16, 0xcf, 0x80, 0x2c,
                0x2a, 0xbe, 0x25, 0x44, 0x02, 0x70, 0x8a, 0x8e,
                0x5c, 0xf9, 0x3a, 0x63, 0x86, 0x58, 0x2d, 0x5d,
                0xb5, 0x84, 0xe8, 0x3d, 0x8e, 0x4b, 0xf7, 0x8b,
                0x79, 0x18, 0xcc, 0x99, 0x30, 0xe4, 0x8b, 0x1d,
                0x7d, 0x41, 0x78, 0xe4, 0x96, 0x7a, 0xad, 0x28,
                0x38, 0x34, 0x3c, 0x9e, 0xfe, 0x86, 0x24, 0xc2,
                0x73, 0x30, 0xd4, 0x15, 0x2c, 0xf4, 0x17, 0x5a,
                0x9c, 0x11, 0x7c, 0x73, 0x7c, 0xf3, 0x77, 0xec,
                0x91, 0xb4, 0x97, 0xeb, 0x19, 0xab, 0x7b, 0x77,
                0xb4, 0x62, 0x69, 0xfd, 0x31, 0x1c, 0xb3, 0xb3,
                0x43, 0x20, 0xe7, 0x57, 0xd6, 0x36, 0x05, 0xb1,
                0xb5, 0x37, 0x43, 0x1d, 0xba, 0x5a, 0x41, 0x18,
                0x28, 0x2d, 0x80, 0x46, 0xfc, 0xde, 0x21, 0x9c,
                0x54, 0x3a, 0xd2, 0xb0, 0x47, 0x6b, 0xd9, 0xec,
                0x90, 0x9c, 0xc3, 0x9a, 0x4e, 0x8e, 0x65, 0x17,
                0x88, 0x5e, 0x91, 0x68, 0x82, 0x2d, 0x11, 0xea,
                0x42, 0xfc, 0xb4, 0xf0, 0xf0, 0x07, 0x75, 0x2e,
                0xe8, 0x8e, 0x16, 0x5f, 0x73, 0x10, 0xae, 0x4a
            }
        },
        {
            std::string("0123456789ABCDEF0123456789ABCDEF0"),
            {
                0x41, 0x45, 0x53, 0x03, 0x00, 0x00, 0x1b, 0x43,
                0x52, 0x45, 0x41, 0x54, 0x45, 0x44, 0x5f, 0x42,
                0x59, 0x00, 0x61, 0x65, 0x73, 0x63, 0x72, 0x79,
                0x70, 0x74, 0x20, 0x34, 0x2e, 0x30, 0x2e, 0x30,
                0x2e, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05,
                0x6b, 0xbb, 0x72, 0x2e, 0xf4, 0xd7, 0x9f, 0x4d,
                0x09, 0xab, 0xa7, 0x53, 0x80, 0xbe, 0x4c, 0x97,
                0xea, 0x1f, 0x5c, 0x3b, 0x48, 0xde, 0x62, 0x66,
                0x48, 0xab, 0x5a, 0xcd, 0xb0, 0x2a, 0x98, 0x02,
                0x49, 0x5b, 0x25, 0xb0, 0x4f, 0x87, 0x30, 0x5a,
                0x39, 0x03, 0x83, 0x64, 0xf2, 0xba, 0xd7, 0xc0,
                0xc9, 0x6d, 0x89, 0x29, 0x40, 0x14, 0x9b, 0x6f,
                0x49, 0xd5, 0x29, 0x45, 0xd2, 0xc1, 0xba, 0x40,
                0x8c, 0x52, 0x39, 0x46, 0x90, 0x64, 0x20, 0x26,
                0x18, 0x98, 0x9b, 0x6b, 0x5c, 0x4f, 0x42, 0x2a,
                0xdd, 0x73, 0x59, 0xb0, 0x45, 0x5e, 0xe2, 0x25,
                0x4b, 0x63, 0xe2, 0x0c, 0x66, 0x68, 0x2b, 0xbb,
                0xdf, 0x75, 0xca, 0xa0, 0x95, 0x46, 0xc5, 0x55,
                0xef, 0xd1, 0xc4, 0xdb, 0xba, 0x03, 0x7c, 0x3e,
                0xf0, 0xe9, 0xe6, 0x61, 0x79, 0xdc, 0x3e, 0xee,
                0xd5, 0xe8, 0x1e, 0xca, 0xf9, 0x27, 0xea, 0x3b,
                0xde, 0xc2, 0xc2, 0x7b, 0x3a, 0x31, 0xa6, 0xd2,
                0xb8, 0xf7, 0xc3, 0x60, 0xf5, 0x70, 0x59, 0x99,
                0x22, 0x48, 0x59, 0xfb, 0xbb, 0x0a, 0xc9, 0xc6,
                0x2f, 0x5b, 0xa2, 0x14, 0xc6, 0x8c, 0xf8, 0x58,
                0xf0, 0x25, 0x1b, 0x21, 0x91, 0x02, 0xe8, 0x04,
                0xc3, 0xa2, 0x97, 0x7d, 0xc0, 0xcf, 0x4a, 0x8c
            }
        },
        {
            std::string("0123456789ABCDEF0123456789ABCDEF0123456789ABCDE"),
            {
                0x41, 0x45, 0x53, 0x03, 0x00, 0x00, 0x1b, 0x43,
                0x52, 0x45, 0x41, 0x54, 0x45, 0x44, 0x5f, 0x42,
                0x59, 0x00, 0x61, 0x65, 0x73, 0x63, 0x72, 0x79,
                0x70, 0x74, 0x20, 0x34, 0x2e, 0x30, 0x2e, 0x30,
                0x2e, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05,
                0xe1, 0xb9, 0x9e, 0xe8, 0x98, 0x29, 0x52, 0xe1,
                0xe9, 0x18, 0xb1, 0x9b, 0x46, 0xf6, 0x93, 0xc3,
                0xeb, 0x5d, 0x66, 0x0e, 0x76, 0xd0, 0xfb, 0xb6,
                0xfa, 0x91, 0x9f, 0x76, 0x7e, 0x03, 0x77, 0x34,
                0x8c, 0xba, 0xe3, 0x19, 0x5d, 0xf0, 0x19, 0x33,
                0x88, 0xc8, 0x2b, 0x90, 0x40, 0xf0, 0x02, 0x5a,
                0xe9, 0xa8, 0x5a, 0x84, 0xa8, 0x66, 0x91, 0x6c,
                0x2c, 0x6b, 0xea, 0xc7, 0x27, 0xba, 0x5a, 0x77,
                0x3f, 0x87, 0xb9, 0x1d, 0x6e, 0x6f, 0x19, 0xd0,
                0x76, 0xf5, 0xed, 0xd9, 0x7c, 0x14, 0x7f, 0x47,
                0x99, 0xe1, 0xc9, 0x0d, 0x65, 0xae, 0x03, 0x7f,
                0x95, 0xac, 0x6e, 0xf1, 0xa1, 0x60, 0x13, 0x5f,
                0xfb, 0x7d, 0xcb, 0xff, 0x9d, 0x02, 0x97, 0xc7,
                0x9e, 0x1c, 0x5f, 0xe5, 0x52, 0x8e, 0x7f, 0x89,
                0x81, 0x5f, 0xc4, 0x02, 0xda, 0xb1, 0x14, 0x17,
                0x89, 0xdd, 0xd9, 0x43, 0x97, 0x33, 0x46, 0x3b,
                0x3f, 0x88, 0x2f, 0x1d, 0xc5, 0xdd, 0x46, 0xae,
                0x1d, 0x58, 0x50, 0x6c, 0xd3, 0x16, 0x5a, 0x52,
                0xc5, 0x1d, 0xbe, 0xfc, 0xb5, 0x99, 0x7d, 0xb9,
                0xbc, 0xd2, 0xcc, 0x67, 0xd9, 0x94, 0xb8, 0x44,
                0xa1, 0xc9, 0x39, 0xd1, 0xab, 0xeb, 0x7e, 0x43,
                0xc2, 0x08, 0x5d, 0xe4, 0x41, 0xf4, 0x5f, 0xa4
            }
        },
        {
            std::string("0123456789ABCDEF0123456789ABCDEF0123456789ABCDEF"),
            {
                0x41, 0x45, 0x53, 0x03, 0x00, 0x00, 0x1b, 0x43,
                0x52, 0x45, 0x41, 0x54, 0x45, 0x44, 0x5f, 0x42,
                0x59, 0x00, 0x61, 0x65, 0x73, 0x63, 0x72, 0x79,
                0x70, 0x74, 0x20, 0x34, 0x2e, 0x30, 0x2e, 0x30,
                0x2e, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05,
                0xfb, 0x81, 0xcd, 0x62, 0x72, 0xa6, 0x3b, 0xdd,
                0xad, 0x4b, 0x6c, 0x81, 0x3e, 0x25, 0x33, 0x68,
                0x10, 0xd9, 0x7a, 0x50, 0x5e, 0x0e, 0x69, 0x90,
                0x8f, 0x2b, 0x44, 0x93, 0x9a, 0x91, 0x46, 0x1b,
                0x32, 0xbb, 0xa0, 0xf6, 0x9b, 0x2c, 0x1d, 0xa4,
                0x72, 0xd1, 0xfd, 0x1d, 0x02, 0xe5, 0x6a, 0x8e,
                0x94, 0x39, 0xeb, 0xe9, 0xb2, 0x1a, 0x44, 0xdf,
                0x39, 0x48, 0xc4, 0x81, 0x53, 0x7f, 0x08, 0xd7,
                0x6a, 0x55, 0x14, 0xde, 0x3d, 0x61, 0x9b, 0x6c,
                0xe1, 0xc9, 0x7c, 0x38, 0x31, 0xd3, 0xa5, 0x21,
                0x82, 0x1e, 0x44, 0xfe, 0xba, 0x18, 0x7c, 0x3d,
                0x29, 0x39, 0x57, 0xbf, 0x5a, 0xdb, 0x31, 0xf5,
                0x97, 0xa1, 0x8c, 0x52, 0x9c, 0x4a, 0x3e, 0xca,
                0xda, 0x8b, 0xc6, 0xd4, 0xcd, 0x8e, 0xfe, 0xc2,
                0x4c, 0x86, 0xab, 0xa3, 0x56, 0xe3, 0x7d, 0x7e,
                0xd1, 0x85, 0x63, 0x8c, 0xc4, 0x53, 0x21, 0x89,
                0x61, 0xef, 0xee, 0xd6, 0xfb, 0x19, 0x14, 0x5a,
                0x45, 0x99, 0x53, 0x40, 0x45, 0x1a, 0x92, 0xa3,
                0x5f, 0x36, 0x81, 0xeb, 0x81, 0xc4, 0xab, 0xfb,
                0xab, 0x8f, 0xbf, 0xba, 0x65, 0xa7, 0x99, 0xdc,
                0x70, 0x27, 0xb8, 0x66, 0x9a, 0xe5, 0x60, 0x1f,
                0xc8, 0xe8, 0xd8, 0xb1, 0x24, 0x75, 0x31, 0x88,
                0xb6, 0xf4, 0x23, 0xed, 0x25, 0x09, 0x08, 0xa2,
                0x85, 0x7d, 0xc0, 0xa3, 0xc2, 0xea, 0x05, 0x7e
            }
        },
        {
            std::string("0123456789ABCDEF0123456789ABCDEF0123456789ABCDEF0"),
            {
                0x41, 0x45, 0x53, 0x03, 0x00, 0x00, 0x1b, 0x43,
                0x52, 0x45, 0x41, 0x54, 0x45, 0x44, 0x5f, 0x42,
                0x59, 0x00, 0x61, 0x65, 0x73, 0x63, 0x72, 0x79,
                0x70, 0x74, 0x20, 0x34, 0x2e, 0x30, 0x2e, 0x30,
                0x2e, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05,
                0x37, 0x19, 0x35, 0x7f, 0x6d, 0xf3, 0x1e, 0x24,
                0xe3, 0x94, 0x28, 0xa1, 0x45, 0xa7, 0x1a, 0x92,
                0xa4, 0x6b, 0xc7, 0xa3, 0xa9, 0x34, 0x8a, 0xfd,
                0xa8, 0xe3, 0xf9, 0xc6, 0x80, 0xd1, 0x1e, 0x5d,
                0xb5, 0xd1, 0xfe, 0x07, 0x22, 0x42, 0x75, 0xd8,
                0xf2, 0xe1, 0x6b, 0xa0, 0x42, 0xb3, 0x83, 0x73,
                0xa0, 0x5f, 0x81, 0x03, 0xae, 0x75, 0x31, 0x4f,
                0x05, 0xee, 0x38, 0xb4, 0x13, 0x1d, 0xaf, 0x97,
                0xd0, 0xdb, 0x7b, 0x68, 0x44, 0x50, 0xcb, 0x25,
                0x74, 0xef, 0x25, 0x76, 0x6e, 0x5c, 0xcf, 0x0c,
                0xbc, 0x5a, 0xa4, 0x39, 0x6a, 0xc2, 0x70, 0x94,
                0xeb, 0x00, 0x13, 0x3c, 0xd8, 0x13, 0xbe, 0x47,
                0x2a, 0xbb, 0xef, 0xb0, 0x74, 0x5b, 0xd1, 0x2e,
                0x32, 0x40, 0xef, 0x91, 0x9c, 0x01, 0x24, 0x68,
                0x01, 0x1b, 0xdc, 0x27, 0xc1, 0xdc, 0x31, 0x60,
                0xa9, 0xeb, 0x8a, 0xff, 0x7e, 0x62, 0xb2, 0xf4,
                0x91, 0x16, 0xd2, 0x0d, 0x50, 0xc9, 0x0a, 0xe2,
                0x1b, 0xd9, 0xf4, 0x9e, 0xeb, 0xa5, 0xcb, 0x62,
                0x1c, 0x9b, 0x45, 0x64, 0x35, 0xdf, 0x62, 0x84,
                0xd2, 0x92, 0x80, 0x90, 0x88, 0xab, 0xba, 0xd7,
                0xd6, 0x80, 0x1a, 0xc3, 0x1d, 0xb0, 0x44, 0xce,
                0xff, 0xd0, 0xf9, 0x43, 0xa4, 0x87, 0x34, 0xd3,
                0x67, 0x68, 0x1f, 0x95, 0x9f, 0xdc, 0x72, 0x86,
                0xbc, 0xfa, 0xf2, 0x3a, 0xe8, 0xbf, 0x15, 0xd3
            }
        },
        {
            std::string("0123456789ABCDEF0123456789ABCDEF0123456789ABCDEF012345"
                        "6789ABCDE"),
            {
                0x41, 0x45, 0x53, 0x03, 0x00, 0x00, 0x1b, 0x43,
                0x52, 0x45, 0x41, 0x54, 0x45, 0x44, 0x5f, 0x42,
                0x59, 0x00, 0x61, 0x65, 0x73, 0x63, 0x72, 0x79,
                0x70, 0x74, 0x20, 0x34, 0x2e, 0x30, 0x2e, 0x30,
                0x2e, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05,
                0x52, 0x8e, 0xbd, 0xa5, 0x04, 0xc1, 0xd1, 0xd6,
                0x67, 0x16, 0xa7, 0x94, 0x9b, 0xf2, 0x6b, 0x03,
                0x47, 0x50, 0x49, 0xed, 0xce, 0x52, 0xcd, 0xcf,
                0x07, 0x07, 0xcb, 0x19, 0x35, 0x43, 0x27, 0xb4,
                0x58, 0xcf, 0x19, 0xd2, 0xd4, 0x93, 0x91, 0x8a,
                0xad, 0xc4, 0x0d, 0xae, 0x26, 0x60, 0x8b, 0x3b,
                0x4a, 0xe6, 0xcf, 0x92, 0xa9, 0xb2, 0xc6, 0x58,
                0xd3, 0x48, 0x74, 0x08, 0xaf, 0x06, 0x83, 0xb5,
                0x60, 0x12, 0xb8, 0x18, 0xf3, 0x8f, 0x6f, 0x87,
                0x00, 0x15, 0xaf, 0x84, 0xb9, 0x71, 0x82, 0x6e,
                0xfe, 0xeb, 0xc7, 0xa9, 0x3e, 0x4f, 0xd5, 0x26,
                0xa4, 0x5c, 0x71, 0xcc, 0x5b, 0x2f, 0x60, 0x2c,
                0xee, 0x56, 0xb8, 0x33, 0xef, 0xdc, 0x29, 0x48,
                0x45, 0x51, 0xe6, 0x97, 0x59, 0x01, 0xe8, 0x9a,
                0xa4, 0x66, 0xf0, 0x26, 0xe5, 0x69, 0x88, 0x0e,
                0x3e, 0x74, 0xf9, 0x19, 0x84, 0x47, 0x05, 0x38,
                0x55, 0xb8, 0x98, 0x33, 0xa2, 0xc5, 0x17, 0xea,
                0xad, 0xc4, 0x60, 0x78, 0x46, 0xf4, 0x24, 0xa8,
                0x59, 0x2c, 0x70, 0xb7, 0x5c, 0xe7, 0xe2, 0x76,
                0xb5, 0x87, 0xd7, 0x8e, 0xb9, 0x84, 0xc4, 0x7d,
                0x15, 0x12, 0x27, 0xc5, 0xac, 0x6c, 0x52, 0x08,
                0x81, 0xe8, 0xdc, 0x37, 0xbc, 0xc3, 0xcb, 0x7d,
                0xd0, 0x43, 0xb2, 0xb3, 0x81, 0xe1, 0x66, 0xfe,
                0x2f, 0xf7, 0xa3, 0x5a, 0xcb, 0x74, 0x74, 0xf6
            }
        },
        {
            std::string("0123456789ABCDEF0123456789ABCDEF0123456789ABCDEF012345"
                        "6789ABCDEF"),
            {
                0x41, 0x45, 0x53, 0x03, 0x00, 0x00, 0x1b, 0x43,
                0x52, 0x45, 0x41, 0x54, 0x45, 0x44, 0x5f, 0x42,
                0x59, 0x00, 0x61, 0x65, 0x73, 0x63, 0x72, 0x79,
                0x70, 0x74, 0x20, 0x34, 0x2e, 0x30, 0x2e, 0x30,
                0x2e, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05,
                0xd2, 0xb5, 0x1f, 0x77, 0xe7, 0xcc, 0x67, 0xa5,
                0x61, 0x73, 0xcf, 0x07, 0xa6, 0xca, 0x0d, 0xa3,
                0xe0, 0xa4, 0x51, 0x62, 0xf0, 0x20, 0x3c, 0xa0,
                0xcf, 0x0d, 0xfb, 0xab, 0x6b, 0x20, 0x06, 0x11,
                0xaf, 0xae, 0xd1, 0x45, 0x90, 0x91, 0x9e, 0x91,
                0xad, 0xc2, 0x44, 0xfe, 0x08, 0x22, 0x60, 0x9d,
                0x67, 0x06, 0x83, 0xa5, 0x50, 0x10, 0x58, 0x31,
                0x4e, 0x0d, 0x5c, 0x65, 0x0a, 0x77, 0x33, 0x92,
                0x9f, 0x66, 0x2a, 0xad, 0x55, 0x8e, 0xc5, 0x5a,
                0xa7, 0x22, 0x7d, 0x8c, 0xc5, 0x87, 0x8b, 0x44,
                0xa4, 0x83, 0x8b, 0xc8, 0x5b, 0x30, 0x79, 0x23,
                0x7d, 0x23, 0x38, 0x85, 0xe3, 0x83, 0x7d, 0x08,
                0x31, 0x22, 0xa5, 0x3d, 0xd6, 0x3c, 0xe3, 0x93,
                0x36, 0x17, 0x42, 0xaa, 0x5f, 0x33, 0x37, 0x40,
                0x8f, 0x24, 0xa3, 0x23, 0xcb, 0xbe, 0xeb, 0xd0,
                0x44, 0xeb, 0x36, 0xc7, 0x14, 0x1e, 0xfc, 0xd4,
                0xa0, 0x0c, 0x05, 0x66, 0xf3, 0x2a, 0x15, 0xb5,
                0xcc, 0xa5, 0x1a, 0xcf, 0xed, 0x7d, 0x18, 0x79,
                0xfb, 0x67, 0xd0, 0xe7, 0xcc, 0x80, 0x4e, 0xa3,
                0x64, 0x2e, 0x6f, 0xfb, 0xc1, 0x83, 0x30, 0xc9,
                0x87, 0xec, 0xbf, 0x9f, 0x66, 0xe8, 0xc6, 0xa4,
                0x92, 0x26, 0x9c, 0xec, 0x92, 0x53, 0x1e, 0xa9,
                0xe5, 0xf8, 0xd6, 0x16, 0xbd, 0x8c, 0xef, 0xef,
                0xb7, 0xad, 0x91, 0x26, 0xac, 0xeb, 0xa9, 0xa7,
                0x59, 0xd3, 0x37, 0xc4, 0xba, 0x3f, 0x07, 0x7e,
                0xc3, 0x23, 0x0e, 0x77, 0xba, 0x0b, 0x31, 0x99
            }
        },
        {
            std::string("0123456789ABCDEF0123456789ABCDEF0123456789ABCDEF012345"
                        "6789ABCDEF0"),
            {
                0x41, 0x45, 0x53, 0x03, 0x00, 0x00, 0x1b, 0x43,
                0x52, 0x45, 0x41, 0x54, 0x45, 0x44, 0x5f, 0x42,
                0x59, 0x00, 0x61, 0x65, 0x73, 0x63, 0x72, 0x79,
                0x70, 0x74, 0x20, 0x34, 0x2e, 0x30, 0x2e, 0x30,
                0x2e, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05,
                0x91, 0x6a, 0xbb, 0x74, 0x1b, 0x78, 0x0c, 0xfb,
                0xa5, 0xd1, 0xab, 0x75, 0x84, 0x08, 0x7b, 0x74,
                0xbf, 0xec, 0xa9, 0x92, 0xce, 0xe6, 0x88, 0xb0,
                0x87, 0x4b, 0x08, 0x6c, 0x60, 0xc3, 0xa7, 0x0c,
                0xe0, 0x45, 0x9e, 0x81, 0x2a, 0x58, 0x96, 0x2f,
                0x3c, 0xb7, 0xbf, 0xf6, 0x40, 0x09, 0x57, 0x60,
                0x15, 0xdc, 0x09, 0x36, 0x63, 0x9e, 0xd4, 0x5f,
                0x21, 0x36, 0xa3, 0xf0, 0xd0, 0x56, 0xe0, 0x2d,
                0xbe, 0xce, 0x82, 0x92, 0xd3, 0xfe, 0x7d, 0x34,
                0x6a, 0x6f, 0xeb, 0xdd, 0xe9, 0x23, 0xc8, 0x63,
                0x85, 0xf3, 0xc5, 0x4c, 0x5c, 0xb2, 0xc6, 0x48,
                0x07, 0xe2, 0x9e, 0x29, 0x5e, 0x49, 0x44, 0x61,
                0x28, 0xc0, 0xb2, 0x6e, 0xbb, 0x69, 0x08, 0x47,
                0x59, 0xd0, 0xe4, 0x87, 0xd9, 0xec, 0x3f, 0x63,
                0x7d, 0xdd, 0x79, 0xfb, 0x94, 0xef, 0xd0, 0x74,
                0x3d, 0x4d, 0x70, 0xb2, 0x98, 0x76, 0x58, 0x83,
                0xbf, 0xdf, 0xaa, 0x5d, 0xfe, 0x60, 0xb5, 0x36,
                0x23, 0x71, 0x59, 0x29, 0x2d, 0x11, 0x30, 0x75,
                0xf1, 0x68, 0x65, 0x1b, 0xd0, 0xaf, 0x1d, 0x2f,
                0xf7, 0xbd, 0xb9, 0xa0, 0xec, 0x55, 0x9f, 0x5e,
                0x69, 0xbf, 0x4b, 0x81, 0x2b, 0x08, 0x9e, 0xd0,
                0x88, 0x85, 0x27, 0x1c, 0xf2, 0x75, 0x42, 0x37,
                0xfb, 0xbb, 0x98, 0xcb, 0x92, 0x37, 0x2b, 0xd9,
                0x55, 0x0b, 0x6c, 0xfb, 0x99, 0x01, 0xec, 0xcd,
                0x9c, 0xda, 0x9e, 0x3a, 0x9a, 0xe9, 0xcb, 0x13,
                0x3f, 0xc1, 0x6b, 0x34, 0x80, 0x94, 0x3b, 0x29
            }
        },
        {
            std::string("0123456789ABCDEF0123456789ABCDEF0123456789ABCDEF012345"
                        "6789ABCDEF0123456789ABCDE"),
            {
                0x41, 0x45, 0x53, 0x03, 0x00, 0x00, 0x1b, 0x43,
                0x52, 0x45, 0x41, 0x54, 0x45, 0x44, 0x5f, 0x42,
                0x59, 0x00, 0x61, 0x65, 0x73, 0x63, 0x72, 0x79,
                0x70, 0x74, 0x20, 0x34, 0x2e, 0x30, 0x2e, 0x30,
                0x2e, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05,
                0xa4, 0x75, 0x0a, 0x8e, 0x6c, 0x11, 0xf4, 0x01,
                0xa5, 0xaa, 0x63, 0x91, 0x5b, 0xe3, 0x2e, 0x2a,
                0xd5, 0x4a, 0xe5, 0xfc, 0xde, 0xb4, 0xbf, 0x6e,
                0x82, 0xc5, 0x94, 0xa9, 0xe4, 0xe2, 0x83, 0xe4,
                0x81, 0x1d, 0xf8, 0x92, 0x24, 0xf5, 0xc9, 0xcc,
                0x18, 0xf2, 0x26, 0xef, 0x87, 0xff, 0x28, 0xda,
                0x2b, 0x96, 0x6b, 0x32, 0x7c, 0xe8, 0xad, 0x99,
                0x25, 0xdf, 0x84, 0x31, 0xaf, 0x28, 0x88, 0xd3,
                0xad, 0x3b, 0x6a, 0x41, 0x7c, 0x98, 0x92, 0xce,
                0x91, 0xaa, 0xc3, 0xcc, 0xe6, 0x9b, 0x0d, 0x92,
                0x42, 0x63, 0x8b, 0xad, 0x0b, 0x99, 0x1e, 0x85,
                0x21, 0x8b, 0x8d, 0x0b, 0x40, 0xea, 0x74, 0x54,
                0xea, 0xe9, 0x37, 0xf5, 0xae, 0x5e, 0xa6, 0x10,
                0x67, 0x60, 0xb8, 0xd1, 0x18, 0xb1, 0x82, 0x59,
                0xdb, 0xa9, 0xc0, 0x2e, 0x49, 0xc4, 0x77, 0xd9,
                0x91, 0x8b, 0x7c, 0x9d, 0xc2, 0x74, 0xf6, 0x47,
                0x9b, 0xf5, 0x56, 0x86, 0x59, 0x65, 0x60, 0x21,
                0x45, 0x53, 0xa9, 0xb0, 0xdb, 0x23, 0x0c, 0x11,
                0xd5, 0x5c, 0xaa, 0x2b, 0xb7, 0x3c, 0xdd, 0x4e,
                0xe9, 0xa8, 0xe9, 0x19, 0x7b, 0xba, 0x97, 0x13,
                0xd7, 0x7c, 0x69, 0x0d, 0xe9, 0xb6, 0x96, 0x15,
                0x19, 0xc8, 0x12, 0xa7, 0x21, 0x29, 0xd4, 0x45,
                0x4c, 0xb5, 0xd2, 0x43, 0x67, 0x0c, 0x48, 0x6b,
                0x0d, 0xc3, 0xe1, 0xe2, 0xba, 0x98, 0xce, 0x5a,
                0x30, 0x2b, 0x21, 0xd1, 0x86, 0xec, 0x27, 0xb7,
                0x4d, 0x57, 0xed, 0x72, 0xdd, 0x9b, 0x92, 0x41
            }
        },
        {
            std::string("0123456789ABCDEF0123456789ABCDEF0123456789ABCDEF012345"
                        "6789ABCDEF0123456789ABCDEF"),
            {
                0x41, 0x45, 0x53, 0x03, 0x00, 0x00, 0x1b, 0x43,
                0x52, 0x45, 0x41, 0x54, 0x45, 0x44, 0x5f, 0x42,
                0x59, 0x00, 0x61, 0x65, 0x73, 0x63, 0x72, 0x79,
                0x70, 0x74, 0x20, 0x34, 0x2e, 0x30, 0x2e, 0x30,
                0x2e, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05,
                0x20, 0x9b, 0xb9, 0xeb, 0xa4, 0x75, 0x1f, 0x08,
                0x30, 0x90, 0xaa, 0xca, 0x06, 0x63, 0xd6, 0x59,
                0x19, 0x3c, 0x76, 0x53, 0x6c, 0x20, 0x02, 0x48,
                0xc6, 0x16, 0x5c, 0xba, 0xc6, 0xcc, 0x03, 0x87,
                0x4d, 0xec, 0xf0, 0x2e, 0x4c, 0xe0, 0xbf, 0xb5,
                0x6f, 0x0d, 0xe8, 0xf1, 0xb4, 0x03, 0x8e, 0xe5,
                0x9b, 0xaf, 0x79, 0x5b, 0xc4, 0x7b, 0x10, 0xeb,
                0x53, 0xa1, 0xc8, 0xe0, 0x0a, 0x21, 0x57, 0x2d,
                0x1b, 0x43, 0xf7, 0x60, 0x7f, 0x9a, 0xad, 0xa6,
                0x37, 0x1d, 0x57, 0x1e, 0xf0, 0x21, 0xf6, 0x21,
                0x6d, 0x2d, 0x03, 0xd9, 0xf8, 0x1d, 0x0e, 0x86,
                0x49, 0x06, 0x05, 0x19, 0xfa, 0xf2, 0x89, 0x9a,
                0x12, 0xa6, 0xf4, 0x8e, 0x18, 0x14, 0x62, 0x51,
                0xfb, 0xdf, 0xf3, 0xfa, 0x19, 0xcc, 0x95, 0x98,
                0x74, 0x48, 0x65, 0xad, 0x79, 0x16, 0x5a, 0xd8,
                0x05, 0x52, 0xeb, 0xf1, 0x28, 0x75, 0x76, 0xb2,
                0x0b, 0x7e, 0xb1, 0xb2, 0x57, 0xf8, 0xec, 0x64,
                0xcf, 0x9b, 0xf5, 0x53, 0x10, 0xf2, 0x51, 0xa1,
                0xff, 0xa3, 0x20, 0x2d, 0x79, 0xcb, 0x04, 0x1a,
                0x99, 0x74, 0x25, 0xcb, 0x7d, 0xd7, 0xf6, 0x48,
                0xac, 0x73, 0x76, 0xf7, 0x6b, 0x97, 0xa5, 0x1f,
                0xa3, 0x53, 0xd0, 0xed, 0xd8, 0x66, 0xbb, 0x75,
                0xe3, 0xc6, 0xfc, 0xe7, 0xdb, 0x7d, 0xf4, 0xba,
                0x4a, 0xa3, 0x64, 0xa1, 0x57, 0x07, 0x4d, 0xc8,
                0x8d, 0xd5, 0x2e, 0x66, 0x6f, 0x0c, 0xfc, 0xee,
                0xf1, 0xdf, 0xb0, 0xf1, 0x4f, 0x35, 0xa9, 0xaf,
                0xe1, 0x6c, 0x0b, 0x05, 0x85, 0x7f, 0xfb, 0x47,
                0x73, 0x3e, 0x1f, 0x60, 0x4c, 0x5e, 0x69, 0x8f
            }
        },
        {
            std::string("0123456789ABCDEF0123456789ABCDEF0123456789ABCDEF012345"
                        "6789ABCDEF0123456789ABCDEF0"),
            {
                0x41, 0x45, 0x53, 0x03, 0x00, 0x00, 0x1b, 0x43,
                0x52, 0x45, 0x41, 0x54, 0x45, 0x44, 0x5f, 0x42,
                0x59, 0x00, 0x61, 0x65, 0x73, 0x63, 0x72, 0x79,
                0x70, 0x74, 0x20, 0x34, 0x2e, 0x30, 0x2e, 0x30,
                0x2e, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05,
                0x77, 0xfa, 0xda, 0x65, 0x84, 0xf6, 0x1f, 0xa3,
                0x8e, 0xa5, 0x95, 0x7b, 0x77, 0x8b, 0x67, 0x87,
                0x36, 0x05, 0xe5, 0x4a, 0x1c, 0x00, 0x1b, 0xcf,
                0x05, 0x45, 0x2a, 0x9d, 0x36, 0x94, 0xc9, 0x3e,
                0x76, 0x3b, 0x5e, 0x9a, 0xe7, 0xcc, 0x33, 0x60,
                0xdc, 0xa0, 0x7f, 0x2a, 0xca, 0xf0, 0x42, 0x4f,
                0x8f, 0x67, 0xa5, 0xc4, 0xd9, 0x3b, 0xe6, 0x3c,
                0x7d, 0xf4, 0xc1, 0x7b, 0x83, 0x56, 0xc3, 0xb9,
                0x0d, 0x7d, 0x83, 0x41, 0xdc, 0x83, 0xfb, 0xcb,
                0x0e, 0x15, 0xe8, 0x28, 0xf6, 0x17, 0x2f, 0x2f,
                0xe5, 0xef, 0x11, 0x02, 0xfd, 0xc8, 0xce, 0xa1,
                0x02, 0x4e, 0x94, 0xa0, 0x33, 0xee, 0x6f, 0xb0,
                0xe2, 0xb3, 0x09, 0xa7, 0x94, 0x20, 0xc2, 0x01,
                0x30, 0xaf, 0x55, 0xec, 0xd9, 0x97, 0x95, 0xe6,
                0xa8, 0x76, 0x5b, 0x9c, 0x70, 0x8a, 0xb5, 0x59,
                0x3a, 0xbc, 0x12, 0xb3, 0x90, 0x35, 0xda, 0xd0,
                0x3a, 0x15, 0x34, 0xec, 0xf5, 0x3c, 0xbe, 0x1a,
                0x06, 0x63, 0xd7, 0xf9, 0xbe, 0x35, 0x42, 0xb6,
                0xe6, 0xc8, 0xc4, 0xf6, 0xba, 0xc8, 0x2d, 0x97,
                0x73, 0xde, 0xd1, 0xb5, 0x34, 0xa8, 0xd6, 0x7f,
                0x2e, 0xd6, 0x00, 0xdc, 0xb9, 0xa3, 0xf0, 0xaf,
                0x61, 0x24, 0x69, 0x3d, 0x58, 0x70, 0x7d, 0x26,
                0xd7, 0x03, 0x23, 0x80, 0x29, 0x92, 0x75, 0x00,
                0xd1, 0xfa, 0x9c, 0x05, 0xb1, 0x12, 0x3f, 0xfa,
                0x07, 0xe7, 0x81, 0x15, 0x9e, 0x2b, 0x86, 0x29,
                0xdb, 0x06, 0xbf, 0x3a, 0xfe, 0xa7, 0x57, 0x63,
                0xe5, 0xb1, 0xff, 0xe4, 0x96, 0x6c, 0x51, 0x18,
                0x9a, 0x76, 0x30, 0xbe, 0x97, 0xa5, 0x6e, 0x31
            }
        },
        {
            std::string("0123456789ABCDEF0123456789ABCDEF0123456789ABCDEF012345"
                        "6789ABCDEF0123456789ABCDEF0123456789ABCDEF0123456789AB"
                        "CDEF0123456789ABCDEF0123456789ABCDEF0123456789ABCDEF01"
                        "23456789ABCDEF0123456789ABCDEF0123456789ABCDEF01234567"
                        "89ABCDEF0123456789ABCDEF0123456789ABCDE"),
            {
                0x41, 0x45, 0x53, 0x03, 0x00, 0x00, 0x1b, 0x43,
                0x52, 0x45, 0x41, 0x54, 0x45, 0x44, 0x5f, 0x42,
                0x59, 0x00, 0x61, 0x65, 0x73, 0x63, 0x72, 0x79,
                0x70, 0x74, 0x20, 0x34, 0x2e, 0x30, 0x2e, 0x30,
                0x2e, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05,
                0x9b, 0xe9, 0xa8, 0x94, 0xdf, 0xb9, 0x65, 0x87,
                0x4f, 0x7c, 0xef, 0x94, 0xd3, 0xde, 0xa6, 0x9f,
                0xf4, 0x5a, 0xf0, 0xfa, 0x03, 0xbf, 0xbc, 0xe3,
                0x23, 0x99, 0x04, 0x5b, 0x7f, 0x16, 0xc4, 0x70,
                0x43, 0x19, 0x8d, 0x41, 0x10, 0xe2, 0xb2, 0xe3,
                0xd5, 0x15, 0x55, 0xc3, 0x93, 0x8d, 0xe2, 0x37,
                0x9b, 0xf3, 0x10, 0x46, 0x1a, 0xb5, 0xc7, 0x9b,
                0xef, 0x74, 0x92, 0xc3, 0x0a, 0x7c, 0xde, 0x05,
                0x80, 0xdb, 0xb8, 0x39, 0x0b, 0xc3, 0x3a, 0xa4,
                0x7a, 0x38, 0x0e, 0xd4, 0x19, 0x41, 0xd9, 0x61,
                0x77, 0xda, 0x32, 0xf5, 0xc5, 0x84, 0xba, 0xa6,
                0x83, 0xa2, 0x42, 0x45, 0x36, 0xf1, 0xe1, 0x19,
                0xe3, 0x57, 0x10, 0x1b, 0xc4, 0xb7, 0x86, 0x6f,
                0xa2, 0x1c, 0x82, 0xf8, 0x84, 0x1b, 0xba, 0x2d,
                0x0d, 0x1d, 0x36, 0x17, 0xba, 0x3d, 0x3b, 0x83,
                0x3a, 0xd7, 0x33, 0x14, 0x05, 0x38, 0x88, 0x4a,
                0x8c, 0xb9, 0xea, 0x7d, 0xa2, 0x1a, 0x8b, 0xc9,
                0xff, 0x9d, 0x5e, 0x14, 0xe2, 0xee, 0xdf, 0xda,
                0x17, 0x40, 0x45, 0x46, 0xca, 0xbb, 0x79, 0x7d,
                0xad, 0x92, 0x8e, 0x0c, 0xdd, 0x31, 0x8a, 0xbe,
                0xd0, 0x33, 0x85, 0xd7, 0xe5, 0xd5, 0xf1, 0xbd,
                0x5d, 0x49, 0x7d, 0x4f, 0x1f, 0x8c, 0x12, 0x0f,
                0x39, 0x50, 0xcd, 0x21, 0x92, 0x0e, 0xac, 0x71,
                0x6f, 0xae, 0xfb, 0x08, 0x83, 0xa1, 0x75, 0x55,
                0x3d, 0x7a, 0x65, 0xc6, 0x82, 0xc0, 0x28, 0x1b,
                0xaa, 0x3f, 0xda, 0x9f, 0x32, 0xf9, 0x6f, 0x02,
                0x74, 0xa3, 0xc7, 0x77, 0x7e, 0x28, 0x13, 0x98,
                0x76, 0x86, 0x02, 0x0f, 0x3c, 0xed, 0x86, 0xec,
                0x18, 0xb5, 0xeb, 0x05, 0x80, 0x2f, 0x63, 0x30,
                0x5a, 0xd0, 0xa4, 0x4f, 0x31, 0x42, 0x04, 0x93,
                0xf4, 0xd7, 0xcf, 0x82, 0x59, 0x56, 0x48, 0x06,
                0x43, 0x02, 0x10, 0x4f, 0x54, 0xa7, 0xac, 0xb5,
                0x01, 0x1a, 0x49, 0x49, 0xef, 0x27, 0xd1, 0x5c,
                0x5e, 0xd9, 0x4e, 0xbf, 0x15, 0x03, 0x23, 0xa4,
                0x4a, 0x6a, 0x10, 0x73, 0xf6, 0xf6, 0x92, 0x60,
                0x4c, 0xe8, 0x73, 0x41, 0x43, 0xd9, 0x77, 0xdb,
                0x77, 0x16, 0x61, 0x35, 0xad, 0x48, 0x65, 0x64,
                0x4a, 0xa1, 0x2e, 0x39, 0x06, 0x20, 0xd6, 0x87,
                0xec, 0xdc, 0xc2, 0x34, 0xca, 0xe9, 0xaf, 0x8e,
                0xc4, 0xbd, 0x66, 0xf1, 0x8a, 0x61, 0x5e, 0x50,
                0x00, 0xb4, 0xc0, 0xfa, 0xeb, 0xea, 0x69, 0xeb,
                0x31, 0x1e, 0x29, 0xa5, 0xef, 0x19, 0x21, 0x1c,
                0x2f, 0x96, 0xa5, 0x25, 0xf0, 0x36, 0x90, 0x30,
                0x7b, 0x1b, 0x2f, 0xbb, 0xf8, 0x81, 0xef, 0x73,
                0xfa, 0x4f, 0x06, 0x74, 0x2b, 0x62, 0xbd, 0xd6,
                0x96, 0x92, 0xec, 0x7d, 0x05, 0x43, 0xdc, 0xff,
                0x65, 0xe4, 0xb6, 0xa5, 0x0e, 0xc8, 0xda, 0xd6,
                0x8c, 0x66, 0xb7, 0xda, 0x11, 0x87, 0x96, 0xde
            }
        },
        {
            std::string("0123456789ABCDEF0123456789ABCDEF0123456789ABCDEF012345"
                        "6789ABCDEF0123456789ABCDEF0123456789ABCDEF0123456789AB"
                        "CDEF0123456789ABCDEF0123456789ABCDEF0123456789ABCDEF01"
                        "23456789ABCDEF0123456789ABCDEF0123456789ABCDEF01234567"
                        "89ABCDEF0123456789ABCDEF0123456789ABCDEF"),
            {
                0x41, 0x45, 0x53, 0x03, 0x00, 0x00, 0x1b, 0x43,
                0x52, 0x45, 0x41, 0x54, 0x45, 0x44, 0x5f, 0x42,
                0x59, 0x00, 0x61, 0x65, 0x73, 0x63, 0x72, 0x79,
                0x70, 0x74, 0x20, 0x34, 0x2e, 0x30, 0x2e, 0x30,
                0x2e, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05,
                0xdb, 0x48, 0xa5, 0xdd, 0x11, 0x81, 0x1f, 0xfc,
                0xb6, 0x54, 0xb4, 0x2e, 0x6c, 0xb7, 0x1c, 0xa7,
                0x30, 0x9e, 0x6c, 0x0c, 0x1e, 0xc5, 0xdc, 0x42,
                0xb3, 0xd6, 0x94, 0xc3, 0xc1, 0xe7, 0x77, 0x6a,
                0x78, 0xf7, 0xa3, 0x1d, 0x69, 0xfb, 0xc3, 0x8e,
                0xa8, 0x60, 0x8a, 0x7b, 0xe3, 0xf1, 0x0f, 0xea,
                0xdc, 0x7d, 0x5e, 0x11, 0xa9, 0x32, 0x5e, 0xa5,
                0x77, 0xb2, 0x74, 0x40, 0x40, 0x11, 0x39, 0x39,
                0xb8, 0x15, 0xa0, 0x0e, 0x6d, 0xd8, 0xbe, 0xd7,
                0x7c, 0xb7, 0x27, 0xb7, 0x95, 0x2a, 0x78, 0xa0,
                0x29, 0x8a, 0xba, 0x48, 0x3b, 0x8b, 0x98, 0xbc,
                0x36, 0x9f, 0xf9, 0x8a, 0xcc, 0x31, 0xd5, 0x7e,
                0x03, 0x1b, 0x6c, 0xf6, 0x2d, 0xc8, 0x68, 0x98,
                0x11, 0x97, 0x25, 0x30, 0x99, 0x36, 0xc5, 0x13,
                0xe0, 0x31, 0x08, 0xaf, 0x42, 0xec, 0x94, 0x13,
                0x0d, 0x9f, 0x2e, 0x18, 0xd4, 0xd7, 0x69, 0x29,
                0x52, 0x2e, 0x1d, 0x69, 0x5e, 0x82, 0x0f, 0x2c,
                0x99, 0xbf, 0xd5, 0x0c, 0x45, 0x5a, 0x74, 0x2c,
                0x08, 0x88, 0xd7, 0xaa, 0x56, 0xc5, 0xc2, 0x78,
                0x00, 0x62, 0xfb, 0x74, 0xfa, 0x8e, 0xb2, 0xb0,
                0xae, 0x27, 0xb9, 0xff, 0xd4, 0xb5, 0xe5, 0x87,
                0x8f, 0x5c, 0xfd, 0x1d, 0xe1, 0xd4, 0xd9, 0xf2,
                0x3c, 0x37, 0xae, 0x3e, 0x86, 0x5d, 0xcd, 0x99,
                0x0f, 0x98, 0x86, 0xdb, 0x68, 0xd6, 0x55, 0x89,
                0x64, 0xd2, 0xf0, 0x2a, 0x1f, 0xc4, 0xa5, 0x09,
                0x4f, 0x64, 0xd4, 0xcd, 0xca, 0xc2, 0x2d, 0xcc,
                0x9c, 0x3d, 0x53, 0xd0, 0x88, 0x9a, 0x93, 0x98,
                0x97, 0x45, 0xb6, 0xfa, 0x50, 0xe5, 0xd0, 0x36,
                0xa8, 0x84, 0x87, 0xc4, 0x85, 0xdf, 0x84, 0x4a,
                0x27, 0xe9, 0xd0, 0x9b, 0xdf, 0x2a, 0xf3, 0x09,
                0x99, 0x17, 0x36, 0xd0, 0xdf, 0x5a, 0x5f, 0xd2,
                0x17, 0x5a, 0x45, 0x96, 0xcd, 0xad, 0xbc, 0x4a,
                0x00, 0xc3, 0x1e, 0x8d, 0x1f, 0x30, 0xf4, 0x0b,
                0xc8, 0x9a, 0x77, 0x2a, 0x38, 0x09, 0x95, 0x9f,
                0x8a, 0x08, 0xba, 0x18, 0x9b, 0x29, 0x2f, 0x20,
                0x49, 0xdb, 0x6f, 0x1b, 0xad, 0x2e, 0x99, 0x89,
                0x8a, 0xfc, 0x49, 0xa0, 0x2a, 0x30, 0x02, 0xbf,
                0x43, 0xa4, 0xe5, 0xda, 0xbe, 0x9a, 0xf1, 0xa8,
                0x2e, 0x73, 0xe8, 0x64, 0x34, 0xbd, 0x74, 0xb7,
                0x55, 0x69, 0x99, 0x99, 0xb8, 0x11, 0xd6, 0x6e,
                0x1e, 0x68, 0xd3, 0xfb, 0x89, 0x78, 0xf2, 0x6e,
                0x1a, 0x1e, 0x5a, 0x14, 0x2f, 0xf9, 0x8a, 0xb3,
                0xe2, 0x78, 0xd8, 0xb5, 0xee, 0x26, 0xfe, 0xee,
                0x74, 0x78, 0x3f, 0x90, 0x0f, 0x46, 0xd2, 0x10,
                0x35, 0xdd, 0x56, 0x31, 0xe4, 0x53, 0xc0, 0x17,
                0xb3, 0xe0, 0x64, 0xd1, 0x1a, 0x7a, 0xa5, 0x65,
                0x52, 0x95, 0xba, 0x7b, 0x69, 0x84, 0x27, 0x56,
                0x5e, 0x9b, 0xec, 0xfc, 0x25, 0xd8, 0x0c, 0x58,
                0x0a, 0x8a, 0x77, 0x1f, 0x4b, 0xe8, 0x22, 0xfd,
                0xfe, 0xf0, 0xfa, 0xbd, 0x94, 0x88, 0x05, 0x37
            }
        },
        {
            std::string("0123456789ABCDEF0123456789ABCDEF0123456789ABCDEF012345"
                        "6789ABCDEF0123456789ABCDEF0123456789ABCDEF0123456789AB"
                        "CDEF0123456789ABCDEF0123456789ABCDEF0123456789ABCDEF01"
                        "23456789ABCDEF0123456789ABCDEF0123456789ABCDEF01234567"
                        "89ABCDEF0123456789ABCDEF0123456789ABCDEF0"),
            {
                0x41, 0x45, 0x53, 0x03, 0x00, 0x00, 0x1b, 0x43,
                0x52, 0x45, 0x41, 0x54, 0x45, 0x44, 0x5f, 0x42,
                0x59, 0x00, 0x61, 0x65, 0x73, 0x63, 0x72, 0x79,
                0x70, 0x74, 0x20, 0x34, 0x2e, 0x30, 0x2e, 0x30,
                0x2e, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05,
                0xc4, 0x22, 0x46, 0x9a, 0xcb, 0xe5, 0x2c, 0x8a,
                0x1f, 0x86, 0x42, 0x41, 0xbf, 0x82, 0x48, 0x7b,
                0xd8, 0xef, 0x9b, 0xe3, 0x67, 0xbd, 0xad, 0xb2,
                0xf8, 0x72, 0xe5, 0xa0, 0xc3, 0x26, 0xf1, 0xfc,
                0x85, 0x78, 0x3f, 0x84, 0xc1, 0xd6, 0x6d, 0x5c,
                0xbe, 0x7e, 0x2e, 0xa4, 0x53, 0x8a, 0x0c, 0x1c,
                0xad, 0xa2, 0xee, 0xfa, 0x70, 0x0b, 0x71, 0xaf,
                0xc4, 0x52, 0xa4, 0xd7, 0x1e, 0xf6, 0x8f, 0xe2,
                0xcc, 0x2d, 0x98, 0x11, 0x9b, 0x17, 0x6f, 0xad,
                0x47, 0x54, 0x04, 0xac, 0x75, 0x83, 0x6a, 0xb2,
                0x0b, 0x63, 0x93, 0x8e, 0x21, 0x70, 0xae, 0x18,
                0xf1, 0xf1, 0x71, 0xef, 0x50, 0xc8, 0xb5, 0x70,
                0xc0, 0x40, 0x0e, 0x9f, 0x92, 0xae, 0x7b, 0xb8,
                0xa1, 0x25, 0x34, 0x81, 0x05, 0xaf, 0x3b, 0x42,
                0xea, 0xb0, 0x85, 0x7e, 0x76, 0x05, 0x75, 0x2b,
                0x97, 0x38, 0x02, 0x5b, 0x9e, 0xa5, 0x76, 0x95,
                0x72, 0xd5, 0x85, 0x5e, 0x57, 0xa0, 0x28, 0xe4,
                0xe9, 0xe7, 0x72, 0xc2, 0xa4, 0x68, 0x6d, 0xbb,
                0x10, 0x61, 0xce, 0x2d, 0xe9, 0xa7, 0xa3, 0x60,
                0x9a, 0x73, 0x10, 0xb7, 0xb5, 0xea, 0xd1, 0xd3,
                0x79, 0x1a, 0x94, 0x13, 0x90, 0x2c, 0xe5, 0x6f,
                0x2d, 0x05, 0xb9, 0x68, 0xa9, 0x8e, 0x16, 0xe1,
                0x0a, 0x70, 0xe5, 0x3d, 0x7e, 0x9a, 0x67, 0xbb,
                0x16, 0x3b, 0xfc, 0xfc, 0xad, 0x56, 0x4a, 0xc0,
                0x44, 0x96, 0xf9, 0x64, 0xb1, 0xd7, 0xb4, 0x8b,
                0x6a, 0x24, 0x79, 0xe8, 0x96, 0x9e, 0xb9, 0x64,
                0xd2, 0x8e, 0x29, 0x3e, 0x33, 0xe7, 0xa2, 0x85,
                0x7d, 0x31, 0xe9, 0x2c, 0x68, 0xed, 0x5d, 0xed,
                0xfd, 0x66, 0xaf, 0x15, 0x0f, 0x49, 0x65, 0x53,
                0xdf, 0x14, 0x6f, 0x52, 0x41, 0x01, 0x2d, 0xc3,
                0x9c, 0x1d, 0xd0, 0x05, 0x3a, 0xd5, 0x95, 0xf5,
                0xe4, 0x5c, 0x22, 0x7d, 0x64, 0x29, 0xbd, 0xa9,
                0xde, 0x98, 0xff, 0xe2, 0x85, 0x7a, 0x4b, 0x98,
                0xf2, 0x6d, 0x43, 0x21, 0x95, 0xd3, 0x8c, 0x49,
                0x3f, 0xa5, 0xc6, 0x1f, 0x4e, 0x9a, 0x93, 0x2e,
                0xc6, 0x61, 0x04, 0x88, 0x24, 0xe6, 0x34, 0x82,
                0xdf, 0x64, 0xd4, 0xbb, 0x18, 0xcb, 0x9a, 0x78,
                0x6f, 0xb1, 0xd4, 0x14, 0x87, 0x72, 0xde, 0xc0,
                0x28, 0x08, 0x31, 0xdc, 0x77, 0xe3, 0x36, 0xd0,
                0x6e, 0xe8, 0x08, 0x32, 0x73, 0x71, 0x9a, 0x1c,
                0x48, 0x7a, 0x3e, 0x35, 0xed, 0x5a, 0xa2, 0x5b,
                0x17, 0xb7, 0x98, 0x81, 0x5a, 0x23, 0x17, 0x1b,
                0x8c, 0x72, 0x79, 0x2e, 0x03, 0x1b, 0x81, 0xa5,
                0xb5, 0x62, 0x53, 0x07, 0xd5, 0x35, 0x36, 0x85,
                0x73, 0xfd, 0x17, 0x68, 0x33, 0xc2, 0x0f, 0x71,
                0x5c, 0x64, 0x40, 0x8d, 0x50, 0x6a, 0x08, 0x1b,
                0x16, 0x87, 0x79, 0x5f, 0xea, 0x77, 0x86, 0x32,
                0xa1, 0xa0, 0x8b, 0xcd, 0xef, 0xbf, 0xa8, 0x22,
                0x53, 0x4b, 0xe9, 0x54, 0x55, 0x06, 0xcb, 0xba,
                0x7e, 0x13, 0x51, 0x87, 0x1a, 0x8f, 0x57, 0x9d
            }
        }
    };

    // Create a Logger object; chance null_ostream to std::cerr if debugging
    NullOStream null_ostream;
    LoggerPointer logger =
        std::make_shared<Logger>(null_ostream, LogLevel::Debug);

    // Iterate over the test vector
    std::size_t counter = 0;
    for (const auto &[plaintext, ciphertext] : tests)
    {
        // Create a Decryptor object
        Decryptor decryptor(logger, std::to_string(counter++));

        // Define a place to put the decrypted text
        std::ostringstream oss;

        // Create an input stream over the ciphertext
        StringStreamBuffer cipher_stream_buffer(ciphertext);
        std::istream iss(&cipher_stream_buffer);

        // Decrypt the ciphertext
        auto result = decryptor.Decrypt(password, iss, oss, 0);
        STF_ASSERT_EQ(DecryptResult::Success, result);

        // Ensure the output string length is the expected length
        STF_ASSERT_EQ(plaintext.size(), oss.str().size());
    }
}
