/*
 *  test_stream_2.cpp
 *
 *  Copyright (C) 2024
 *  Terrapane Corporation
 *  All Rights Reserved
 *
 *  Author:
 *      Paul E. Jones <paulej@packetizer.com>
 *
 *  Description:
 *      This module will test the AES Crypt Engine's ability to decrypt
 *      various known plaintext / ciphertext pairs following stream format 2.
 *
 *  Portability Issues:
 *      None.
 */

#include <vector>
#include <sstream>
#include <string>
#include <terra/aescrypt/engine/decryptor.h>
#include <terra/logger/null_ostream.h>
#include <terra/logger/logger.h>
#include <terra/stf/stf.h>
#include "../string_stream_buffer.h"

using namespace Terra::AESCrypt::Engine;
using namespace Terra::Logger;

// It is assumed that a char and uint8_t are the same size
static_assert(sizeof(char) == sizeof(std::uint8_t));

namespace
{

// Structure to hold plaintext and ciphertext pairs
struct PlaintextCiphertext
{
    const std::string plaintext;
    const std::vector<uint8_t> ciphertext;
};

} // namespace

STF_TEST(TestDecryption, Stream_Format_2)
{
    const std::u8string password = u8"Hello";
    const std::vector<PlaintextCiphertext> tests =
    {
        {
            std::string(""),
            {
                0x41, 0x45, 0x53, 0x02, 0x00, 0x00, 0x18, 0x43,
                0x52, 0x45, 0x41, 0x54, 0x45, 0x44, 0x5f, 0x42,
                0x59, 0x00, 0x61, 0x65, 0x73, 0x63, 0x72, 0x79,
                0x70, 0x74, 0x20, 0x33, 0x2e, 0x31, 0x36, 0x00,
                0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0xad, 0xa7, 0x2b, 0x7c, 0x53,
                0x4d, 0x0b, 0xbc, 0x1f, 0x22, 0x97, 0xd8, 0x95,
                0xc9, 0x0c, 0xbe, 0xea, 0x61, 0x24, 0xff, 0x4e,
                0xf0, 0x93, 0x96, 0x47, 0x7c, 0xae, 0x21, 0x12,
                0x9b, 0xb5, 0x0f, 0x76, 0x51, 0x9a, 0x89, 0xb5,
                0x43, 0x02, 0xb5, 0x2c, 0x15, 0x56, 0x41, 0xd9,
                0x94, 0xe5, 0x60, 0xf2, 0x99, 0x60, 0xb9, 0x45,
                0x1b, 0x1b, 0x4d, 0x01, 0x04, 0x44, 0xa4, 0xbf,
                0x76, 0x18, 0xbe, 0x0d, 0x62, 0xe3, 0xc1, 0x08,
                0xd4, 0x82, 0xfc, 0xef, 0xa3, 0x93, 0x8a, 0xe5,
                0x4f, 0x86, 0x7c, 0x67, 0x51, 0x17, 0x6b, 0x78,
                0x60, 0x2f, 0x4c, 0xab, 0xec, 0x24, 0x32, 0x23,
                0x13, 0xf2, 0x71, 0x00, 0x4a, 0x1c, 0x4f, 0x5b,
                0xe0, 0x2a, 0xea, 0x4f, 0xdd, 0xa8, 0xe9, 0x3c,
                0x20, 0xc1, 0x8c, 0x0e, 0x79, 0x70, 0x3e, 0xb9,
                0xfd, 0x8d, 0x18, 0xe2, 0x59, 0xdd, 0x20, 0xc9,
                0x7d, 0xa4, 0x8c, 0x17
            }
        },
        {
            std::string("0"),
            {
                0x41, 0x45, 0x53, 0x02, 0x00, 0x00, 0x18, 0x43,
                0x52, 0x45, 0x41, 0x54, 0x45, 0x44, 0x5f, 0x42,
                0x59, 0x00, 0x61, 0x65, 0x73, 0x63, 0x72, 0x79,
                0x70, 0x74, 0x20, 0x33, 0x2e, 0x31, 0x36, 0x00,
                0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0xf0, 0xc9, 0x37, 0xa3, 0x4d,
                0x5c, 0x6b, 0xfb, 0x83, 0x28, 0x22, 0x11, 0x64,
                0xe1, 0x1d, 0x5e, 0xcd, 0x0e, 0xb9, 0x52, 0x5c,
                0xe7, 0xf9, 0xd5, 0xa1, 0xf1, 0x8b, 0x8f, 0xad,
                0xe5, 0xfc, 0x1e, 0xa0, 0x87, 0x3f, 0x32, 0x96,
                0x05, 0x82, 0xd7, 0x0f, 0x4f, 0x3f, 0x21, 0x69,
                0xd2, 0xc8, 0x76, 0x18, 0x21, 0x30, 0xad, 0x05,
                0xb1, 0xf4, 0x31, 0xf8, 0xa6, 0x3b, 0x3a, 0x9a,
                0xa1, 0x4f, 0x3c, 0xef, 0x17, 0xc6, 0xd3, 0x99,
                0x96, 0xf7, 0xa3, 0x22, 0x05, 0x78, 0xdb, 0xec,
                0x99, 0x01, 0x8d, 0x49, 0xe1, 0x7f, 0x7a, 0x07,
                0x40, 0x8a, 0x33, 0x01, 0x6b, 0x5e, 0x9a, 0xe1,
                0xe3, 0x9b, 0x5a, 0x73, 0x5d, 0xaa, 0xa5, 0x52,
                0x69, 0xf9, 0x0d, 0xfc, 0x19, 0x5b, 0x4c, 0xa0,
                0xc9, 0x0c, 0x8a, 0x01, 0x30, 0x7d, 0xbb, 0xce,
                0x2b, 0xe9, 0x0c, 0x92, 0x2c, 0x08, 0xbf, 0x46,
                0xb1, 0x89, 0x85, 0xc8, 0x6d, 0xf0, 0xaa, 0x02,
                0x96, 0xef, 0x4a, 0x07, 0xa8, 0xe1, 0x5a, 0xfa,
                0xa7, 0x02, 0xa9, 0xbe
            }
        },
        {
            std::string("012"),
            {
                0x41, 0x45, 0x53, 0x02, 0x00, 0x00, 0x18, 0x43,
                0x52, 0x45, 0x41, 0x54, 0x45, 0x44, 0x5f, 0x42,
                0x59, 0x00, 0x61, 0x65, 0x73, 0x63, 0x72, 0x79,
                0x70, 0x74, 0x20, 0x33, 0x2e, 0x31, 0x36, 0x00,
                0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0xf2, 0xa8, 0x37, 0xcb, 0xa2,
                0x2f, 0x7a, 0x16, 0x7d, 0x2b, 0x66, 0x53, 0xe1,
                0x5a, 0xb6, 0x3e, 0x2c, 0xe7, 0x3a, 0xd3, 0xc2,
                0x58, 0xd2, 0x36, 0x55, 0x18, 0xad, 0x37, 0x4b,
                0x4e, 0x10, 0x81, 0x1c, 0x9d, 0xed, 0x76, 0x56,
                0xf2, 0x18, 0xdf, 0xa0, 0xfa, 0x39, 0x6c, 0xc1,
                0x97, 0x89, 0x72, 0xd2, 0x87, 0xc1, 0xb5, 0x3f,
                0xb1, 0x21, 0x14, 0xe2, 0x59, 0xc4, 0x49, 0x6c,
                0x72, 0x37, 0xea, 0xc3, 0x4b, 0x35, 0xba, 0x22,
                0xbd, 0x7c, 0x75, 0x98, 0xfb, 0x17, 0xf6, 0x7c,
                0x7e, 0x8e, 0x66, 0x56, 0x54, 0x19, 0x0f, 0x06,
                0xaa, 0x4b, 0x11, 0xe5, 0x5f, 0xee, 0x91, 0xaa,
                0x9d, 0x42, 0xcd, 0xe2, 0xdf, 0xd2, 0x0a, 0x7f,
                0x83, 0xf3, 0xf1, 0xfa, 0x26, 0x0b, 0x3b, 0x85,
                0xed, 0xc2, 0x7c, 0x03, 0xda, 0xa0, 0x4d, 0xb7,
                0x43, 0x7a, 0x75, 0x01, 0x10, 0x4f, 0x0f, 0x22,
                0x66, 0x6d, 0xfd, 0x4d, 0x37, 0x9f, 0x23, 0x2b,
                0x77, 0x84, 0x0e, 0x5d, 0xba, 0x23, 0xa8, 0x43,
                0x92, 0xdc, 0x0b, 0xb3
            }
        },
        {
            std::string("0123456789ABCDE"),
            {
                0x41, 0x45, 0x53, 0x02, 0x00, 0x00, 0x18, 0x43,
                0x52, 0x45, 0x41, 0x54, 0x45, 0x44, 0x5f, 0x42,
                0x59, 0x00, 0x61, 0x65, 0x73, 0x63, 0x72, 0x79,
                0x70, 0x74, 0x20, 0x33, 0x2e, 0x31, 0x36, 0x00,
                0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x01, 0xb5, 0x3a, 0x7d, 0xba,
                0x40, 0x38, 0x7c, 0x17, 0x5e, 0x77, 0x2f, 0x95,
                0xd5, 0xac, 0x29, 0x41, 0x5c, 0x13, 0xa6, 0x11,
                0xcf, 0x27, 0xdc, 0xfa, 0xd1, 0x59, 0xc0, 0x70,
                0xa7, 0xd6, 0x71, 0x00, 0x53, 0x47, 0x19, 0xda,
                0x9f, 0x34, 0x2d, 0x7a, 0x37, 0x09, 0xab, 0xdc,
                0x0e, 0x58, 0xb6, 0xbc, 0x0b, 0x92, 0x3d, 0x7a,
                0xc7, 0x9f, 0xac, 0xb9, 0x46, 0xc4, 0x27, 0x30,
                0x4e, 0x35, 0xdc, 0xed, 0xa1, 0xfc, 0x57, 0xea,
                0xa0, 0x20, 0x19, 0x16, 0xea, 0xc5, 0xf3, 0x07,
                0x2e, 0xf2, 0x6c, 0xba, 0x63, 0x11, 0x4d, 0xb0,
                0x2c, 0x3f, 0x93, 0x60, 0x62, 0x3e, 0x93, 0x00,
                0x09, 0x0b, 0x9f, 0xa4, 0xa5, 0x7f, 0xfc, 0xd9,
                0xde, 0xa1, 0x65, 0x01, 0x7f, 0xe9, 0xd7, 0x5a,
                0xd9, 0x67, 0x4f, 0x0f, 0x40, 0x76, 0xc3, 0x6c,
                0x98, 0xcb, 0x69, 0xa8, 0xf4, 0x7e, 0x3c, 0x3a,
                0x5a, 0x7b, 0x3c, 0xdb, 0x20, 0x76, 0xe3, 0xd9,
                0xb2, 0x59, 0xa8, 0x24, 0x5c, 0x7c, 0xcc, 0x8b,
                0x04, 0x99, 0x9d, 0x1b
            }
        },
        {
            std::string("0123456789ABCDEF"),
            {
                0x41, 0x45, 0x53, 0x02, 0x00, 0x00, 0x18, 0x43,
                0x52, 0x45, 0x41, 0x54, 0x45, 0x44, 0x5f, 0x42,
                0x59, 0x00, 0x61, 0x65, 0x73, 0x63, 0x72, 0x79,
                0x70, 0x74, 0x20, 0x33, 0x2e, 0x31, 0x36, 0x00,
                0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0xa9, 0x1c, 0xd7, 0x39, 0xc8,
                0xc6, 0xc1, 0x25, 0x71, 0x61, 0x95, 0x01, 0x85,
                0x1f, 0xc0, 0x9c, 0x90, 0x66, 0xbb, 0x81, 0x25,
                0x95, 0x7c, 0xfd, 0x65, 0x02, 0xc0, 0x62, 0x1e,
                0x20, 0x43, 0x31, 0x1c, 0x68, 0xcf, 0xcf, 0x80,
                0xb3, 0xfa, 0x00, 0xfc, 0xd5, 0xec, 0x7c, 0x21,
                0x6e, 0xf8, 0x4e, 0x80, 0x04, 0x1f, 0xc5, 0x3f,
                0xc5, 0xce, 0xcf, 0xf0, 0xbf, 0x62, 0x58, 0x09,
                0xbe, 0x8d, 0x30, 0x6d, 0xee, 0xf4, 0x36, 0xdf,
                0xf1, 0x4a, 0xca, 0x94, 0xd2, 0xbd, 0x1a, 0x81,
                0x23, 0x8d, 0xde, 0x7d, 0xfe, 0xad, 0xcf, 0xaf,
                0x4d, 0x86, 0xfd, 0xc0, 0x19, 0xbc, 0x77, 0x8f,
                0x16, 0xd1, 0x1b, 0x16, 0x2f, 0x06, 0xc4, 0xd6,
                0x46, 0x2d, 0x99, 0x1a, 0x2b, 0xd7, 0xdf, 0x38,
                0x48, 0x3c, 0x5f, 0x00, 0xae, 0x6b, 0x6d, 0x24,
                0x1d, 0x1a, 0x16, 0xd2, 0xb4, 0x9a, 0x39, 0x70,
                0xee, 0xf9, 0x7b, 0xa2, 0x68, 0xa2, 0x20, 0xd1,
                0x58, 0x13, 0x49, 0xdc, 0xbd, 0x21, 0xdf, 0xed,
                0xeb, 0x1a, 0x35, 0x37
            }
        },
        {
            std::string("0123456789ABCDEF0"),
            {
                0x41, 0x45, 0x53, 0x02, 0x00, 0x00, 0x18, 0x43,
                0x52, 0x45, 0x41, 0x54, 0x45, 0x44, 0x5f, 0x42,
                0x59, 0x00, 0x61, 0x65, 0x73, 0x63, 0x72, 0x79,
                0x70, 0x74, 0x20, 0x33, 0x2e, 0x31, 0x36, 0x00,
                0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0xbb, 0x8b, 0xf6, 0x0a, 0x80,
                0x7a, 0xfd, 0xfe, 0x75, 0xb8, 0x0c, 0x6b, 0x36,
                0xf6, 0x5c, 0x9a, 0xae, 0x51, 0x23, 0x70, 0x37,
                0x37, 0x2a, 0xe9, 0xf6, 0x57, 0x9e, 0x6a, 0xd6,
                0xe0, 0xff, 0xa6, 0xce, 0x34, 0x50, 0x97, 0x4b,
                0x83, 0x48, 0xb9, 0x2b, 0xc1, 0xbb, 0xd7, 0x3f,
                0xfc, 0x3c, 0xb3, 0x9d, 0x4c, 0x31, 0xfe, 0x81,
                0xdd, 0x5e, 0x56, 0xdc, 0xa3, 0x07, 0xa7, 0xde,
                0x0d, 0x5a, 0xdb, 0xbb, 0x93, 0xde, 0x1c, 0xcd,
                0x4d, 0x48, 0xad, 0x94, 0xf8, 0xee, 0x4e, 0xf8,
                0x97, 0xa4, 0x3b, 0x75, 0x45, 0x6f, 0x39, 0x7a,
                0xae, 0x4a, 0xb3, 0x4b, 0x4d, 0x75, 0xb5, 0x9d,
                0x9e, 0x8a, 0x79, 0xe0, 0x3c, 0x0e, 0x7a, 0xc3,
                0x5d, 0x82, 0x67, 0xf1, 0x98, 0x46, 0xfa, 0xbb,
                0x14, 0x4d, 0xbf, 0xde, 0xd4, 0xec, 0x4b, 0xd6,
                0x9b, 0xb9, 0xd6, 0x85, 0xf1, 0x68, 0x5e, 0x32,
                0xb5, 0x8d, 0xf0, 0x01, 0x93, 0xf8, 0x05, 0xaf,
                0xf5, 0xf2, 0xa8, 0x21, 0xc7, 0x40, 0xb9, 0x7e,
                0xff, 0x7b, 0xb8, 0x1d, 0xa7, 0x27, 0x35, 0x9f,
                0x41, 0x1a, 0xe9, 0x1b, 0xe2, 0x7d, 0x48, 0xbf,
                0xfd, 0x1f, 0x7d, 0x83
            }
        },
        {
            std::string("0123456789ABCDEF0123456789ABCDE"),
            {
                0x41, 0x45, 0x53, 0x02, 0x00, 0x00, 0x18, 0x43,
                0x52, 0x45, 0x41, 0x54, 0x45, 0x44, 0x5f, 0x42,
                0x59, 0x00, 0x61, 0x65, 0x73, 0x63, 0x72, 0x79,
                0x70, 0x74, 0x20, 0x33, 0x2e, 0x31, 0x36, 0x00,
                0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0xca, 0x86, 0xb3, 0x2d, 0x1e,
                0x52, 0x87, 0xe2, 0xdb, 0xb3, 0x28, 0xf1, 0xd7,
                0x51, 0x69, 0x8b, 0xe5, 0x88, 0xc4, 0x4c, 0x36,
                0xc9, 0x1d, 0x65, 0xfe, 0x86, 0xb7, 0x69, 0xa6,
                0x79, 0xce, 0x3a, 0xa9, 0x47, 0x0f, 0x34, 0xf3,
                0xdf, 0x99, 0x0c, 0x23, 0x31, 0x9b, 0x56, 0x99,
                0x0a, 0xb3, 0xd0, 0x3c, 0x20, 0x9b, 0xf5, 0xc3,
                0x6a, 0xb4, 0x9b, 0x14, 0x22, 0x3f, 0xca, 0x5a,
                0x33, 0x36, 0xf2, 0x48, 0x26, 0xf2, 0x66, 0xbf,
                0xba, 0x75, 0xc8, 0x5f, 0x56, 0xfc, 0x4a, 0x52,
                0xaf, 0xaa, 0x74, 0x2c, 0x97, 0x72, 0x8a, 0xa3,
                0x21, 0x57, 0xed, 0xc3, 0xc0, 0x57, 0x29, 0xc6,
                0xb2, 0x18, 0xef, 0x12, 0x52, 0x80, 0xac, 0x9b,
                0x9b, 0x22, 0x8a, 0x24, 0x0c, 0xa4, 0x83, 0xd6,
                0x58, 0xf1, 0x8f, 0xaf, 0x67, 0xac, 0x55, 0xae,
                0x65, 0x49, 0x20, 0x64, 0x0b, 0xae, 0x9a, 0xdf,
                0x5c, 0xda, 0x92, 0x0f, 0xba, 0x87, 0xee, 0x6c,
                0x2e, 0x92, 0x46, 0xa5, 0x02, 0xd6, 0xaf, 0x4f,
                0x94, 0x8b, 0xb0, 0x23, 0x11, 0xaa, 0x60, 0xd0,
                0x20, 0x1d, 0x0e, 0x94, 0xb1, 0xb4, 0x19, 0x7a,
                0x80, 0xcc, 0xe0, 0xd4
            }
        },
        {
            std::string("0123456789ABCDEF0123456789ABCDEF"),
            {
                0x41, 0x45, 0x53, 0x02, 0x00, 0x00, 0x18, 0x43,
                0x52, 0x45, 0x41, 0x54, 0x45, 0x44, 0x5f, 0x42,
                0x59, 0x00, 0x61, 0x65, 0x73, 0x63, 0x72, 0x79,
                0x70, 0x74, 0x20, 0x33, 0x2e, 0x31, 0x36, 0x00,
                0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x0b, 0x30, 0xc5, 0x97, 0x84,
                0xda, 0xd7, 0xaf, 0x85, 0x2a, 0x35, 0xfe, 0x3a,
                0x96, 0x31, 0x60, 0x87, 0xca, 0x3b, 0x10, 0x21,
                0x69, 0x3e, 0xf1, 0x22, 0xc1, 0x7d, 0xd0, 0x6b,
                0x1e, 0xb6, 0xbc, 0x48, 0xa4, 0x09, 0xc0, 0x5d,
                0x2a, 0xd5, 0x21, 0x76, 0x4c, 0xdd, 0xe7, 0x3e,
                0x9b, 0xd8, 0x44, 0x8c, 0x9e, 0x94, 0x68, 0xc2,
                0x34, 0xe2, 0xa4, 0x3b, 0xbd, 0x30, 0x3b, 0x99,
                0xac, 0xda, 0x43, 0x7a, 0xbe, 0x76, 0x19, 0x9c,
                0x89, 0xb5, 0x88, 0xa3, 0xfb, 0x0e, 0xb0, 0x5e,
                0x56, 0xc9, 0x20, 0x1b, 0xbd, 0x21, 0x97, 0x9d,
                0x90, 0xe0, 0xed, 0x07, 0xb0, 0xc7, 0xbc, 0x20,
                0x9f, 0x83, 0xdb, 0xc9, 0x92, 0xd6, 0x12, 0xc1,
                0x21, 0x79, 0x2e, 0x31, 0x0c, 0xe0, 0xb7, 0x9a,
                0x49, 0x28, 0x1a, 0xcf, 0x7f, 0x6d, 0xcc, 0x21,
                0x36, 0xea, 0x39, 0x16, 0x5a, 0x41, 0x2e, 0x4e,
                0x3e, 0xf2, 0x9b, 0x00, 0xf3, 0x02, 0x7e, 0x26,
                0xfb, 0x76, 0x9f, 0x8a, 0x3b, 0x7f, 0xdb, 0xfc,
                0xfb, 0xca, 0x07, 0x8a, 0x15, 0x40, 0xf9, 0xcc,
                0xd4, 0x4b, 0x1f, 0xb6, 0xc1, 0x2a, 0x81, 0xa3,
                0x13, 0xde, 0xbd, 0x1b
            }
        },
        {
            std::string("0123456789ABCDEF0123456789ABCDEF0"),
            {
                0x41, 0x45, 0x53, 0x02, 0x00, 0x00, 0x18, 0x43,
                0x52, 0x45, 0x41, 0x54, 0x45, 0x44, 0x5f, 0x42,
                0x59, 0x00, 0x61, 0x65, 0x73, 0x63, 0x72, 0x79,
                0x70, 0x74, 0x20, 0x33, 0x2e, 0x31, 0x36, 0x00,
                0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x53, 0x64, 0xec, 0xa0, 0xb9,
                0xfe, 0xc4, 0xa0, 0x51, 0xd6, 0xb6, 0xf3, 0x50,
                0x44, 0xaa, 0x3c, 0x75, 0xf1, 0xd9, 0x0a, 0xf0,
                0xfc, 0xc7, 0xc4, 0xcb, 0xf3, 0x83, 0x8e, 0x08,
                0xb9, 0xdb, 0x7a, 0x28, 0x3b, 0x6e, 0x86, 0xd4,
                0xd5, 0x78, 0x8e, 0xe0, 0x99, 0xde, 0x7a, 0xb6,
                0x8d, 0xaa, 0x56, 0x31, 0x16, 0x6a, 0x72, 0x5e,
                0xe9, 0x6e, 0x37, 0x1a, 0xba, 0xa5, 0x12, 0x16,
                0x54, 0x34, 0x95, 0x04, 0x5f, 0x5c, 0xa8, 0xb3,
                0x06, 0x39, 0x72, 0x2e, 0xec, 0x0f, 0x92, 0x87,
                0x91, 0xa9, 0xa0, 0xbe, 0x1d, 0xfc, 0x75, 0x27,
                0x17, 0x35, 0x6d, 0x85, 0x21, 0xa9, 0xe5, 0x3c,
                0xeb, 0x3d, 0xae, 0xc3, 0x96, 0xad, 0x1b, 0xf1,
                0xdd, 0xbf, 0x3d, 0x14, 0x86, 0xa4, 0x61, 0x0d,
                0x38, 0x71, 0x07, 0x7a, 0x02, 0xa6, 0x6e, 0x4b,
                0x80, 0xd4, 0x90, 0xaa, 0x2c, 0xe8, 0xb8, 0xcd,
                0x4c, 0x83, 0x7d, 0x84, 0x84, 0x3f, 0x13, 0x62,
                0xe9, 0x27, 0xa4, 0xa5, 0xeb, 0x1d, 0xe9, 0x11,
                0x83, 0x69, 0xe9, 0x01, 0x14, 0x6c, 0x4b, 0x0d,
                0x17, 0xfd, 0x49, 0x62, 0x7d, 0x56, 0xbb, 0x51,
                0xc9, 0xcf, 0x89, 0x70, 0x32, 0xd6, 0x39, 0x6c,
                0x36, 0x55, 0x32, 0x15, 0x26, 0x72, 0x09, 0x5f,
                0x5e, 0xc1, 0x08, 0x20
            }
        },
        {
            std::string("0123456789ABCDEF0123456789ABCDEF0123456789ABCDE"),
            {
                0x41, 0x45, 0x53, 0x02, 0x00, 0x00, 0x18, 0x43,
                0x52, 0x45, 0x41, 0x54, 0x45, 0x44, 0x5f, 0x42,
                0x59, 0x00, 0x61, 0x65, 0x73, 0x63, 0x72, 0x79,
                0x70, 0x74, 0x20, 0x33, 0x2e, 0x31, 0x36, 0x00,
                0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x88, 0xe4, 0x69, 0x53, 0xbd,
                0xf6, 0xf1, 0x31, 0xe6, 0x7b, 0x2f, 0x00, 0x86,
                0x4b, 0xfa, 0x64, 0x6b, 0x94, 0x73, 0xcb, 0xd0,
                0x75, 0xe3, 0x00, 0x2c, 0xd5, 0xca, 0x6e, 0xb1,
                0xf6, 0xe6, 0x05, 0xfa, 0x03, 0x13, 0x5b, 0xbc,
                0xc8, 0x8a, 0x9b, 0xad, 0x53, 0x88, 0x16, 0x61,
                0x29, 0x41, 0x2a, 0x1e, 0x55, 0x5e, 0xaa, 0x12,
                0xe9, 0xc8, 0x52, 0x94, 0xdd, 0xb4, 0x11, 0x11,
                0x60, 0xd7, 0x48, 0xdc, 0x6a, 0x8a, 0x8a, 0x88,
                0x05, 0x31, 0xe6, 0x78, 0xbe, 0x88, 0x00, 0xb1,
                0x5b, 0xb9, 0xcd, 0x88, 0x40, 0xc5, 0x6d, 0x07,
                0x38, 0x0c, 0xe9, 0x14, 0x29, 0xc1, 0x82, 0x61,
                0xe0, 0x52, 0xb3, 0xc4, 0x1b, 0xb0, 0xc0, 0x3f,
                0x6f, 0x74, 0x8d, 0x40, 0xb0, 0x76, 0x61, 0x0b,
                0x2f, 0xbc, 0x91, 0xcc, 0xe2, 0x83, 0x5a, 0xa4,
                0x3e, 0x4f, 0x82, 0x03, 0x5b, 0x5b, 0x94, 0xf9,
                0xb1, 0x52, 0x4a, 0x6b, 0x8a, 0x17, 0x0a, 0xf1,
                0x63, 0x5b, 0xff, 0x04, 0x29, 0xfa, 0xf0, 0x16,
                0x64, 0x02, 0xaf, 0x0f, 0x9f, 0x9c, 0xa8, 0x20,
                0x59, 0xdd, 0x3c, 0xef, 0x25, 0x77, 0x73, 0x4e,
                0xd0, 0x02, 0x9c, 0x45, 0x9f, 0x6a, 0x32, 0xfd,
                0x74, 0xb3, 0xfb, 0x6c, 0xbd, 0x2f, 0x09, 0xd0,
                0x83, 0x5a, 0x76, 0x5f
            }
        },
        {
            std::string("0123456789ABCDEF0123456789ABCDEF0123456789ABCDEF"),
            {
                0x41, 0x45, 0x53, 0x02, 0x00, 0x00, 0x18, 0x43,
                0x52, 0x45, 0x41, 0x54, 0x45, 0x44, 0x5f, 0x42,
                0x59, 0x00, 0x61, 0x65, 0x73, 0x63, 0x72, 0x79,
                0x70, 0x74, 0x20, 0x33, 0x2e, 0x31, 0x36, 0x00,
                0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x9c, 0x6a, 0xfd, 0x16, 0xb6,
                0x2e, 0x32, 0x4e, 0x80, 0x86, 0xe1, 0xda, 0x6f,
                0x05, 0x6c, 0x99, 0x87, 0x95, 0xf5, 0x23, 0x4c,
                0xa6, 0x1c, 0xda, 0xfb, 0x4d, 0xda, 0x68, 0xb5,
                0x6c, 0x77, 0xe5, 0x1b, 0x27, 0xca, 0xb0, 0x7a,
                0xa2, 0x6c, 0x41, 0x64, 0xb3, 0x15, 0x29, 0x0d,
                0x4f, 0x4c, 0x3c, 0x24, 0x86, 0x90, 0x08, 0x84,
                0x29, 0xcd, 0x71, 0x93, 0xd2, 0xb6, 0xc0, 0xd0,
                0x44, 0xda, 0x91, 0x90, 0x69, 0x61, 0x81, 0xe1,
                0xcb, 0x3c, 0x7d, 0xc9, 0xe1, 0x33, 0x00, 0x54,
                0x71, 0x5d, 0xcf, 0x7a, 0xe3, 0x3b, 0xf0, 0x42,
                0x1f, 0x40, 0x3d, 0xb7, 0xcf, 0x3e, 0x89, 0x2f,
                0x94, 0x49, 0x4c, 0x22, 0x3b, 0x78, 0x09, 0xa9,
                0x47, 0x7e, 0xe5, 0xf0, 0xf0, 0x0d, 0x6c, 0x9f,
                0xa0, 0xa6, 0x29, 0x41, 0x7f, 0x06, 0x5b, 0x51,
                0xf2, 0xa7, 0x8b, 0x2f, 0x21, 0x29, 0x35, 0xcb,
                0xcc, 0x41, 0x64, 0x0e, 0x24, 0x92, 0xb8, 0xb8,
                0x3d, 0x57, 0x27, 0x19, 0xe1, 0x51, 0xc6, 0x0d,
                0x9f, 0x57, 0x00, 0x00, 0xce, 0x23, 0xf8, 0x40,
                0x83, 0x74, 0x6b, 0x9b, 0xee, 0xd3, 0xdf, 0x7c,
                0xc1, 0x6b, 0xd9, 0xc1, 0xc8, 0xd9, 0x6a, 0x9a,
                0xe0, 0x88, 0x8b, 0x33, 0xe7, 0x13, 0x6b, 0xe9,
                0xe9, 0x4f, 0x1b, 0x40
            }
        },
        {
            std::string("0123456789ABCDEF0123456789ABCDEF0123456789ABCDEF0"),
            {
                0x41, 0x45, 0x53, 0x02, 0x00, 0x00, 0x18, 0x43,
                0x52, 0x45, 0x41, 0x54, 0x45, 0x44, 0x5f, 0x42,
                0x59, 0x00, 0x61, 0x65, 0x73, 0x63, 0x72, 0x79,
                0x70, 0x74, 0x20, 0x33, 0x2e, 0x31, 0x36, 0x00,
                0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x71, 0xbd, 0x99, 0xbd, 0x1b,
                0x10, 0x43, 0x8e, 0x10, 0x7f, 0x0e, 0x5f, 0x43,
                0xe3, 0x13, 0xca, 0x2d, 0x9a, 0xc3, 0x42, 0x9d,
                0x77, 0x77, 0x8d, 0x5c, 0xb4, 0x4d, 0xb8, 0xe2,
                0x8b, 0x68, 0x0b, 0x2b, 0xb1, 0xb9, 0xd3, 0x93,
                0xc1, 0xe0, 0x45, 0xa8, 0x82, 0x80, 0x8d, 0x38,
                0x61, 0x3e, 0x10, 0xb4, 0xa0, 0x23, 0x2f, 0x6e,
                0xc1, 0xef, 0x5e, 0x7a, 0x27, 0x10, 0x7a, 0x15,
                0xa3, 0x55, 0x76, 0x6a, 0xef, 0x4e, 0xf6, 0xe1,
                0x42, 0xdb, 0x0c, 0xb5, 0x0b, 0xf7, 0x62, 0x60,
                0xb1, 0x49, 0x04, 0x22, 0x8e, 0x54, 0xc5, 0xd4,
                0x74, 0x30, 0xc6, 0x49, 0xbd, 0xe1, 0x6a, 0x69,
                0x4c, 0x97, 0xab, 0x79, 0xb5, 0x56, 0x61, 0xce,
                0x2f, 0x91, 0xa3, 0x2c, 0xd8, 0x6c, 0x30, 0x27,
                0x42, 0xbf, 0x70, 0x8f, 0x11, 0xbe, 0x89, 0xb0,
                0xbc, 0xb7, 0x63, 0xc0, 0xd5, 0x1d, 0xf0, 0xc3,
                0xba, 0xed, 0x55, 0x83, 0x71, 0x71, 0x89, 0xce,
                0xb4, 0x1c, 0x15, 0x65, 0x43, 0x9a, 0x73, 0xc9,
                0x7f, 0x6b, 0x33, 0x99, 0x82, 0x0c, 0x8f, 0xfb,
                0x60, 0x7f, 0x57, 0x13, 0x78, 0x66, 0xb4, 0xfa,
                0xca, 0x89, 0x7c, 0x01, 0xb6, 0xac, 0xc5, 0xd5,
                0x9d, 0x1d, 0x19, 0x48, 0x94, 0x40, 0xc9, 0x31,
                0x8b, 0xab, 0x9c, 0x99, 0x5c, 0xb4, 0xac, 0xb9,
                0xe8, 0x8b, 0x5a, 0x56, 0x03, 0x91, 0x8f, 0x18,
                0xaa, 0x18, 0x97, 0x5e
            }
        },
        {
            std::string("0123456789ABCDEF0123456789ABCDEF0123456789ABCDEF012345"
                        "6789ABCDE"),
            {
                0x41, 0x45, 0x53, 0x02, 0x00, 0x00, 0x18, 0x43,
                0x52, 0x45, 0x41, 0x54, 0x45, 0x44, 0x5f, 0x42,
                0x59, 0x00, 0x61, 0x65, 0x73, 0x63, 0x72, 0x79,
                0x70, 0x74, 0x20, 0x33, 0x2e, 0x31, 0x36, 0x00,
                0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x9b, 0xbb, 0xbb, 0x85, 0x24,
                0x7c, 0x8d, 0x10, 0x93, 0xb4, 0x24, 0x03, 0x97,
                0xd5, 0xc5, 0x72, 0x1b, 0x88, 0xb5, 0x6d, 0x5f,
                0xa6, 0xf6, 0x7b, 0x94, 0x1c, 0x91, 0x0e, 0x5f,
                0xfb, 0xe4, 0x6b, 0xf1, 0xd0, 0x15, 0xef, 0x61,
                0x40, 0xf3, 0xae, 0x5c, 0x50, 0xf7, 0x5c, 0x71,
                0x0a, 0x7f, 0x5d, 0x5e, 0x68, 0xba, 0x91, 0xbc,
                0x8f, 0xc5, 0x70, 0xeb, 0xb2, 0x9e, 0x79, 0x7c,
                0x24, 0xbe, 0x0a, 0xf9, 0x78, 0xfa, 0x07, 0xd0,
                0x7f, 0xf7, 0xe6, 0x44, 0x03, 0xb8, 0xe3, 0x8b,
                0x09, 0x44, 0xfd, 0x07, 0x21, 0x8e, 0x03, 0x1a,
                0xcb, 0x4d, 0x3b, 0x19, 0x30, 0x05, 0x94, 0xdd,
                0x64, 0x64, 0xd0, 0x21, 0xd0, 0xab, 0xef, 0xf2,
                0x36, 0xd7, 0x02, 0x69, 0x35, 0xa7, 0x67, 0x7a,
                0x43, 0x43, 0xb4, 0x06, 0x39, 0x43, 0x46, 0x41,
                0x85, 0x09, 0x45, 0x50, 0x3a, 0x65, 0xc7, 0xa8,
                0xa6, 0x0f, 0x97, 0x04, 0x5f, 0xd8, 0x9d, 0x50,
                0x0b, 0x97, 0x4b, 0x21, 0x8f, 0xdb, 0x3c, 0x84,
                0x65, 0x94, 0x32, 0x23, 0x12, 0xb0, 0xa2, 0xdb,
                0x70, 0x05, 0x5e, 0x48, 0x11, 0xb1, 0xb1, 0x46,
                0x94, 0xb0, 0xf7, 0x0f, 0x2f, 0x3c, 0xcd, 0xc7,
                0x92, 0x8f, 0x67, 0x23, 0x61, 0xe3, 0x15, 0x7a,
                0xb6, 0xa9, 0x47, 0x75, 0x69, 0x26, 0xe1, 0x94,
                0xe4, 0x8f, 0x10, 0x04, 0x89, 0x1d, 0xa0, 0x2d,
                0xd5, 0xb5, 0xd6, 0x71
            }
        },
        {
            std::string("0123456789ABCDEF0123456789ABCDEF0123456789ABCDEF012345"
                        "6789ABCDEF"),
            {
                0x41, 0x45, 0x53, 0x02, 0x00, 0x00, 0x18, 0x43,
                0x52, 0x45, 0x41, 0x54, 0x45, 0x44, 0x5f, 0x42,
                0x59, 0x00, 0x61, 0x65, 0x73, 0x63, 0x72, 0x79,
                0x70, 0x74, 0x20, 0x33, 0x2e, 0x31, 0x36, 0x00,
                0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0xed, 0x3b, 0xf0, 0x77, 0x4c,
                0x88, 0x0d, 0x85, 0xe8, 0xba, 0xc2, 0xe5, 0x8e,
                0xea, 0x00, 0x3a, 0x79, 0x1f, 0x37, 0x24, 0xa8,
                0x65, 0x87, 0x4a, 0x82, 0x44, 0xef, 0x51, 0x07,
                0xfd, 0x0e, 0xc3, 0x49, 0x7d, 0x42, 0x2a, 0xb7,
                0xf6, 0xcc, 0x0d, 0xfa, 0x49, 0x9b, 0xae, 0xbf,
                0x05, 0x65, 0xa1, 0xef, 0xb7, 0xc6, 0xb8, 0xda,
                0xd0, 0x8e, 0xdc, 0xc2, 0xfd, 0xb9, 0x63, 0x38,
                0x0e, 0xa9, 0xed, 0x7a, 0x7e, 0x67, 0x5c, 0x86,
                0xde, 0x35, 0xaa, 0x47, 0x5a, 0xd9, 0x58, 0x07,
                0x50, 0x77, 0xa0, 0x5e, 0x7c, 0xf2, 0x0d, 0xd8,
                0x73, 0x9d, 0x2a, 0xd5, 0xdb, 0x8e, 0xed, 0x80,
                0x12, 0xe2, 0x09, 0x1a, 0xb2, 0xd3, 0xf6, 0xe0,
                0x69, 0x74, 0xd4, 0x46, 0x5c, 0x0e, 0xee, 0x1f,
                0x1e, 0x7b, 0x8a, 0x09, 0x7d, 0x07, 0xd3, 0xe1,
                0xb9, 0xaf, 0xbc, 0xb5, 0xe5, 0x15, 0x37, 0xb0,
                0xeb, 0x07, 0x6d, 0x18, 0xbc, 0xd4, 0xf1, 0x8d,
                0x0e, 0x0b, 0xcf, 0x73, 0xf6, 0x82, 0xc4, 0xe6,
                0xd6, 0x60, 0x05, 0x72, 0x20, 0x9b, 0xbd, 0xf5,
                0xf1, 0x12, 0x1f, 0x55, 0x97, 0xdf, 0x86, 0xca,
                0x29, 0x13, 0xab, 0x00, 0x51, 0xbb, 0xee, 0xc0,
                0xf9, 0xed, 0xd2, 0x2e, 0x50, 0x6c, 0x3a, 0xfe,
                0x83, 0x99, 0x29, 0xe1, 0x69, 0x27, 0x59, 0x2a,
                0xec, 0xa6, 0xf1, 0xbb, 0x0f, 0x76, 0x73, 0x15,
                0x58, 0x63, 0x0f, 0x80
            }
        },
        {
            std::string("0123456789ABCDEF0123456789ABCDEF0123456789ABCDEF012345"
                        "6789ABCDEF0"),
            {
                0x41, 0x45, 0x53, 0x02, 0x00, 0x00, 0x18, 0x43,
                0x52, 0x45, 0x41, 0x54, 0x45, 0x44, 0x5f, 0x42,
                0x59, 0x00, 0x61, 0x65, 0x73, 0x63, 0x72, 0x79,
                0x70, 0x74, 0x20, 0x33, 0x2e, 0x31, 0x36, 0x00,
                0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x42, 0x16, 0x56, 0xa0, 0x82,
                0x01, 0x65, 0xed, 0x46, 0x3a, 0xfe, 0x1d, 0xea,
                0xdc, 0x05, 0x2f, 0x6d, 0xf7, 0xd2, 0x54, 0xcf,
                0xe2, 0x45, 0xdd, 0x21, 0xd7, 0xfd, 0x50, 0xaf,
                0x72, 0xc2, 0x60, 0x71, 0x08, 0xbc, 0xe8, 0x73,
                0x54, 0xb4, 0x53, 0x6c, 0x2f, 0x71, 0x41, 0x7d,
                0xa5, 0x2f, 0xef, 0xbd, 0x4b, 0x9a, 0xb5, 0x2d,
                0x0b, 0xa3, 0xe9, 0x4a, 0x46, 0x34, 0x96, 0x12,
                0x1c, 0x28, 0x7a, 0x98, 0x59, 0xaf, 0x8f, 0xce,
                0x32, 0x4d, 0xd5, 0xfe, 0x19, 0x6e, 0xc3, 0x09,
                0x4f, 0xe1, 0xe9, 0xa7, 0xe6, 0xb9, 0x31, 0x71,
                0x08, 0x2a, 0x65, 0x2f, 0x3a, 0x72, 0x1d, 0x3a,
                0x05, 0xc4, 0x89, 0xd8, 0x58, 0x5b, 0x9a, 0x8a,
                0x9d, 0x38, 0xc1, 0x1c, 0xbf, 0xf1, 0x78, 0xf5,
                0xac, 0x84, 0xe7, 0x42, 0x0c, 0x9a, 0x43, 0x0d,
                0xcb, 0x20, 0x60, 0x65, 0x89, 0x7d, 0x98, 0x1f,
                0x21, 0x60, 0x58, 0x0d, 0x0c, 0x11, 0xbe, 0x90,
                0xc4, 0x66, 0x8d, 0xf4, 0x27, 0xb6, 0x58, 0x1b,
                0xca, 0x2c, 0x7a, 0x24, 0xe0, 0x88, 0x3a, 0x1e,
                0x4b, 0x73, 0xa7, 0xcc, 0xa7, 0x2f, 0xe5, 0x44,
                0xa0, 0x52, 0x45, 0xfc, 0x8b, 0x24, 0x54, 0x1e,
                0xb8, 0x84, 0xfe, 0x0a, 0xc8, 0x78, 0xed, 0x83,
                0x2f, 0x59, 0xc4, 0x01, 0x94, 0x4f, 0x85, 0x15,
                0x93, 0x07, 0x29, 0xd1, 0x8e, 0xc7, 0x95, 0xbd,
                0x65, 0x57, 0xea, 0x7e, 0x84, 0x67, 0x5a, 0x9e,
                0xa6, 0x7e, 0xb7, 0x33, 0xe0, 0x5e, 0xbd, 0x9b,
                0xf0, 0x4e, 0x53, 0x2a
            }
        },
        {
            std::string("0123456789ABCDEF0123456789ABCDEF0123456789ABCDEF012345"
                        "6789ABCDEF0123456789ABCDE"),
            {
                0x41, 0x45, 0x53, 0x02, 0x00, 0x00, 0x18, 0x43,
                0x52, 0x45, 0x41, 0x54, 0x45, 0x44, 0x5f, 0x42,
                0x59, 0x00, 0x61, 0x65, 0x73, 0x63, 0x72, 0x79,
                0x70, 0x74, 0x20, 0x33, 0x2e, 0x31, 0x36, 0x00,
                0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x4b, 0x4f, 0x24, 0x4d, 0x65,
                0xc9, 0x6d, 0x5e, 0x5a, 0x44, 0x22, 0x93, 0x41,
                0xf6, 0xa4, 0xe7, 0x2b, 0x2b, 0x20, 0xa7, 0x8b,
                0x34, 0x38, 0x31, 0x93, 0x69, 0xc8, 0x01, 0x22,
                0xc8, 0x90, 0x4d, 0xe3, 0x82, 0x6f, 0x86, 0x27,
                0x39, 0x12, 0xfb, 0xc0, 0x16, 0xf4, 0xa5, 0x54,
                0x38, 0xfb, 0x0c, 0xfc, 0x15, 0x99, 0x1e, 0x75,
                0x6b, 0xbc, 0xff, 0xb6, 0x03, 0x97, 0x93, 0xe2,
                0x41, 0x71, 0x75, 0xfc, 0x67, 0x37, 0x34, 0x5f,
                0x41, 0x91, 0x6f, 0xb0, 0x5e, 0x2f, 0x8b, 0x29,
                0x59, 0x58, 0xc9, 0xfd, 0xf7, 0x1c, 0xae, 0xd6,
                0xf0, 0xe3, 0x63, 0xff, 0x81, 0x85, 0x5c, 0x9b,
                0xa9, 0xf5, 0x2c, 0x6e, 0x7e, 0x3c, 0x5a, 0xaf,
                0xfc, 0xfb, 0x3a, 0x8b, 0x05, 0x1d, 0x97, 0x1f,
                0xe2, 0x06, 0xe2, 0xe2, 0xfa, 0x55, 0x03, 0xeb,
                0x13, 0x02, 0x50, 0xdb, 0xda, 0x6c, 0x28, 0x23,
                0x6a, 0xca, 0x26, 0x66, 0x5d, 0xf4, 0x94, 0x77,
                0x6c, 0xe6, 0xa0, 0x1c, 0x87, 0x18, 0x2d, 0xfc,
                0xf9, 0x6d, 0x74, 0xe3, 0x2a, 0xe7, 0xbb, 0x76,
                0x43, 0x68, 0xc2, 0x7c, 0x04, 0x25, 0x61, 0xe9,
                0x91, 0xaa, 0x32, 0x1a, 0xf6, 0x6b, 0xa6, 0x7d,
                0x8d, 0x41, 0x0f, 0x38, 0x7d, 0x41, 0x63, 0x26,
                0x3b, 0xed, 0x99, 0x0f, 0x42, 0x8b, 0x8f, 0x4d,
                0x6d, 0x46, 0xcf, 0x85, 0x9f, 0xc0, 0xb6, 0x5c,
                0x6a, 0xf3, 0x47, 0x86, 0x9f, 0x66, 0xcd, 0x9e,
                0xb8, 0xf5, 0x58, 0x0f, 0xdf, 0xff, 0x7b, 0x6a,
                0xe3, 0xc2, 0x84, 0x1e
            }
        },
        {
            std::string("0123456789ABCDEF0123456789ABCDEF0123456789ABCDEF012345"
                        "6789ABCDEF0123456789ABCDEF"),
            {
                0x41, 0x45, 0x53, 0x02, 0x00, 0x00, 0x18, 0x43,
                0x52, 0x45, 0x41, 0x54, 0x45, 0x44, 0x5f, 0x42,
                0x59, 0x00, 0x61, 0x65, 0x73, 0x63, 0x72, 0x79,
                0x70, 0x74, 0x20, 0x33, 0x2e, 0x31, 0x36, 0x00,
                0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x56, 0xd2, 0x83, 0xab, 0x80,
                0xb2, 0x10, 0x30, 0xf3, 0xb3, 0xa5, 0x40, 0xc1,
                0x08, 0x30, 0x90, 0x66, 0xdf, 0x81, 0x1e, 0x6a,
                0x8d, 0xd3, 0xe0, 0x1d, 0x46, 0x21, 0x97, 0xf0,
                0xd5, 0x79, 0x63, 0xcd, 0x51, 0x54, 0x17, 0xc4,
                0x28, 0x4d, 0x93, 0x3a, 0x90, 0xb5, 0x0a, 0xbf,
                0x65, 0xdd, 0x77, 0x80, 0x40, 0x8e, 0x1c, 0x00,
                0x55, 0xb4, 0x66, 0x39, 0x4c, 0xde, 0x24, 0x4a,
                0xa6, 0x1f, 0xc9, 0xca, 0xdc, 0xbf, 0x38, 0xdc,
                0x60, 0x20, 0xf5, 0x1f, 0x6b, 0x9d, 0x3d, 0x11,
                0x8b, 0xcb, 0x72, 0xf5, 0xf3, 0x2f, 0x60, 0x9e,
                0x65, 0xc9, 0xc6, 0x39, 0x8a, 0xd6, 0x1e, 0x25,
                0x20, 0x3d, 0x7a, 0xb0, 0x71, 0xc8, 0x4f, 0xcf,
                0xb2, 0x32, 0x43, 0x3b, 0x9d, 0x7a, 0xd2, 0xe7,
                0x52, 0xba, 0x66, 0x35, 0x3a, 0x0f, 0x93, 0xcf,
                0x3c, 0xc8, 0xcd, 0x83, 0x60, 0x05, 0xdd, 0xe6,
                0x05, 0x11, 0xbe, 0x07, 0x18, 0xd3, 0x83, 0xea,
                0x83, 0x4e, 0x5e, 0x91, 0x18, 0x94, 0x7b, 0xd5,
                0xfa, 0x4f, 0x9a, 0x2a, 0xa2, 0x72, 0x58, 0x87,
                0x5f, 0xdc, 0x1e, 0xaa, 0xd3, 0x58, 0x09, 0xa8,
                0xb5, 0xa8, 0xa7, 0xf4, 0x24, 0x22, 0x06, 0xe8,
                0x49, 0x97, 0x31, 0x79, 0xd5, 0x67, 0x0c, 0x85,
                0x45, 0xbc, 0x15, 0x00, 0x44, 0x12, 0xdd, 0xbd,
                0x94, 0xa6, 0xb6, 0x2d, 0x7f, 0x65, 0xff, 0x4f,
                0x8c, 0x41, 0xf8, 0xd9, 0xec, 0x05, 0xb5, 0xc9,
                0x44, 0x00, 0x2e, 0x3c, 0xb9, 0x36, 0x8b, 0xe9,
                0x96, 0xb7, 0x80, 0xc8
            }
        },
        {
            std::string("0123456789ABCDEF0123456789ABCDEF0123456789ABCDEF012345"
                        "6789ABCDEF0123456789ABCDEF0"),
            {
                0x41, 0x45, 0x53, 0x02, 0x00, 0x00, 0x18, 0x43,
                0x52, 0x45, 0x41, 0x54, 0x45, 0x44, 0x5f, 0x42,
                0x59, 0x00, 0x61, 0x65, 0x73, 0x63, 0x72, 0x79,
                0x70, 0x74, 0x20, 0x33, 0x2e, 0x31, 0x36, 0x00,
                0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x5a, 0x63, 0xe7, 0x20, 0x8d,
                0xc2, 0x38, 0xd8, 0x51, 0x2d, 0x5a, 0xa1, 0xcc,
                0xc5, 0xc0, 0x39, 0x90, 0xa1, 0x2a, 0xa7, 0xe4,
                0x25, 0x03, 0xfc, 0xc2, 0xb8, 0x21, 0x98, 0xd7,
                0x28, 0xe9, 0x9d, 0xfe, 0x3f, 0xdd, 0xdd, 0x71,
                0xc2, 0x34, 0x74, 0x88, 0x08, 0x64, 0x37, 0x8b,
                0xd1, 0xec, 0xef, 0xf8, 0x37, 0xa8, 0x95, 0x49,
                0x9f, 0x86, 0x02, 0xbc, 0xa9, 0x8e, 0xff, 0x4a,
                0xec, 0xa9, 0x0a, 0x35, 0x36, 0x0d, 0x95, 0x0d,
                0x95, 0x5d, 0xb4, 0xc7, 0x18, 0x2a, 0x3f, 0x9f,
                0x50, 0x1e, 0x1f, 0xa0, 0xc8, 0x4b, 0x7e, 0xcb,
                0x49, 0xb6, 0xba, 0x3f, 0x38, 0x7e, 0xdd, 0xbe,
                0x9a, 0x5a, 0x23, 0xe6, 0x54, 0x9c, 0xa8, 0xf6,
                0x1c, 0xe1, 0x7e, 0x04, 0x71, 0x5c, 0x58, 0xf7,
                0xd7, 0xe5, 0x53, 0xde, 0x93, 0x2d, 0x75, 0xd1,
                0xf8, 0x7f, 0x0e, 0x91, 0x9f, 0x2c, 0xb6, 0xdc,
                0xf7, 0x2c, 0x98, 0x37, 0x43, 0xe5, 0x97, 0xc2,
                0xca, 0xca, 0xe5, 0xd5, 0x4f, 0x62, 0x28, 0x85,
                0xcc, 0xfa, 0x08, 0x7a, 0xf8, 0x7e, 0x42, 0xaa,
                0x11, 0xf3, 0x9b, 0xae, 0x04, 0x7d, 0x52, 0xad,
                0x56, 0x32, 0x7b, 0x9c, 0xcc, 0x64, 0x1f, 0xb8,
                0xd9, 0xa8, 0x6d, 0x87, 0x7b, 0x78, 0x79, 0x62,
                0xaf, 0x72, 0x68, 0x37, 0x2d, 0x66, 0x5e, 0x6e,
                0x93, 0xef, 0x58, 0x72, 0x8d, 0xe2, 0xc3, 0x20,
                0xfb, 0x5b, 0xf0, 0x01, 0xad, 0xf9, 0x2d, 0xbf,
                0x40, 0x01, 0x19, 0x78, 0x50, 0xc4, 0xf8, 0x8b,
                0x36, 0x62, 0xb4, 0xdd, 0x5d, 0xba, 0x42, 0x61,
                0xbf, 0x88, 0xba, 0xd2, 0x49, 0x30, 0xe0, 0xaa,
                0x38, 0xb9, 0xc5, 0xdd
            }
        },
        {
            std::string("0123456789ABCDEF0123456789ABCDEF0123456789ABCDEF012345"
                        "6789ABCDEF0123456789ABCDEF0123456789ABCDEF0123456789AB"
                        "CDEF0123456789ABCDEF0123456789ABCDEF0123456789ABCDEF01"
                        "23456789ABCDEF0123456789ABCDEF0123456789ABCDEF01234567"
                        "89ABCDEF0123456789ABCDEF0123456789ABCDE"),
            {
                0x41, 0x45, 0x53, 0x02, 0x00, 0x00, 0x18, 0x43,
                0x52, 0x45, 0x41, 0x54, 0x45, 0x44, 0x5f, 0x42,
                0x59, 0x00, 0x61, 0x65, 0x73, 0x63, 0x72, 0x79,
                0x70, 0x74, 0x20, 0x33, 0x2e, 0x31, 0x36, 0x00,
                0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x5e, 0xb1, 0x4d, 0x71, 0xec,
                0x58, 0x75, 0xec, 0xd0, 0x19, 0x39, 0xae, 0xda,
                0x8a, 0x8d, 0x5d, 0x5f, 0xa0, 0xf8, 0xa4, 0x3a,
                0xa6, 0x3a, 0x9e, 0x2a, 0x3e, 0x2c, 0xac, 0x12,
                0x37, 0x4a, 0x32, 0xb6, 0xee, 0x5b, 0xb6, 0xc5,
                0xe4, 0xf8, 0x57, 0xb5, 0x15, 0x10, 0x3a, 0x2d,
                0xb9, 0x58, 0xd3, 0x03, 0x08, 0x6e, 0x19, 0x8d,
                0x82, 0xee, 0x11, 0x9e, 0x0d, 0xef, 0x79, 0x53,
                0x9c, 0x61, 0xc2, 0x7e, 0x2f, 0x14, 0x19, 0x5e,
                0x28, 0xe2, 0xdf, 0x3a, 0x27, 0xfd, 0xe6, 0xf2,
                0xb7, 0x26, 0x4a, 0x5b, 0xd4, 0xa1, 0x76, 0x16,
                0x6f, 0xe6, 0x4b, 0xb8, 0x5f, 0x21, 0x07, 0x33,
                0x9b, 0xfa, 0x6e, 0x84, 0xb5, 0xbb, 0x9b, 0xff,
                0x1f, 0xd5, 0x0b, 0xde, 0x26, 0xa9, 0xc8, 0x94,
                0x1c, 0x0a, 0x53, 0x21, 0xd3, 0xa3, 0x44, 0x5b,
                0xec, 0x6a, 0xd8, 0xd9, 0x09, 0xb6, 0xbb, 0xe9,
                0x0a, 0x0d, 0xcf, 0x61, 0xde, 0x66, 0xa8, 0xe2,
                0xde, 0x57, 0x9d, 0x01, 0x9c, 0xa6, 0x03, 0x10,
                0x36, 0xb1, 0xd9, 0xd2, 0x4d, 0xde, 0xca, 0x98,
                0x29, 0x0c, 0xdf, 0x7d, 0xb0, 0x22, 0xf6, 0x90,
                0x16, 0x89, 0x50, 0x4b, 0xd1, 0xef, 0xf8, 0xa8,
                0x67, 0x0d, 0x8f, 0x81, 0x59, 0x8e, 0x7e, 0x41,
                0x72, 0x34, 0x9d, 0xa0, 0x64, 0xf9, 0xe0, 0x68,
                0xe4, 0x07, 0xb3, 0x4c, 0x49, 0x8f, 0x29, 0xef,
                0x79, 0x08, 0xfb, 0x1b, 0x83, 0xdc, 0x88, 0x86,
                0x89, 0x23, 0x4d, 0x62, 0x8c, 0x3c, 0x9b, 0xd3,
                0x86, 0x73, 0x5f, 0xb4, 0xfd, 0x55, 0xac, 0x24,
                0xb5, 0x02, 0x75, 0x76, 0xb5, 0xad, 0xc2, 0xe9,
                0x64, 0xba, 0x5c, 0x4b, 0xd5, 0x5d, 0xb4, 0x21,
                0x42, 0x85, 0x9a, 0x5e, 0xe2, 0x9a, 0x20, 0x6d,
                0xfb, 0x7f, 0x61, 0xc7, 0x68, 0x14, 0x2f, 0x40,
                0x3a, 0x4f, 0xdc, 0x75, 0xa4, 0xca, 0x70, 0x70,
                0xf6, 0x00, 0xa8, 0xdf, 0x5f, 0x48, 0x91, 0x64,
                0xee, 0xe0, 0x07, 0x6c, 0x57, 0x97, 0x47, 0x4b,
                0xe1, 0xa0, 0xa7, 0xa4, 0xd3, 0x6b, 0xb6, 0x84,
                0x91, 0x71, 0x4c, 0x60, 0xee, 0x98, 0xf1, 0x4d,
                0x4b, 0xee, 0x44, 0x18, 0xa2, 0x97, 0x1c, 0xee,
                0x4c, 0x16, 0x98, 0x4f, 0xb3, 0x8a, 0x81, 0x16,
                0x9a, 0x74, 0xe5, 0xf8, 0x22, 0x1d, 0x81, 0x18,
                0x5c, 0xf0, 0x3f, 0x25, 0x9f, 0xaa, 0xb7, 0x31,
                0xf9, 0x10, 0x2c, 0x08, 0x31, 0xb3, 0x59, 0x3e,
                0xd4, 0x84, 0xea, 0x8d, 0x79, 0x96, 0xe6, 0xf5,
                0xf0, 0x2e, 0xc5, 0xb7, 0xf3, 0x88, 0x61, 0x11,
                0x03, 0x08, 0xd7, 0x6b, 0x1d, 0xd7, 0xab, 0xa9,
                0x37, 0x76, 0xe9, 0x0f, 0x8a, 0xd0, 0x24, 0x2e,
                0x86, 0x78, 0x5e, 0x90, 0x78, 0x5c, 0x78, 0x00,
                0x98, 0x9e, 0x81, 0x6e, 0x2f, 0x98, 0xf3, 0x1f,
                0xcb, 0xa4, 0xab, 0xd7, 0x5b, 0xd2, 0x33, 0x0f,
                0x10, 0x58, 0x21, 0xac
            }
        },
        {
            std::string("0123456789ABCDEF0123456789ABCDEF0123456789ABCDEF012345"
                        "6789ABCDEF0123456789ABCDEF0123456789ABCDEF0123456789AB"
                        "CDEF0123456789ABCDEF0123456789ABCDEF0123456789ABCDEF01"
                        "23456789ABCDEF0123456789ABCDEF0123456789ABCDEF01234567"
                        "89ABCDEF0123456789ABCDEF0123456789ABCDEF"),
            {
                0x41, 0x45, 0x53, 0x02, 0x00, 0x00, 0x18, 0x43,
                0x52, 0x45, 0x41, 0x54, 0x45, 0x44, 0x5f, 0x42,
                0x59, 0x00, 0x61, 0x65, 0x73, 0x63, 0x72, 0x79,
                0x70, 0x74, 0x20, 0x33, 0x2e, 0x31, 0x36, 0x00,
                0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x05, 0x92, 0xc3, 0x63, 0x47,
                0xc8, 0xb6, 0x20, 0xf0, 0xf7, 0xdf, 0xbe, 0x23,
                0x0b, 0xce, 0xe5, 0x69, 0x01, 0x58, 0xce, 0xa9,
                0x98, 0xe0, 0x51, 0x43, 0x9a, 0x1e, 0x2c, 0x82,
                0x3f, 0x71, 0x17, 0xab, 0xae, 0x38, 0x86, 0x3c,
                0x1c, 0x86, 0xb1, 0xd2, 0x7d, 0x5b, 0x65, 0xa0,
                0xd5, 0xef, 0xca, 0x68, 0xbd, 0x5f, 0xd1, 0x0b,
                0x76, 0x97, 0x6c, 0xd7, 0x62, 0x12, 0x02, 0xcb,
                0x6a, 0x11, 0x83, 0xe7, 0xcc, 0xfc, 0x89, 0xe1,
                0x60, 0x12, 0x7b, 0xcb, 0x02, 0x3d, 0x64, 0xdd,
                0x2f, 0x90, 0xca, 0x5a, 0x42, 0x65, 0xed, 0x1a,
                0x98, 0x0d, 0x97, 0x13, 0xc1, 0x9f, 0x10, 0xf1,
                0x24, 0x01, 0x00, 0x4f, 0x61, 0x00, 0x99, 0x3d,
                0xff, 0x1e, 0x5d, 0x58, 0x96, 0x29, 0xc4, 0x32,
                0xff, 0xa1, 0xc7, 0xa2, 0x3a, 0x50, 0x39, 0x31,
                0x10, 0xfb, 0xa0, 0x2f, 0xbf, 0x3f, 0xb5, 0x47,
                0x0e, 0x81, 0x03, 0x76, 0xf1, 0xf4, 0x58, 0x34,
                0x13, 0x69, 0x7d, 0xbb, 0x46, 0x42, 0x44, 0x2d,
                0xf3, 0x4b, 0x66, 0xf8, 0xda, 0xdd, 0xd0, 0xea,
                0x62, 0x27, 0xf6, 0x82, 0xa5, 0xcd, 0xd8, 0x27,
                0x82, 0x62, 0x7f, 0x91, 0xcc, 0xd4, 0x59, 0xe4,
                0x65, 0xc9, 0x06, 0x2a, 0x71, 0xed, 0x5f, 0xc7,
                0x30, 0x2e, 0x53, 0x24, 0x28, 0xa9, 0xa0, 0x26,
                0x3e, 0xc6, 0x73, 0xbe, 0x80, 0x6f, 0xfa, 0x0f,
                0xe6, 0xcf, 0x81, 0x76, 0xd9, 0xe2, 0x3a, 0x45,
                0x0a, 0x59, 0x6a, 0x6a, 0x82, 0x8f, 0x43, 0x65,
                0x7c, 0x40, 0x71, 0x09, 0xeb, 0xa6, 0xd5, 0x0f,
                0xcf, 0x29, 0xa2, 0xd2, 0xe3, 0x8a, 0xcc, 0x97,
                0x4c, 0xcb, 0x41, 0xef, 0x21, 0x47, 0xd1, 0x13,
                0x5d, 0x3d, 0x17, 0x04, 0x12, 0x78, 0x1a, 0x0c,
                0xe3, 0x23, 0xc8, 0xeb, 0xbd, 0xa2, 0xfe, 0x22,
                0xf0, 0x6b, 0xb8, 0x67, 0x50, 0x55, 0x86, 0xa0,
                0x70, 0x64, 0xf3, 0x1c, 0xbc, 0xe6, 0x83, 0xf8,
                0x96, 0x7a, 0x0b, 0x75, 0x9e, 0xfc, 0x1e, 0x52,
                0xc6, 0xc2, 0x8f, 0x90, 0x55, 0xba, 0x63, 0xbd,
                0x70, 0xa8, 0x8e, 0x7a, 0xa1, 0xd2, 0xac, 0xad,
                0x9e, 0xba, 0x6c, 0x76, 0xbe, 0xdb, 0xa4, 0xd4,
                0x7a, 0xce, 0xf7, 0x87, 0x23, 0xae, 0x85, 0x71,
                0x94, 0x52, 0x35, 0x16, 0xa4, 0xb6, 0x57, 0x2d,
                0x81, 0xeb, 0xf7, 0xeb, 0x43, 0xc6, 0xb4, 0xe9,
                0xab, 0xb9, 0x7a, 0xbe, 0x93, 0x45, 0x13, 0xd1,
                0x0e, 0xaa, 0x1f, 0x19, 0x34, 0xcc, 0xa8, 0x7e,
                0x6e, 0x52, 0xb7, 0x6e, 0x73, 0xce, 0x48, 0x03,
                0x7e, 0x45, 0x36, 0x6f, 0xe3, 0x48, 0xf6, 0x85,
                0xbb, 0x70, 0x81, 0x00, 0xcf, 0x5e, 0x4b, 0x98,
                0xf4, 0x7e, 0xe4, 0xfe, 0x36, 0xda, 0x98, 0x75,
                0xb4, 0x68, 0xf5, 0x62, 0x9c, 0xa1, 0xd5, 0x24,
                0x8c, 0x20, 0x5c, 0xf8, 0x2d, 0x68, 0xa6, 0x27,
                0xea, 0x82, 0xd0, 0x14
            }
        },
        {
            std::string("0123456789ABCDEF0123456789ABCDEF0123456789ABCDEF012345"
                        "6789ABCDEF0123456789ABCDEF0123456789ABCDEF0123456789AB"
                        "CDEF0123456789ABCDEF0123456789ABCDEF0123456789ABCDEF01"
                        "23456789ABCDEF0123456789ABCDEF0123456789ABCDEF01234567"
                        "89ABCDEF0123456789ABCDEF0123456789ABCDEF0"),
            {
                0x41, 0x45, 0x53, 0x02, 0x00, 0x00, 0x18, 0x43,
                0x52, 0x45, 0x41, 0x54, 0x45, 0x44, 0x5f, 0x42,
                0x59, 0x00, 0x61, 0x65, 0x73, 0x63, 0x72, 0x79,
                0x70, 0x74, 0x20, 0x33, 0x2e, 0x31, 0x36, 0x00,
                0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x27, 0xc2, 0xb1, 0x19, 0xe8,
                0x8c, 0xfb, 0x04, 0xd5, 0x06, 0xab, 0x4f, 0x68,
                0xe8, 0xac, 0x25, 0x6d, 0x98, 0xd1, 0xc4, 0x95,
                0xd4, 0x46, 0x7b, 0x99, 0x65, 0x31, 0xa4, 0xd5,
                0x89, 0x1f, 0xa8, 0xa9, 0x35, 0x79, 0xce, 0x1b,
                0x36, 0x64, 0xe5, 0x03, 0x7a, 0x3a, 0x35, 0x1e,
                0x70, 0x89, 0x1a, 0x8c, 0xf6, 0x33, 0x89, 0x8d,
                0x57, 0x62, 0xbe, 0x82, 0x7b, 0xd3, 0xb9, 0xa3,
                0xd4, 0xfa, 0xe8, 0xd0, 0xab, 0x50, 0x1b, 0x01,
                0x5a, 0x20, 0xab, 0xb6, 0xc6, 0xd1, 0xda, 0x93,
                0x89, 0xa1, 0x90, 0x3f, 0x68, 0x2f, 0x3e, 0xcb,
                0xb9, 0xe4, 0x41, 0xd0, 0x80, 0xa4, 0xf4, 0xf7,
                0xf8, 0x7e, 0x5c, 0x91, 0x7e, 0x02, 0x81, 0x7d,
                0x45, 0xcd, 0x04, 0x88, 0x97, 0x6c, 0x51, 0x8a,
                0x10, 0xa6, 0xde, 0x2f, 0x6d, 0xed, 0xea, 0xd1,
                0x18, 0x29, 0xd0, 0x72, 0x45, 0x11, 0xcb, 0xe7,
                0xb0, 0xce, 0x9a, 0x32, 0x69, 0xc4, 0x82, 0xd4,
                0xfa, 0x6c, 0x27, 0x34, 0x3f, 0x8f, 0x7d, 0xa3,
                0xc1, 0xaf, 0x8b, 0xf0, 0x6e, 0xc6, 0x44, 0x51,
                0xac, 0x4b, 0x51, 0xf2, 0xfd, 0x74, 0xdf, 0x81,
                0x26, 0x03, 0x0c, 0xef, 0x52, 0xe0, 0xcc, 0x79,
                0x90, 0xcd, 0x4c, 0x4b, 0xcf, 0x56, 0x7a, 0xfd,
                0xb9, 0xe4, 0x2b, 0x60, 0xb0, 0x10, 0xf9, 0x06,
                0x58, 0x4d, 0x02, 0x1e, 0xce, 0xff, 0xa3, 0x73,
                0xce, 0x6b, 0x07, 0x0b, 0x9d, 0x82, 0xff, 0x42,
                0x4a, 0xef, 0xfb, 0x85, 0xff, 0x8d, 0xd8, 0x85,
                0x56, 0xd3, 0x7a, 0xc1, 0xf6, 0xe4, 0x4a, 0xe4,
                0x3a, 0x2d, 0x9a, 0xfc, 0x35, 0xd5, 0x3e, 0xd4,
                0x9f, 0x45, 0xbe, 0xb5, 0x2e, 0x43, 0xd1, 0x87,
                0x65, 0xe9, 0x1e, 0xd9, 0x8e, 0xc8, 0x6f, 0xaa,
                0x5f, 0xda, 0x10, 0xd1, 0x8e, 0x62, 0x4d, 0xab,
                0x9f, 0x0e, 0x7e, 0x4f, 0xc8, 0x25, 0x3d, 0x52,
                0x73, 0x44, 0xf9, 0x9d, 0xc2, 0xa5, 0xb2, 0x99,
                0x85, 0x57, 0xb6, 0x37, 0x2d, 0x3e, 0xc8, 0xbb,
                0x42, 0x41, 0x80, 0x67, 0x5a, 0x68, 0x9d, 0x89,
                0x33, 0xfc, 0xce, 0x30, 0x2e, 0xcf, 0xde, 0x91,
                0xdd, 0xcb, 0x20, 0x79, 0x3a, 0x96, 0x3e, 0x4d,
                0xe0, 0xc5, 0xdb, 0x5a, 0x7d, 0xf1, 0x74, 0x80,
                0xf2, 0xe3, 0xd9, 0xa8, 0x27, 0x86, 0xfc, 0x91,
                0x1f, 0x40, 0x40, 0xc7, 0x9a, 0x56, 0x11, 0x04,
                0xa9, 0xac, 0x80, 0x7d, 0xcb, 0xb5, 0xe8, 0x17,
                0x3f, 0x28, 0x48, 0x58, 0x65, 0xc4, 0xe3, 0x4e,
                0x13, 0x16, 0xad, 0x2d, 0x9c, 0x89, 0xa7, 0xd9,
                0x81, 0x74, 0x0a, 0xe0, 0x64, 0x71, 0x91, 0x5f,
                0x45, 0x34, 0x3a, 0x60, 0x8c, 0xf3, 0x06, 0xdf,
                0x86, 0x54, 0xeb, 0x2e, 0x31, 0xda, 0x8c, 0xd2,
                0xf1, 0xec, 0x69, 0x01, 0x86, 0x5e, 0xc6, 0xc2,
                0x00, 0xa2, 0xe3, 0x77, 0xd7, 0x4f, 0x55, 0x1f,
                0x9f, 0xe8, 0x44, 0x29, 0x7c, 0x9b, 0x98, 0xf9,
                0x16, 0x8e, 0xa4, 0x7d, 0x16, 0xbe, 0x90, 0xbf,
                0xba, 0x68, 0x29, 0xf8
            }
        }
    };

    // Create a Logger object; chance null_ostream to std::cerr if debugging
    NullOStream null_ostream;
    LoggerPointer logger =
        std::make_shared<Logger>(null_ostream, LogLevel::Debug);

    // Iterate over the test vector
    std::size_t counter = 0;
    for (const auto &[plaintext, ciphertext] : tests)
    {
        // Create a Decryptor object
        Decryptor decryptor(logger, std::to_string(counter++));

        // Define a place to put the decrypted text
        std::ostringstream oss;

        // Create an input stream over the ciphertext
        StringStreamBuffer cipher_stream_buffer(ciphertext);
        std::istream iss(&cipher_stream_buffer);

        // Decrypt the ciphertext
        auto result = decryptor.Decrypt(password, iss, oss, 0);
        STF_ASSERT_EQ(DecryptResult::Success, result);

        // Ensure the output string length is the expected length
        STF_ASSERT_EQ(plaintext.size(), oss.str().size());
    }
}
