/*
 *  test_stream_1.cpp
 *
 *  Copyright (C) 2024
 *  Terrapane Corporation
 *  All Rights Reserved
 *
 *  Author:
 *      Paul E. Jones <paulej@packetizer.com>
 *
 *  Description:
 *      This module will test the AES Crypt Engine's ability to decrypt
 *      various known plaintext / ciphertext pairs following stream format 1.
 *
 *  Portability Issues:
 *      None.
 */

#include <vector>
#include <sstream>
#include <string>
#include <terra/aescrypt/engine/decryptor.h>
#include <terra/logger/logger.h>
#include <terra/logger/null_ostream.h>
#include <terra/stf/stf.h>
#include "../string_stream_buffer.h"

using namespace Terra::AESCrypt::Engine;
using namespace Terra::Logger;

// It is assumed that a char and uint8_t are the same size
static_assert(sizeof(char) == sizeof(std::uint8_t));

namespace
{

// Structure to hold plaintext and ciphertext pairs
struct PlaintextCiphertext
{
    const std::string plaintext;
    const std::vector<uint8_t> ciphertext;
};

} // namespace

STF_TEST(TestDecryption, Stream_Format_1)
{
    const std::u8string password = u8"Hello";
    const std::vector<PlaintextCiphertext> tests =
    {
        {
            std::string(""),
            {
                0x41, 0x45, 0x53, 0x01, 0x00, 0x59, 0xbd, 0x83,
                0x0f, 0x97, 0x65, 0x74, 0x2a, 0x6d, 0xd1, 0xdf,
                0x33, 0xa0, 0x90, 0x42, 0xb3, 0x87, 0x7f, 0x47,
                0x54, 0xa2, 0x51, 0x69, 0xdf, 0x25, 0x9d, 0xc2,
                0x44, 0x37, 0xe4, 0x6a, 0x7b, 0x5e, 0xaa, 0x2d,
                0x65, 0xf6, 0x1a, 0x33, 0x8d, 0x5a, 0xc1, 0x59,
                0xb7, 0x9c, 0x30, 0x12, 0x9c, 0x2b, 0xbc, 0x10,
                0x91, 0x67, 0x68, 0x70, 0xef, 0xa5, 0x63, 0x1d,
                0x00, 0xce, 0x95, 0x6d, 0x48, 0x41, 0xfa, 0xe4,
                0xc9, 0x70, 0x2f, 0x4f, 0x4a, 0x8d, 0x5f, 0xa8,
                0xf8, 0x43, 0xf4, 0x6a, 0x2c, 0xd9, 0x8e, 0x10,
                0xb8, 0x56, 0x45, 0xe3, 0xc3, 0xdf, 0xd8, 0x11,
                0xa6, 0x21, 0xc5, 0x2d, 0xe7, 0x0d, 0x02, 0x40,
                0x07, 0x83, 0xae, 0x31, 0x1b, 0xfc, 0x24, 0x42,
                0x2b, 0x94, 0xc5, 0x45, 0xf9, 0x97, 0xcd, 0xc2,
                0xaf, 0xdb, 0xb0, 0x80, 0x12, 0x36, 0x27, 0x79,
                0x5e, 0x01, 0xa8, 0xe1, 0x4b, 0xc7
            }
        },
        {
            std::string("0"),
            {
                0x41, 0x45, 0x53, 0x01, 0x00, 0x7a, 0xc5, 0x28,
                0xc2, 0xe0, 0x10, 0x3c, 0x77, 0x72, 0xcc, 0x31,
                0x83, 0xdb, 0xee, 0xfa, 0x72, 0x2f, 0xdc, 0x80,
                0x6c, 0x28, 0x32, 0x65, 0x7b, 0x85, 0x02, 0x0b,
                0xa2, 0xac, 0xda, 0x6b, 0x28, 0x92, 0x01, 0xaf,
                0x3f, 0x62, 0x6b, 0x18, 0x7e, 0xf2, 0xea, 0x19,
                0x94, 0x77, 0xba, 0x7e, 0x98, 0x70, 0xd5, 0xa2,
                0x35, 0x86, 0xb1, 0x6b, 0xff, 0xc1, 0x81, 0xee,
                0xaa, 0x06, 0xeb, 0x57, 0x50, 0xbe, 0x9d, 0x4c,
                0xde, 0xbc, 0x10, 0x54, 0x00, 0x28, 0xcc, 0x68,
                0xbd, 0x90, 0x56, 0xe3, 0x3d, 0xb8, 0x74, 0xd5,
                0xb0, 0x50, 0x2d, 0xab, 0x7b, 0x2c, 0x65, 0x35,
                0xa0, 0xbc, 0x9e, 0x3e, 0x76, 0x3d, 0x8d, 0xb6,
                0x10, 0x26, 0x52, 0x4e, 0x62, 0xa3, 0x51, 0xa5,
                0xef, 0xbf, 0xfb, 0xe0, 0xf0, 0x01, 0xc3, 0x6f,
                0x56, 0x7f, 0xec, 0x3e, 0x50, 0x35, 0xa1, 0x05,
                0xe2, 0x71, 0xb1, 0x7a, 0x7d, 0xc7, 0xa3, 0x6c,
                0x27, 0x4d, 0x47, 0x1b, 0xe0, 0x07, 0xac, 0x4d,
                0xa8, 0xa9, 0xb3, 0x37, 0x39, 0xb4
            }
        },
        {
            std::string("012"),
            {
                0x41, 0x45, 0x53, 0x01, 0x00, 0x7f, 0xae, 0x64,
                0xaa, 0x4e, 0xd5, 0x7c, 0xcc, 0x7a, 0xb5, 0xb9,
                0xc4, 0x1c, 0x31, 0x8f, 0xad, 0xb2, 0xed, 0xf5,
                0xea, 0xfe, 0xcd, 0x3c, 0x4a, 0x52, 0x9f, 0xb1,
                0xb6, 0x89, 0x4c, 0xc6, 0x82, 0xb4, 0x2f, 0xb3,
                0xdc, 0x4b, 0x87, 0x64, 0xf1, 0x0f, 0xeb, 0xc3,
                0x39, 0xac, 0xf9, 0xd8, 0x1a, 0x23, 0xf4, 0x30,
                0xbd, 0x4a, 0x9c, 0xd4, 0x40, 0xb8, 0x39, 0xff,
                0x05, 0x11, 0x96, 0x6c, 0x62, 0x1d, 0x0a, 0x02,
                0xe1, 0x59, 0xc3, 0x13, 0x79, 0xd0, 0x44, 0xef,
                0x90, 0x91, 0x65, 0x51, 0x66, 0x5d, 0x3e, 0xe4,
                0x09, 0x6d, 0x3e, 0x50, 0x9e, 0xb0, 0x2d, 0xb2,
                0x93, 0xb4, 0x0c, 0x04, 0x5e, 0x52, 0xfb, 0x7a,
                0x59, 0xdd, 0x44, 0xae, 0xd6, 0xb0, 0x5d, 0x68,
                0xdc, 0x55, 0x0e, 0xbf, 0x4f, 0x03, 0x97, 0xbf,
                0xaf, 0x2a, 0xa3, 0x34, 0x67, 0x0a, 0x82, 0x26,
                0x88, 0x2d, 0x81, 0xa8, 0xa5, 0xe7, 0x9a, 0x15,
                0xc2, 0x2e, 0xbd, 0x18, 0x4e, 0x84, 0x7a, 0x07,
                0x34, 0x3a, 0xa2, 0x66, 0x8c, 0x76
            }
        },
        {
            std::string("0123456789ABCDE"),
            {
                0x41, 0x45, 0x53, 0x01, 0x00, 0x61, 0x99, 0x0b,
                0x32, 0x23, 0x78, 0xcb, 0x8b, 0x88, 0xb7, 0x4f,
                0x1f, 0x73, 0x9d, 0xe6, 0xa4, 0x28, 0xaf, 0x24,
                0xc0, 0xf2, 0xf4, 0xd1, 0x7e, 0xec, 0x92, 0xbc,
                0x72, 0x8f, 0x45, 0xc2, 0x45, 0xf0, 0xa2, 0xaa,
                0x99, 0x5e, 0xdf, 0xfa, 0xca, 0x3d, 0xb9, 0x5a,
                0xb1, 0x70, 0xff, 0x03, 0x52, 0xff, 0x3b, 0xba,
                0x75, 0x54, 0x4d, 0x06, 0x11, 0x33, 0xfb, 0x66,
                0xef, 0xb9, 0x61, 0xde, 0x6f, 0xea, 0xcb, 0x88,
                0x6a, 0x46, 0x25, 0x93, 0x2b, 0xa4, 0xf0, 0x8c,
                0x38, 0xfd, 0x0c, 0x71, 0xea, 0x3a, 0x31, 0x9f,
                0xc1, 0xcd, 0xd4, 0x38, 0x69, 0xa0, 0x00, 0xe6,
                0x5a, 0x47, 0xd2, 0xab, 0x10, 0xfe, 0xe6, 0x19,
                0x94, 0xc7, 0x79, 0x20, 0x73, 0x21, 0xea, 0xd1,
                0x99, 0xfc, 0x9d, 0xd2, 0x92, 0x0f, 0x21, 0xb3,
                0x4b, 0x1b, 0x33, 0x1d, 0xbe, 0x8a, 0x56, 0xee,
                0x76, 0x14, 0x33, 0x04, 0x57, 0xd9, 0x21, 0x7b,
                0x68, 0xe4, 0x06, 0xa2, 0xa4, 0xd7, 0x8f, 0xee,
                0x21, 0x15, 0xef, 0xdb, 0x4e, 0x31
            }
        },
        {
            std::string("0123456789ABCDEF"),
            {
                0x41, 0x45, 0x53, 0x01, 0x00, 0xb7, 0x7e, 0x14,
                0xc5, 0x06, 0xb3, 0x9d, 0x58, 0xb4, 0xb9, 0xb7,
                0x7e, 0x99, 0xc5, 0x95, 0x85, 0xb1, 0xe5, 0x50,
                0xf2, 0x87, 0x18, 0xc3, 0x9b, 0xec, 0xc3, 0x5e,
                0xab, 0xf7, 0x09, 0x16, 0x40, 0x05, 0x16, 0x4f,
                0xfd, 0xfe, 0x99, 0x72, 0x4c, 0xf6, 0xb5, 0x6a,
                0x39, 0xa1, 0xd9, 0xfd, 0xe0, 0xca, 0x82, 0x3c,
                0xb9, 0x51, 0x5e, 0x0e, 0xda, 0x38, 0x6c, 0x73,
                0x83, 0x47, 0xd0, 0x11, 0xd8, 0x2a, 0xd2, 0x0a,
                0x22, 0x59, 0xfb, 0x45, 0x59, 0xbf, 0x3e, 0x9c,
                0x25, 0x1d, 0xc0, 0xe8, 0x44, 0x2d, 0x2f, 0x11,
                0x0c, 0xe7, 0xf8, 0x5b, 0x32, 0x5f, 0x9d, 0x10,
                0x4e, 0x2c, 0x4e, 0xa0, 0x11, 0x20, 0x99, 0x58,
                0x22, 0x96, 0xbb, 0xad, 0x2e, 0x0a, 0x93, 0x93,
                0xd4, 0xe3, 0x1e, 0xd5, 0x9a, 0x00, 0x4d, 0xcb,
                0x89, 0x3e, 0x49, 0x15, 0x85, 0xb6, 0xe5, 0x4c,
                0x34, 0xa9, 0x1f, 0x13, 0x25, 0xe8, 0xa3, 0xbc,
                0x6d, 0x18, 0x89, 0x34, 0x62, 0x47, 0x63, 0xe5,
                0x14, 0xfc, 0x0b, 0xf4, 0xb5, 0x40
            }
        },
        {
            std::string("0123456789ABCDEF0"),
            {
                0x41, 0x45, 0x53, 0x01, 0x00, 0x91, 0x2d, 0x1e,
                0xcb, 0xed, 0xea, 0x50, 0x10, 0x5c, 0x12, 0xf2,
                0xbb, 0xcd, 0x40, 0x6b, 0x8e, 0xf6, 0xfd, 0x2a,
                0xeb, 0x07, 0x1b, 0x82, 0x61, 0x6a, 0xe7, 0x7a,
                0x9f, 0x14, 0xfd, 0x6e, 0x08, 0xf3, 0x96, 0xeb,
                0x8b, 0xd5, 0x7d, 0x8d, 0xc4, 0x8b, 0x36, 0xfe,
                0x10, 0xfb, 0xeb, 0x41, 0x5d, 0x07, 0x6c, 0xa8,
                0x9b, 0x92, 0xc9, 0x2e, 0x2e, 0xa7, 0x54, 0xef,
                0x17, 0x84, 0xd5, 0xf3, 0xc2, 0x3e, 0xed, 0x3b,
                0xe5, 0x61, 0xff, 0xac, 0xd4, 0x3a, 0x8a, 0x8e,
                0xb1, 0x88, 0xbd, 0x58, 0xd0, 0xa3, 0x9a, 0x50,
                0x08, 0x45, 0x6e, 0x7c, 0x90, 0x40, 0x19, 0xe5,
                0x1b, 0x19, 0x02, 0xdf, 0x02, 0x01, 0xab, 0x61,
                0x50, 0x88, 0x72, 0x10, 0x32, 0x93, 0x19, 0xe0,
                0x6d, 0xa7, 0x0f, 0x40, 0x9d, 0x66, 0xb6, 0xd7,
                0x73, 0x6b, 0x15, 0x8c, 0x27, 0x58, 0x01, 0xca,
                0xb3, 0x1a, 0x13, 0xea, 0x97, 0x01, 0x5b, 0x73,
                0x4e, 0xef, 0xaf, 0x58, 0x9a, 0x58, 0x0a, 0xbb,
                0x11, 0x14, 0xd6, 0x5f, 0x58, 0xc2, 0xf7, 0xd9,
                0xb9, 0xb0, 0xaa, 0x57, 0x16, 0x2a, 0x79, 0xdb,
                0x8d, 0x9b, 0x99, 0xc9, 0x34, 0xeb
            }
        },
        {
            std::string("0123456789ABCDEF0123456789ABCDE"),
            {
                0x41, 0x45, 0x53, 0x01, 0x00, 0xba, 0xb4, 0x14,
                0x5e, 0x2e, 0xb9, 0x13, 0x08, 0x7a, 0xbc, 0x24,
                0x16, 0xe1, 0x95, 0x31, 0x30, 0x7a, 0xc2, 0x3e,
                0xe7, 0x41, 0xeb, 0x19, 0xa7, 0xb2, 0x79, 0x23,
                0xef, 0x1a, 0xe9, 0x29, 0x5e, 0x04, 0x42, 0xec,
                0x20, 0x7e, 0x5a, 0x0a, 0xdc, 0x7e, 0xc4, 0xc3,
                0xa8, 0x59, 0xa9, 0x07, 0xed, 0xe4, 0x40, 0x91,
                0xe2, 0x5a, 0xda, 0x47, 0x24, 0xd6, 0xd1, 0xe3,
                0x9b, 0x43, 0x70, 0x08, 0x50, 0x4a, 0xf7, 0xa3,
                0x25, 0xa3, 0x37, 0x35, 0xc2, 0x09, 0xe1, 0x74,
                0x62, 0x8b, 0xca, 0xa7, 0x29, 0x5c, 0x48, 0x0a,
                0x3f, 0x12, 0xa9, 0x81, 0xc2, 0xd3, 0x6f, 0xbb,
                0x2c, 0x76, 0x33, 0x1a, 0x0a, 0xb1, 0xaa, 0xb3,
                0x8c, 0xa8, 0xe6, 0x5f, 0x3c, 0xbf, 0x3c, 0xd1,
                0x17, 0x28, 0xa5, 0x49, 0x0c, 0xa5, 0xa1, 0xdd,
                0x54, 0xf1, 0xcb, 0xd2, 0x8a, 0xac, 0xbe, 0x36,
                0x77, 0x16, 0x50, 0x6f, 0x99, 0x0f, 0x0f, 0x29,
                0x09, 0x46, 0xd3, 0x5f, 0x16, 0x7f, 0x19, 0x0e,
                0x29, 0xfe, 0xb1, 0x53, 0xe8, 0x59, 0xae, 0x52,
                0x78, 0x24, 0xb5, 0xde, 0xdc, 0x1a, 0x97, 0xde,
                0x03, 0xb6, 0x80, 0xa5, 0xf7, 0xad
            }
        },
        {
            std::string("0123456789ABCDEF0123456789ABCDEF"),
            {
                0x41, 0x45, 0x53, 0x01, 0x00, 0xe6, 0xe6, 0xf5,
                0x16, 0x0c, 0x4f, 0x95, 0xce, 0x1c, 0x30, 0x16,
                0x8d, 0xe0, 0x04, 0x84, 0x8c, 0x32, 0x88, 0xef,
                0x13, 0xa9, 0xe8, 0xb4, 0x33, 0xa4, 0x7e, 0xe0,
                0x09, 0x83, 0x87, 0xf1, 0x11, 0x5d, 0xab, 0x49,
                0xa0, 0x35, 0xe1, 0x1d, 0xda, 0x07, 0x55, 0xbe,
                0x37, 0xce, 0xd9, 0xbc, 0xee, 0x17, 0xd2, 0xce,
                0x00, 0x4c, 0xbd, 0x19, 0xc2, 0x04, 0x34, 0x6e,
                0x0a, 0x4b, 0x83, 0x81, 0x5a, 0x45, 0x7a, 0x36,
                0x93, 0x33, 0x4f, 0x51, 0x82, 0x8a, 0xdd, 0x3b,
                0x61, 0x2a, 0x84, 0xaf, 0x42, 0x1e, 0xca, 0xbb,
                0xea, 0x49, 0x09, 0x3c, 0x95, 0xdf, 0xdc, 0xa0,
                0x3a, 0x69, 0x5a, 0x76, 0x09, 0xec, 0xbd, 0xd8,
                0x9c, 0xb4, 0xca, 0x88, 0xa3, 0xc7, 0x63, 0x1f,
                0x56, 0x36, 0x78, 0x90, 0x53, 0x64, 0x5e, 0xb4,
                0xaa, 0x5d, 0x66, 0xaf, 0xa6, 0xa6, 0x37, 0x00,
                0xfc, 0x79, 0x04, 0xe1, 0xf3, 0x00, 0x2b, 0xb3,
                0xb7, 0x05, 0xf8, 0x29, 0xd7, 0xab, 0x84, 0x8e,
                0x76, 0x4e, 0xc7, 0xb0, 0x33, 0xed, 0x72, 0xbc,
                0x4b, 0x6f, 0xe1, 0x50, 0x90, 0xd1, 0xd0, 0x00,
                0xee, 0xa6, 0xd0, 0xb3, 0xb2, 0x67
            }
        },
        {
            std::string("0123456789ABCDEF0123456789ABCDEF0"),
            {
                0x41, 0x45, 0x53, 0x01, 0x00, 0x26, 0x1b, 0x2c,
                0xc5, 0xb7, 0x46, 0xc3, 0xff, 0x66, 0x8a, 0x5e,
                0x04, 0xd0, 0xd2, 0x75, 0xc2, 0xda, 0xe8, 0xec,
                0xf5, 0xcc, 0x66, 0xa8, 0x4a, 0xb1, 0xd7, 0x72,
                0xc2, 0x77, 0x99, 0xcd, 0x0a, 0x5f, 0x99, 0xa0,
                0xaf, 0xa4, 0x4c, 0x15, 0x47, 0x84, 0xaa, 0xca,
                0x8d, 0xc9, 0x84, 0x76, 0x3e, 0x06, 0xa4, 0x58,
                0xb1, 0x13, 0xe1, 0xe7, 0x2c, 0xf6, 0x80, 0x55,
                0x74, 0xdc, 0xd7, 0xba, 0x14, 0xcc, 0x0e, 0x2e,
                0xc8, 0x3d, 0xf0, 0xd1, 0x6f, 0x1f, 0xf9, 0xac,
                0xa1, 0x2b, 0xd3, 0x94, 0xea, 0x1c, 0x5f, 0x65,
                0x7e, 0x70, 0xee, 0x23, 0xcd, 0xe2, 0x3d, 0xd7,
                0x33, 0xe4, 0x08, 0x5c, 0x9c, 0x50, 0xe0, 0xc4,
                0x1b, 0x66, 0xf8, 0xbb, 0x65, 0x02, 0x85, 0xee,
                0x72, 0x1c, 0xeb, 0xce, 0x4a, 0x6b, 0x6b, 0x03,
                0xbe, 0x84, 0xe5, 0xdf, 0x6a, 0x3a, 0x88, 0x13,
                0x6f, 0x5a, 0x58, 0x75, 0x34, 0xb2, 0xc5, 0xd5,
                0xce, 0xd7, 0x13, 0x9a, 0xca, 0xf5, 0xdc, 0x63,
                0x66, 0x14, 0x1e, 0xcb, 0x95, 0x01, 0x4d, 0x23,
                0x2d, 0x97, 0x69, 0xd3, 0x67, 0x15, 0xc2, 0x39,
                0x5a, 0x2a, 0x7d, 0xa7, 0xea, 0xc4, 0xbe, 0xc0,
                0xfd, 0xce, 0x17, 0x56, 0xe3, 0xe3, 0xd8, 0x62,
                0x3f, 0xe7, 0x02, 0x92, 0x99, 0x2b
            }
        },
        {
            std::string("0123456789ABCDEF0123456789ABCDEF0123456789ABCDE"),
            {
                0x41, 0x45, 0x53, 0x01, 0x00, 0x43, 0x20, 0x3a,
                0x87, 0x70, 0x29, 0x15, 0xa9, 0xba, 0xeb, 0x29,
                0x35, 0xbc, 0xc4, 0x8e, 0xe3, 0x44, 0x24, 0x54,
                0xd7, 0x0b, 0xde, 0x9b, 0x9d, 0xea, 0x0d, 0x2b,
                0x57, 0xc5, 0xbc, 0x32, 0xad, 0x3e, 0xa9, 0x8c,
                0x2a, 0xfe, 0xf5, 0xe6, 0x09, 0xfd, 0x85, 0x75,
                0xc0, 0xce, 0x14, 0x7d, 0xda, 0x5a, 0x8f, 0x4c,
                0x65, 0x29, 0x6b, 0x83, 0x7c, 0x2a, 0x90, 0x5a,
                0x7c, 0xbd, 0xa5, 0x3c, 0x9f, 0x6c, 0xc8, 0x35,
                0x73, 0x35, 0x08, 0x05, 0x19, 0xcd, 0xf1, 0x1d,
                0x82, 0x7f, 0x13, 0xf8, 0xb6, 0x79, 0xfe, 0xf0,
                0xbf, 0xf3, 0x68, 0x98, 0x74, 0xc6, 0xeb, 0x3f,
                0x07, 0x7c, 0x2e, 0x2a, 0xb6, 0xaf, 0x70, 0x27,
                0xe2, 0xa6, 0x49, 0x33, 0x4e, 0xeb, 0xb9, 0x48,
                0xd5, 0xdb, 0xe7, 0x90, 0x10, 0x05, 0x8b, 0x87,
                0x88, 0x80, 0x55, 0x2f, 0x73, 0x3a, 0x97, 0xa6,
                0x1d, 0xf3, 0x65, 0x57, 0x9d, 0x98, 0x13, 0x61,
                0x79, 0x71, 0x55, 0xb3, 0xf9, 0x45, 0xb1, 0xbf,
                0x50, 0x86, 0x21, 0xd2, 0x3c, 0x0f, 0x22, 0x6c,
                0x08, 0xfd, 0x76, 0x83, 0x76, 0x50, 0x87, 0x63,
                0x6d, 0x34, 0xc6, 0xa8, 0x5a, 0xf7, 0xe0, 0xd7,
                0xf5, 0xb3, 0x56, 0x7d, 0x74, 0xf8, 0x4e, 0x3c,
                0x1b, 0xd9, 0x4f, 0x86, 0x4d, 0x55
            }
        },
        {
            std::string("0123456789ABCDEF0123456789ABCDEF0123456789ABCDEF"),
            {
                0x41, 0x45, 0x53, 0x01, 0x00, 0xdc, 0x94, 0x92,
                0x99, 0x3b, 0x39, 0x4d, 0x92, 0x4d, 0x4c, 0xde,
                0xa4, 0x22, 0x62, 0x9d, 0x64, 0x65, 0x58, 0xd5,
                0x97, 0xd7, 0x94, 0xf6, 0xc4, 0xd8, 0xf6, 0x36,
                0xca, 0xf7, 0xe8, 0x4a, 0x2d, 0x85, 0x66, 0xfb,
                0x94, 0xef, 0x79, 0x75, 0xf8, 0x0f, 0xe5, 0x71,
                0xa3, 0x7a, 0x55, 0x53, 0xba, 0x04, 0xf0, 0x44,
                0x92, 0x6d, 0x50, 0x18, 0xd8, 0xa2, 0x3d, 0x44,
                0x29, 0xe7, 0xd4, 0x9a, 0x90, 0x76, 0xa4, 0x2c,
                0xdb, 0xc1, 0xb8, 0x26, 0x22, 0x6e, 0xfe, 0x46,
                0x0c, 0xac, 0x5e, 0x70, 0xb6, 0x42, 0x1d, 0x20,
                0x7c, 0x6b, 0x25, 0x27, 0x1a, 0x6f, 0x9b, 0x42,
                0x78, 0xac, 0x23, 0x83, 0xf6, 0x32, 0xe0, 0x4e,
                0x3a, 0x45, 0xb4, 0x7e, 0xa8, 0x7c, 0x07, 0x1b,
                0x96, 0xe8, 0x57, 0xc3, 0x21, 0x6e, 0x28, 0x21,
                0x8c, 0x72, 0xd8, 0xfc, 0x92, 0xf9, 0x89, 0x50,
                0xef, 0xff, 0xc0, 0xbe, 0x45, 0x65, 0x0a, 0x70,
                0xa8, 0x47, 0x38, 0x4c, 0x3f, 0x2a, 0x51, 0x1c,
                0x20, 0x7e, 0x5e, 0x26, 0x3e, 0x00, 0x74, 0x5d,
                0x6d, 0x1f, 0x3b, 0x64, 0x4c, 0x28, 0xff, 0x5c,
                0x39, 0x01, 0xab, 0x6d, 0xb9, 0xbd, 0x42, 0xb0,
                0x6d, 0xe0, 0xbb, 0xd6, 0xaf, 0xf4, 0x89, 0x3b,
                0xd5, 0xee, 0x5f, 0xdc, 0xd7, 0x6d
            }
        },
        {
            std::string("0123456789ABCDEF0123456789ABCDEF0123456789ABCDEF0"),
            {
                0x41, 0x45, 0x53, 0x01, 0x00, 0xd3, 0xd0, 0x95,
                0x33, 0x12, 0x82, 0x31, 0xa9, 0xfb, 0x91, 0xda,
                0xe7, 0xbc, 0x80, 0x12, 0xb9, 0xfd, 0x73, 0x2d,
                0x82, 0xc8, 0x15, 0x35, 0x7e, 0x3c, 0xbb, 0xb6,
                0x56, 0xe8, 0xfa, 0x8a, 0xde, 0xe6, 0xe6, 0x2a,
                0x10, 0x9c, 0x81, 0x96, 0x73, 0x00, 0xed, 0x65,
                0x91, 0x7f, 0x70, 0x56, 0xbc, 0xb0, 0x1e, 0xb3,
                0xdb, 0xc2, 0x78, 0xff, 0xb6, 0xa4, 0x55, 0x46,
                0x08, 0x30, 0xa1, 0x29, 0x40, 0x6b, 0xbc, 0xeb,
                0x4a, 0xd7, 0x24, 0x89, 0x3a, 0xa9, 0xbe, 0xb9,
                0xd4, 0x76, 0x0e, 0x77, 0x02, 0xea, 0xd9, 0x2a,
                0xd2, 0x7f, 0xd2, 0x7a, 0x0e, 0xf8, 0xff, 0x8a,
                0x51, 0x92, 0xe0, 0x8d, 0x59, 0xce, 0x25, 0x7d,
                0xed, 0xf8, 0x93, 0x26, 0xc6, 0x34, 0x93, 0x80,
                0x8a, 0x4d, 0xf5, 0xac, 0x9a, 0xa9, 0x57, 0xc5,
                0x75, 0x5c, 0xef, 0x13, 0x26, 0x55, 0x77, 0x2d,
                0xbb, 0x5f, 0xa5, 0xdb, 0xc4, 0x7a, 0xdb, 0x2a,
                0xf7, 0x04, 0x6b, 0xc4, 0x14, 0xc7, 0x9d, 0x76,
                0x27, 0x68, 0x64, 0xf8, 0x40, 0xea, 0xc0, 0xf5,
                0x74, 0xb1, 0x59, 0x47, 0x17, 0xb9, 0x2d, 0x2a,
                0x3f, 0x95, 0x13, 0x07, 0x5a, 0x01, 0xb9, 0xc4,
                0x9e, 0x7d, 0xaa, 0xc1, 0x1e, 0x21, 0x51, 0xfc,
                0x4c, 0x07, 0xd4, 0x8e, 0xcc, 0xfe, 0x99, 0xf4,
                0xd8, 0xde, 0x80, 0x82, 0x0d, 0xc8, 0x12, 0x8c,
                0x7d, 0x96, 0xe3, 0xc5, 0x5b, 0xc2
            }
        },
        {
            std::string("0123456789ABCDEF0123456789ABCDEF0123456789ABCDEF012345"
                        "6789ABCDE"),
            {
                0x41, 0x45, 0x53, 0x01, 0x00, 0x05, 0x13, 0x36,
                0xe7, 0x26, 0xfc, 0x0e, 0x6a, 0x7b, 0x41, 0x94,
                0x69, 0xf8, 0xd4, 0x1d, 0xc7, 0x8e, 0x54, 0xf5,
                0x75, 0x09, 0xec, 0xa8, 0xd3, 0xe7, 0xa7, 0x0b,
                0xf7, 0x94, 0x81, 0x2e, 0xff, 0x00, 0xfb, 0x5c,
                0xa9, 0xf2, 0xb4, 0xc1, 0x5f, 0xaf, 0xef, 0x25,
                0x5e, 0x42, 0x4f, 0xb9, 0x36, 0x55, 0xf0, 0x3f,
                0x46, 0xd0, 0xc9, 0xf9, 0x91, 0x1c, 0x22, 0x97,
                0x34, 0x37, 0xec, 0x98, 0xd3, 0xec, 0xb0, 0xaa,
                0xd7, 0xdc, 0x3c, 0x5f, 0x54, 0xa7, 0x9a, 0x43,
                0x51, 0xe3, 0xd0, 0x8b, 0x08, 0xcc, 0x1e, 0xa5,
                0x5f, 0x80, 0x23, 0xc7, 0xc7, 0x6b, 0x81, 0xf2,
                0x05, 0x2a, 0xdc, 0x37, 0xf1, 0xbf, 0x4b, 0x91,
                0xe0, 0x86, 0xc1, 0xf7, 0x76, 0xc7, 0x69, 0xdc,
                0x2c, 0xec, 0x20, 0xe5, 0xcc, 0x64, 0x43, 0xea,
                0xa6, 0x2c, 0xd4, 0xcd, 0x34, 0xcf, 0x8b, 0xc7,
                0x40, 0xef, 0xc6, 0x7b, 0x50, 0xfd, 0xf0, 0x7a,
                0x5d, 0xcb, 0xea, 0xe2, 0xe8, 0xee, 0x3e, 0xcb,
                0x3f, 0xdb, 0xde, 0xa2, 0xc7, 0x79, 0x5d, 0xfa,
                0x3a, 0x9c, 0x27, 0x1f, 0x01, 0x70, 0x82, 0x19,
                0x6c, 0x89, 0x33, 0x07, 0xe8, 0x0f, 0x96, 0x74,
                0xb4, 0x7f, 0x0f, 0xb3, 0x06, 0xb5, 0xb5, 0x6b,
                0xc1, 0xfe, 0x71, 0x7d, 0xb9, 0x44, 0x52, 0xfc,
                0x16, 0x15, 0xa2, 0xb2, 0xeb, 0xe0, 0xb1, 0x86,
                0xd8, 0x94, 0x5b, 0xa6, 0xfe, 0xef
            }
        },
        {
            std::string("0123456789ABCDEF0123456789ABCDEF0123456789ABCDEF012345"
                        "6789ABCDEF"),
            {
                0x41, 0x45, 0x53, 0x01, 0x00, 0x25, 0xfc, 0xc6,
                0xe2, 0x0e, 0xc7, 0xc5, 0x5a, 0xc2, 0xa4, 0x47,
                0x5c, 0x17, 0x2b, 0x2d, 0x88, 0x54, 0x3d, 0x3c,
                0xdf, 0xde, 0xdb, 0x3c, 0x83, 0xf0, 0xb7, 0x89,
                0x09, 0x68, 0x38, 0x29, 0x64, 0xaa, 0xf8, 0x41,
                0x39, 0x59, 0xa0, 0x74, 0xf4, 0xa5, 0x0f, 0x16,
                0x55, 0x8c, 0x97, 0x5e, 0x47, 0xaa, 0xe8, 0xf1,
                0x6b, 0x68, 0x1b, 0x22, 0x57, 0x77, 0x99, 0xd1,
                0x8f, 0xb0, 0xe1, 0xfd, 0x86, 0xf3, 0xaf, 0xcd,
                0xe9, 0x17, 0x42, 0x1d, 0x3f, 0xc5, 0x0e, 0x93,
                0x01, 0x9e, 0x59, 0x33, 0x32, 0x2b, 0xe2, 0x0f,
                0xc0, 0xe9, 0x65, 0xe8, 0xf9, 0x8b, 0x9a, 0xea,
                0x15, 0x69, 0x36, 0xd5, 0xd9, 0xa7, 0x1f, 0xbf,
                0xc8, 0x0b, 0xfe, 0x84, 0x56, 0x80, 0x8d, 0xc1,
                0x23, 0xda, 0x6e, 0xb2, 0x4d, 0xde, 0x60, 0x54,
                0x1d, 0x3d, 0x18, 0xc7, 0xf8, 0x4c, 0x60, 0x70,
                0x3a, 0xa1, 0xbd, 0xf3, 0xcd, 0x8c, 0x80, 0xe3,
                0xd1, 0x73, 0xc6, 0x6d, 0xa1, 0xc0, 0x7d, 0xf3,
                0xc3, 0x5c, 0x98, 0x7a, 0xcb, 0xa7, 0x57, 0xb7,
                0x55, 0xb7, 0xb3, 0x7a, 0x63, 0xf8, 0x2f, 0x78,
                0x19, 0x85, 0x21, 0x86, 0x8c, 0x00, 0x2d, 0x51,
                0xde, 0xcc, 0x50, 0x78, 0x20, 0x38, 0x11, 0x0f,
                0xde, 0xa8, 0xa0, 0x35, 0x05, 0x71, 0xc7, 0x77,
                0xdc, 0x44, 0x2f, 0x5d, 0xa2, 0x93, 0x8a, 0x36,
                0xba, 0xc7, 0x28, 0x19, 0x23, 0x6a
            }
        },
        {
            std::string("0123456789ABCDEF0123456789ABCDEF0123456789ABCDEF012345"
                        "6789ABCDEF0"),
            {
                0x41, 0x45, 0x53, 0x01, 0x00, 0xf8, 0x6f, 0x59,
                0xbc, 0xed, 0xa6, 0xa8, 0x01, 0x67, 0x78, 0xcf,
                0xe0, 0x10, 0x42, 0x9b, 0xd7, 0x4c, 0xa6, 0x14,
                0x0c, 0xe1, 0x1a, 0x32, 0x5c, 0xd8, 0x43, 0xa6,
                0x0c, 0xb1, 0x12, 0xd2, 0x5b, 0xbd, 0x01, 0xd9,
                0x36, 0x50, 0x46, 0x64, 0x98, 0x80, 0xdc, 0xa3,
                0x27, 0x07, 0xa4, 0xe7, 0x74, 0xb8, 0xc2, 0xe0,
                0x61, 0xf4, 0x12, 0x2e, 0x06, 0xea, 0x30, 0x68,
                0x25, 0x5f, 0x2a, 0x37, 0x05, 0x46, 0x0c, 0xca,
                0xf6, 0xe7, 0xbc, 0x7f, 0xdd, 0x95, 0x27, 0x1d,
                0xfe, 0xd9, 0xb6, 0x33, 0x94, 0x08, 0x46, 0xb4,
                0x4e, 0x39, 0x7f, 0xe2, 0x96, 0x1d, 0xf2, 0xe2,
                0x96, 0xd5, 0xf7, 0x0a, 0x4c, 0xcd, 0xf7, 0x7f,
                0x03, 0x6c, 0x2a, 0xdd, 0xd8, 0xe6, 0x02, 0x70,
                0xdd, 0xf9, 0xdf, 0x3c, 0x5e, 0xe3, 0x1a, 0x7c,
                0xde, 0x76, 0xd3, 0xb7, 0x5a, 0x98, 0xdd, 0xed,
                0x8d, 0x9b, 0x9f, 0x34, 0xcb, 0xfd, 0x98, 0xb4,
                0xe7, 0x8d, 0x31, 0xec, 0xcb, 0x78, 0x8b, 0xe1,
                0x34, 0x3b, 0xb9, 0xb8, 0xdf, 0xfd, 0x3b, 0x06,
                0x31, 0xbe, 0x80, 0xad, 0xb7, 0x67, 0x87, 0x1e,
                0xf4, 0x5c, 0xdf, 0xc4, 0x46, 0x66, 0xd5, 0x9a,
                0x11, 0xf9, 0x69, 0x72, 0x05, 0x6b, 0xc8, 0x05,
                0xd0, 0x79, 0x5a, 0xd4, 0xfe, 0x01, 0xd1, 0x3c,
                0x99, 0x86, 0xfb, 0x91, 0x13, 0x77, 0x3d, 0xb8,
                0x86, 0x70, 0x50, 0x17, 0x67, 0x48, 0x6f, 0x42,
                0x5e, 0x52, 0x9e, 0x70, 0xe6, 0x1b, 0x8f, 0xd7,
                0x61, 0xb2, 0xbb, 0xd9, 0x12, 0x88
            }
        },
        {
            std::string("0123456789ABCDEF0123456789ABCDEF0123456789ABCDEF012345"
                        "6789ABCDEF0123456789ABCDE"),
            {
                0x41, 0x45, 0x53, 0x01, 0x00, 0xb5, 0x11, 0xa6,
                0x53, 0x8e, 0xc8, 0xfb, 0xfe, 0x24, 0x08, 0xa9,
                0x23, 0xd1, 0xe7, 0xba, 0xd8, 0x64, 0xbf, 0xee,
                0xa7, 0xa1, 0xaa, 0x37, 0x53, 0x62, 0x4a, 0x1a,
                0x73, 0x9f, 0xe7, 0x81, 0xea, 0x5e, 0xd3, 0x68,
                0xa7, 0xc7, 0x77, 0x67, 0x3a, 0x21, 0x50, 0x9c,
                0x2e, 0x3c, 0xe7, 0xb5, 0x17, 0x14, 0xe4, 0x0a,
                0xa4, 0xea, 0xab, 0x1a, 0x93, 0xe3, 0x29, 0x4e,
                0x06, 0x9a, 0x0c, 0x3a, 0x39, 0xb1, 0x98, 0x01,
                0x09, 0x5b, 0xbd, 0x85, 0xe1, 0x21, 0xb0, 0xe4,
                0x6a, 0x18, 0x35, 0x48, 0xb1, 0xec, 0x86, 0x1b,
                0x0b, 0xcc, 0xe2, 0xb7, 0xed, 0x3d, 0x9b, 0x73,
                0x10, 0x24, 0xad, 0xef, 0xf3, 0xd3, 0x9c, 0x03,
                0x04, 0x20, 0x60, 0x70, 0x4c, 0x0d, 0xab, 0xe6,
                0x6c, 0x53, 0x8b, 0xc3, 0xa0, 0xb0, 0x76, 0x1c,
                0xa1, 0xdd, 0xef, 0x5c, 0xe5, 0xe3, 0x15, 0xe6,
                0xf8, 0xe9, 0x18, 0xbd, 0x12, 0xe4, 0xdd, 0x25,
                0xd6, 0x1f, 0xfe, 0x37, 0x3f, 0x3e, 0x12, 0x13,
                0x22, 0x8d, 0xec, 0x9d, 0x35, 0x41, 0x19, 0x72,
                0x9a, 0x12, 0x2a, 0x5e, 0x5a, 0xde, 0x02, 0x3b,
                0x95, 0x66, 0xf8, 0xfa, 0x00, 0x88, 0xb6, 0xe5,
                0x73, 0xd5, 0xb2, 0x32, 0xb0, 0x41, 0x32, 0xe4,
                0x6f, 0x4f, 0x8a, 0xc6, 0xb3, 0x0f, 0x03, 0x48,
                0x2b, 0xb9, 0xac, 0x03, 0x10, 0x54, 0x0c, 0xa3,
                0xdb, 0xef, 0x9b, 0x0c, 0x75, 0x60, 0xc7, 0x17,
                0x0a, 0xad, 0xdf, 0x00, 0xd8, 0xa0, 0x35, 0xf1,
                0x70, 0xf9, 0xfe, 0x00, 0x50, 0x13
            }
        },
        {
            std::string("0123456789ABCDEF0123456789ABCDEF0123456789ABCDEF012345"
                        "6789ABCDEF0123456789ABCDEF"),
            {
                0x41, 0x45, 0x53, 0x01, 0x00, 0x81, 0x3f, 0x4c,
                0x5c, 0x0d, 0x90, 0x46, 0x7d, 0xeb, 0x21, 0x30,
                0xec, 0x9a, 0x44, 0xa1, 0xd2, 0x33, 0x4c, 0x3f,
                0xa0, 0x7b, 0x0a, 0x62, 0x0c, 0x1a, 0xe8, 0xe2,
                0x03, 0xe8, 0xd2, 0x80, 0xad, 0xa8, 0x68, 0x81,
                0xe2, 0x3b, 0xab, 0xa5, 0xb0, 0x71, 0x83, 0x4f,
                0x39, 0x87, 0xcd, 0xfe, 0x0d, 0xc9, 0x5d, 0x16,
                0xf4, 0x86, 0xc5, 0x6c, 0x5d, 0x06, 0x21, 0xf5,
                0x8d, 0x87, 0x43, 0x6f, 0x95, 0x6b, 0x8f, 0xc2,
                0x9d, 0x7f, 0x84, 0xbd, 0x1e, 0x99, 0xfe, 0x9f,
                0x3d, 0xc6, 0xa9, 0xe8, 0xd7, 0xd0, 0xe9, 0x1c,
                0xa8, 0xb3, 0xb7, 0x27, 0x25, 0x4f, 0x61, 0x27,
                0x07, 0xe7, 0x8e, 0x80, 0x27, 0x28, 0xd3, 0x8f,
                0x83, 0xf4, 0xfc, 0x8c, 0x50, 0x9b, 0x96, 0xd6,
                0x59, 0x73, 0xf9, 0xd3, 0xf0, 0x54, 0x4a, 0xa1,
                0xd3, 0xb9, 0x79, 0x2f, 0x54, 0x8a, 0xfd, 0x4e,
                0x56, 0xc6, 0xa3, 0x21, 0x11, 0xe1, 0xa9, 0x2d,
                0x9e, 0x57, 0x75, 0x7a, 0x15, 0x47, 0x8e, 0x70,
                0xf9, 0x04, 0x69, 0xca, 0x3b, 0x28, 0x7c, 0x22,
                0x67, 0x95, 0x42, 0x85, 0xea, 0xf7, 0x52, 0x79,
                0x84, 0x0f, 0xb0, 0xf4, 0xf0, 0xac, 0xf2, 0x44,
                0x12, 0x5c, 0xd2, 0xce, 0x9f, 0xd7, 0x99, 0xfe,
                0x31, 0xb4, 0x86, 0xc1, 0x27, 0x00, 0x9a, 0x50,
                0x93, 0x6d, 0x26, 0x20, 0x48, 0x33, 0xbd, 0x58,
                0x43, 0xee, 0xce, 0xa8, 0xdb, 0xba, 0x8c, 0xa9,
                0xdb, 0xb6, 0x5e, 0x47, 0x23, 0x9e, 0x7a, 0x9b,
                0x3b, 0xd8, 0x27, 0xd2, 0x4f, 0x0c
            }
        },
        {
            std::string("0123456789ABCDEF0123456789ABCDEF0123456789ABCDEF012345"
                        "6789ABCDEF0123456789ABCDEF0"),
            {
                0x41, 0x45, 0x53, 0x01, 0x00, 0x89, 0xad, 0xd4,
                0xb9, 0x92, 0x30, 0xeb, 0x0f, 0xd0, 0x3f, 0x6f,
                0x7c, 0xbd, 0x08, 0x12, 0xa0, 0xd9, 0x19, 0xdd,
                0xac, 0x39, 0xbd, 0xdf, 0x8a, 0x62, 0xd3, 0xdd,
                0xd9, 0xc0, 0xf8, 0xbe, 0x08, 0xe2, 0xec, 0x63,
                0xd9, 0xff, 0x9f, 0x3f, 0x7c, 0x9f, 0x38, 0x83,
                0x5a, 0xed, 0x99, 0xcd, 0x9a, 0x84, 0x84, 0xd7,
                0xc2, 0xb7, 0xdd, 0x26, 0xe1, 0x6c, 0xcb, 0x93,
                0x7d, 0xba, 0x39, 0x18, 0x7e, 0xb2, 0x5c, 0x14,
                0x35, 0xcb, 0x85, 0xa5, 0x53, 0x59, 0x93, 0xc4,
                0x64, 0x1e, 0x90, 0x03, 0x07, 0x60, 0x32, 0x65,
                0x4a, 0x14, 0x67, 0xd4, 0xbd, 0xab, 0xd3, 0xc0,
                0x8c, 0x10, 0xfa, 0xbe, 0x5b, 0x98, 0x3e, 0xb8,
                0xb8, 0x36, 0x19, 0x20, 0x35, 0x77, 0x0f, 0xec,
                0x59, 0xdd, 0x83, 0x5c, 0x94, 0x1e, 0xaf, 0xb6,
                0x28, 0xd4, 0x35, 0x16, 0x55, 0x25, 0xf0, 0x11,
                0x3b, 0x7f, 0xed, 0x94, 0x45, 0x71, 0x84, 0x34,
                0x10, 0x53, 0x87, 0xd1, 0xc7, 0x59, 0x40, 0x46,
                0x90, 0x02, 0xc8, 0x03, 0x03, 0xca, 0x20, 0x76,
                0x85, 0xe0, 0x24, 0xbb, 0x65, 0x8f, 0xd5, 0x30,
                0x8f, 0x67, 0x90, 0xfc, 0x02, 0xc1, 0x22, 0x14,
                0xc4, 0x98, 0xf5, 0x0d, 0x2a, 0x19, 0xd0, 0xa1,
                0x82, 0x70, 0xe0, 0x47, 0x0e, 0xb1, 0xbc, 0xd6,
                0x06, 0x3c, 0x15, 0x85, 0x47, 0xf4, 0x15, 0x02,
                0xe2, 0xe9, 0x36, 0x78, 0x18, 0x01, 0x62, 0x13,
                0xa0, 0x2c, 0x07, 0x2d, 0xe6, 0xe6, 0xb9, 0xcc,
                0x0f, 0xf1, 0xb9, 0x52, 0xea, 0x4e, 0xf4, 0xcf,
                0xe2, 0x90, 0xdd, 0xd0, 0xcf, 0x94, 0x71, 0x2b,
                0x6f, 0xe4, 0x0d, 0x2f, 0x9b, 0x2e
            }
        },
        {
            std::string("0123456789ABCDEF0123456789ABCDEF0123456789ABCDEF012345"
                        "6789ABCDEF0123456789ABCDEF0123456789ABCDEF0123456789AB"
                        "CDEF0123456789ABCDEF0123456789ABCDEF0123456789ABCDEF01"
                        "23456789ABCDEF0123456789ABCDEF0123456789ABCDEF01234567"
                        "89ABCDEF0123456789ABCDEF0123456789ABCDE"),
            {
                0x41, 0x45, 0x53, 0x01, 0x00, 0x2b, 0x01, 0x15,
                0x7c, 0x42, 0xdb, 0xde, 0x82, 0xbe, 0x9c, 0x9b,
                0x5c, 0x3f, 0xed, 0x57, 0xa6, 0xbe, 0x78, 0x14,
                0x40, 0xaf, 0x08, 0x78, 0x80, 0x13, 0x0d, 0x62,
                0xc1, 0x4f, 0x40, 0x16, 0x1d, 0xd1, 0x8f, 0xf0,
                0x93, 0xfe, 0x87, 0xed, 0x8b, 0xfa, 0xd8, 0xef,
                0x16, 0x5c, 0xe6, 0x14, 0xd1, 0x4e, 0x90, 0x08,
                0x53, 0x40, 0xd9, 0x0c, 0x03, 0x95, 0x4d, 0x5c,
                0x37, 0x44, 0x6d, 0x4a, 0x69, 0xd2, 0xc3, 0xd1,
                0xd9, 0x3b, 0xac, 0xe9, 0x06, 0x88, 0x49, 0x8d,
                0x91, 0x45, 0xe9, 0xc5, 0x34, 0x92, 0x4b, 0x0b,
                0xca, 0x43, 0xc9, 0x51, 0xf9, 0xd8, 0x7e, 0x1a,
                0xa0, 0xe3, 0xcf, 0xc3, 0xaa, 0x93, 0xc9, 0x31,
                0xac, 0x7b, 0x9e, 0xd9, 0x22, 0x13, 0x2a, 0x25,
                0x91, 0x3d, 0x05, 0x3a, 0x66, 0x87, 0x43, 0x90,
                0x73, 0x24, 0x90, 0x03, 0x71, 0xe8, 0xc5, 0xb3,
                0x9d, 0xc1, 0x2b, 0xd9, 0x4e, 0xc6, 0x6c, 0xfa,
                0xe1, 0xc2, 0x0c, 0x25, 0xb0, 0xf1, 0x5f, 0xae,
                0xb5, 0x9a, 0xbd, 0x49, 0x89, 0x3a, 0x1d, 0x10,
                0x93, 0xa4, 0x62, 0x6f, 0xae, 0x07, 0x8b, 0x86,
                0x8c, 0xa0, 0x16, 0x61, 0xfb, 0x6f, 0xfd, 0x49,
                0xa8, 0x73, 0x39, 0xcf, 0x92, 0x45, 0x29, 0x5f,
                0x84, 0xce, 0x5d, 0x22, 0xd6, 0x7b, 0x39, 0x0f,
                0x2d, 0xed, 0x92, 0x42, 0x24, 0xfe, 0x2a, 0x5f,
                0xb0, 0x64, 0xb0, 0xc2, 0x85, 0xfc, 0xd9, 0x59,
                0x78, 0x6b, 0xbd, 0x45, 0x17, 0x1c, 0x2a, 0x89,
                0x9c, 0xa8, 0x2a, 0x5b, 0xc8, 0xb3, 0x3a, 0x60,
                0x0e, 0xc0, 0xf9, 0xeb, 0x7d, 0x1d, 0x3a, 0xd9,
                0x6a, 0x12, 0x84, 0xe2, 0xc0, 0x65, 0xc9, 0xb0,
                0xc8, 0x73, 0x42, 0xf4, 0xa3, 0x8a, 0xe9, 0x42,
                0x5a, 0x61, 0x4f, 0xab, 0xf5, 0x64, 0x05, 0x04,
                0x30, 0xfa, 0x32, 0xc7, 0xc4, 0x48, 0x5c, 0x57,
                0x03, 0x0f, 0xcf, 0x56, 0x61, 0x3f, 0x8a, 0xee,
                0x68, 0x26, 0x2d, 0x33, 0xe5, 0xa1, 0x4d, 0x35,
                0x82, 0x92, 0x3e, 0x52, 0xc3, 0x73, 0x00, 0xf8,
                0xec, 0x10, 0x0f, 0x19, 0x04, 0x33, 0xa1, 0x11,
                0x55, 0x20, 0xa1, 0x0e, 0x6a, 0xa6, 0xda, 0x54,
                0xaa, 0xb2, 0x4d, 0xae, 0x8d, 0x68, 0xae, 0x60,
                0x53, 0xe7, 0xd6, 0x66, 0xfb, 0x25, 0x9f, 0x07,
                0xcb, 0x7f, 0x94, 0xda, 0xf3, 0x16, 0x35, 0x4a,
                0x6e, 0xc6, 0xaa, 0xe0, 0xd2, 0x34, 0x77, 0x49,
                0xd5, 0xca, 0xcb, 0xc5, 0x1e, 0x9e, 0x0c, 0x01,
                0x81, 0xac, 0xec, 0x6d, 0x95, 0x79, 0xf3, 0xcb,
                0x87, 0x8a, 0x66, 0x07, 0xa6, 0x67, 0x6c, 0xa5,
                0xdf, 0x01, 0xce, 0xa4, 0x32, 0x0f, 0x4c, 0xdd,
                0x17, 0x0b, 0x3f, 0x7a, 0x75, 0xe1, 0xa3, 0x54,
                0xc1, 0xaf, 0xac, 0x29, 0x91, 0x38, 0x08, 0x4d,
                0xe8, 0xea, 0x0d, 0x9e, 0xc7, 0xed, 0xa3, 0x21,
                0xb6, 0xab, 0xbe, 0x45, 0x95, 0xc0
            }
        },
        {
            std::string("0123456789ABCDEF0123456789ABCDEF0123456789ABCDEF012345"
                        "6789ABCDEF0123456789ABCDEF0123456789ABCDEF0123456789AB"
                        "CDEF0123456789ABCDEF0123456789ABCDEF0123456789ABCDEF01"
                        "23456789ABCDEF0123456789ABCDEF0123456789ABCDEF01234567"
                        "89ABCDEF0123456789ABCDEF0123456789ABCDEF"),
            {
                0x41, 0x45, 0x53, 0x01, 0x00, 0xa4, 0x17, 0x89,
                0x9f, 0x9a, 0xdf, 0xab, 0x6a, 0xb6, 0x90, 0x33,
                0x59, 0x80, 0x52, 0x9b, 0x7d, 0x57, 0x7c, 0x6b,
                0x3f, 0x68, 0x6f, 0x7a, 0x47, 0xc8, 0xc2, 0x24,
                0xb1, 0x3d, 0x65, 0xe2, 0x7a, 0x32, 0x19, 0x47,
                0x7b, 0x8f, 0x83, 0x95, 0x28, 0x33, 0xc5, 0x97,
                0x91, 0x57, 0x7e, 0xe6, 0x91, 0x26, 0xfa, 0x38,
                0xb4, 0x69, 0x07, 0x87, 0xe9, 0x0f, 0xaf, 0xce,
                0xd5, 0xbb, 0x95, 0x96, 0x4f, 0xbf, 0xff, 0x52,
                0x8b, 0x84, 0xf3, 0x2d, 0x9e, 0xe7, 0xc9, 0xaf,
                0xfd, 0x78, 0x9f, 0xa0, 0xee, 0x1e, 0xcc, 0xba,
                0xa2, 0x31, 0x19, 0x32, 0x67, 0x61, 0xe0, 0xc7,
                0xec, 0xe0, 0xc5, 0xe2, 0x0e, 0x2f, 0x7a, 0xe2,
                0xf8, 0x53, 0x5d, 0x43, 0x0e, 0xbf, 0xb8, 0xc5,
                0x63, 0xd6, 0x58, 0x8a, 0xa7, 0xc5, 0x2a, 0x8a,
                0x00, 0x82, 0xa3, 0x80, 0x29, 0xfe, 0x77, 0x40,
                0x8f, 0x7c, 0x47, 0x12, 0x26, 0x8a, 0x6e, 0x04,
                0x4a, 0x13, 0x02, 0x31, 0xf2, 0xb3, 0x00, 0xea,
                0xbe, 0x6c, 0x66, 0xad, 0x0b, 0x07, 0xd0, 0x3e,
                0x00, 0x80, 0xd5, 0x6b, 0x11, 0x43, 0x25, 0xc9,
                0x9b, 0x4f, 0xe5, 0xf0, 0x26, 0xb3, 0x5d, 0x68,
                0x03, 0xf5, 0xb7, 0x15, 0x92, 0x13, 0x90, 0x26,
                0xf8, 0x92, 0xd3, 0xa1, 0xa6, 0xc4, 0xdb, 0x7a,
                0xc5, 0xbb, 0x75, 0x7a, 0xf8, 0x68, 0x40, 0x86,
                0x66, 0x89, 0xfb, 0xd5, 0xd7, 0xce, 0x9e, 0x98,
                0x85, 0x1f, 0x9a, 0xfe, 0xeb, 0xfc, 0x84, 0x2c,
                0x1e, 0x11, 0x3e, 0xcb, 0x82, 0x34, 0x5e, 0x1a,
                0x0f, 0x30, 0xd0, 0x1f, 0x8b, 0x44, 0x66, 0x91,
                0xcc, 0x75, 0x02, 0x73, 0x66, 0x7c, 0xb5, 0x79,
                0xba, 0xfc, 0x40, 0x29, 0x81, 0xd3, 0x1f, 0xfd,
                0x9e, 0x9b, 0x94, 0xa0, 0x7c, 0x5f, 0x51, 0x23,
                0x81, 0x10, 0x65, 0xfa, 0x9b, 0xb4, 0x75, 0xed,
                0x7f, 0xfa, 0x79, 0x2b, 0xe1, 0xd2, 0x20, 0x5a,
                0xd1, 0x55, 0x3a, 0xba, 0x16, 0xbe, 0x33, 0xcf,
                0xba, 0x21, 0x03, 0x5c, 0x59, 0xc6, 0x4f, 0xe2,
                0x23, 0x93, 0xc4, 0xbd, 0xee, 0x0f, 0xf2, 0xd2,
                0x9c, 0x4e, 0xca, 0xc4, 0x8e, 0x7a, 0x55, 0x9e,
                0x1e, 0x76, 0xd7, 0x26, 0x65, 0x4d, 0x9b, 0x6f,
                0xe4, 0x85, 0xad, 0xe2, 0x94, 0x69, 0x9e, 0xac,
                0x79, 0x65, 0xe8, 0x7a, 0xf7, 0x50, 0x84, 0x6f,
                0xe5, 0x2a, 0x3c, 0x5c, 0x72, 0x84, 0xe3, 0x50,
                0xbd, 0xac, 0x1a, 0x26, 0x58, 0xd4, 0x07, 0x74,
                0xab, 0x96, 0x57, 0x95, 0xd5, 0x20, 0x34, 0x93,
                0x41, 0x55, 0xe7, 0x47, 0x95, 0x95, 0x3a, 0x33,
                0xd8, 0xe7, 0x9f, 0xcb, 0xca, 0x00, 0x6b, 0x1b,
                0xe7, 0x7d, 0xe6, 0x45, 0xf5, 0x36, 0x20, 0xec,
                0xc7, 0x6d, 0x0c, 0x4c, 0x7b, 0xd3, 0xbe, 0xb8,
                0xfe, 0x82, 0x65, 0x35, 0x57, 0x34, 0x36, 0x22,
                0x3f, 0x0c, 0x39, 0x01, 0xb5, 0x7b
            }
        },
        {
            std::string("0123456789ABCDEF0123456789ABCDEF0123456789ABCDEF012345"
                        "6789ABCDEF0123456789ABCDEF0123456789ABCDEF0123456789AB"
                        "CDEF0123456789ABCDEF0123456789ABCDEF0123456789ABCDEF01"
                        "23456789ABCDEF0123456789ABCDEF0123456789ABCDEF01234567"
                        "89ABCDEF0123456789ABCDEF0123456789ABCDEF0"),
            {
                0x41, 0x45, 0x53, 0x01, 0x00, 0x9f, 0xc7, 0xbd,
                0xfc, 0xe7, 0x59, 0x02, 0xe6, 0xbb, 0x88, 0x35,
                0xcb, 0x46, 0x09, 0xd3, 0xe6, 0x1d, 0x2d, 0x7e,
                0x77, 0xa7, 0x78, 0x90, 0x89, 0x93, 0xcc, 0x3b,
                0x9e, 0xe6, 0x06, 0x22, 0x02, 0x8a, 0x1f, 0x4b,
                0xd5, 0x0a, 0xa2, 0xdb, 0xe2, 0xbc, 0xb2, 0x09,
                0x6e, 0x0a, 0xa4, 0xdb, 0x35, 0x6d, 0xbe, 0x95,
                0xfc, 0x43, 0xd4, 0x23, 0x37, 0x66, 0xe3, 0x13,
                0x15, 0x13, 0xe7, 0x97, 0x56, 0xfc, 0x4b, 0xcc,
                0xc1, 0x80, 0x61, 0x43, 0xa3, 0xca, 0x5e, 0x91,
                0x77, 0x93, 0x1d, 0xe0, 0xb2, 0x37, 0x63, 0x5a,
                0xa6, 0x41, 0x67, 0xf1, 0xb2, 0x27, 0x99, 0x4e,
                0x24, 0x42, 0x43, 0x00, 0x95, 0x74, 0x4b, 0x11,
                0x11, 0x63, 0xa0, 0x89, 0x9c, 0x6e, 0x54, 0xed,
                0x5c, 0x32, 0xfe, 0x4a, 0xf2, 0xe5, 0xc4, 0x61,
                0x0c, 0x11, 0xd8, 0xaf, 0xb6, 0xf1, 0xc9, 0xf8,
                0xff, 0xe3, 0x72, 0x41, 0x45, 0x3e, 0xad, 0xd8,
                0x1b, 0xf1, 0x73, 0xab, 0x1d, 0x7b, 0xe1, 0x6d,
                0x2f, 0xca, 0xd8, 0x5d, 0x41, 0x0a, 0xa3, 0x90,
                0x7e, 0xcc, 0x06, 0x05, 0x34, 0xe7, 0x6f, 0x27,
                0x57, 0x8e, 0x65, 0x0f, 0xa2, 0xac, 0x7d, 0x27,
                0x36, 0x3b, 0xfb, 0x5e, 0x47, 0xd1, 0x06, 0x05,
                0xc2, 0x05, 0xaa, 0x88, 0x56, 0xda, 0xef, 0x6a,
                0x63, 0x1e, 0xea, 0x0f, 0x58, 0xdf, 0x9a, 0x96,
                0xce, 0x4c, 0xed, 0x14, 0xb8, 0x78, 0xbe, 0x71,
                0x2d, 0xef, 0xb1, 0x4e, 0x37, 0x59, 0x42, 0xa8,
                0xc7, 0x62, 0x6f, 0xfc, 0x6b, 0x2f, 0x41, 0x79,
                0x7c, 0x15, 0x3e, 0x23, 0x55, 0x9d, 0xfb, 0x3c,
                0x78, 0x59, 0x2e, 0x5b, 0x43, 0xdc, 0x82, 0x1b,
                0x0f, 0x4c, 0x52, 0x3f, 0x45, 0x9b, 0xde, 0x1c,
                0xea, 0xad, 0xe1, 0xfd, 0x82, 0xd3, 0x55, 0x72,
                0x67, 0x5f, 0x75, 0xd9, 0xb7, 0xb0, 0xaf, 0x3b,
                0xda, 0x5f, 0x2b, 0x13, 0x08, 0x24, 0xce, 0x06,
                0x99, 0xa0, 0x4f, 0x07, 0x6d, 0x81, 0xbb, 0xcb,
                0x6f, 0xeb, 0x16, 0xe0, 0x5c, 0xc6, 0xef, 0x5a,
                0x62, 0x51, 0x18, 0x1c, 0xc2, 0xdc, 0x21, 0x20,
                0xfe, 0x2d, 0x06, 0xff, 0x49, 0xd9, 0x6a, 0x86,
                0x3c, 0x36, 0xaf, 0x69, 0x9b, 0x5e, 0xcb, 0xf0,
                0x2b, 0xb9, 0x5a, 0x6c, 0x0b, 0x0c, 0x6a, 0x5d,
                0x9a, 0x99, 0x24, 0x00, 0xd6, 0xb2, 0xc6, 0xdf,
                0x97, 0x5e, 0x1a, 0x46, 0x98, 0x8b, 0xbb, 0x3d,
                0x2b, 0xbf, 0xcc, 0x6e, 0x31, 0xda, 0x6f, 0x07,
                0xcb, 0x38, 0x26, 0xd5, 0xc1, 0x98, 0xff, 0xb7,
                0xe4, 0x24, 0xc2, 0x3f, 0x8b, 0xd5, 0xde, 0x7c,
                0x1f, 0x6b, 0x76, 0xef, 0x34, 0xfe, 0x69, 0xea,
                0x29, 0xa4, 0xa2, 0xd6, 0x22, 0x8a, 0x34, 0xd2,
                0x5f, 0x2e, 0x01, 0xa0, 0xb4, 0x01, 0x5f, 0x83,
                0x50, 0xde, 0xce, 0xbf, 0xe8, 0xa9, 0xb0, 0x64,
                0x28, 0x5b, 0x52, 0x6e, 0xab, 0x74, 0xb9, 0x71,
                0xe8, 0x82, 0xf1, 0x1d, 0x28, 0x38, 0x9c, 0x33,
                0xde, 0x99, 0x74, 0x58, 0x66, 0xaa
            }
        }
    };

    // Create a Logger object; chance null_ostream to std::cerr if debugging
    NullOStream null_ostream;
    LoggerPointer logger =
        std::make_shared<Logger>(null_ostream, LogLevel::Debug);

    // Iterate over the test vector
    std::size_t counter = 0;
    for (const auto &[plaintext, ciphertext] : tests)
    {
        // Create a Decryptor object
        Decryptor decryptor(logger, std::to_string(counter++));

        // Define a place to put the decrypted text
        std::ostringstream oss;

        // Create an input stream over the ciphertext
        StringStreamBuffer cipher_stream_buffer(ciphertext);
        std::istream iss(&cipher_stream_buffer);

        // Decrypt the ciphertext
        auto result = decryptor.Decrypt(password, iss, oss, 0);
        STF_ASSERT_EQ(DecryptResult::Success, result);

        // Ensure the output string length is the expected length
        STF_ASSERT_EQ(plaintext.size(), oss.str().size());
    }
}
