/*
 *  test_stream_0.cpp
 *
 *  Copyright (C) 2024
 *  Terrapane Corporation
 *  All Rights Reserved
 *
 *  Author:
 *      Paul E. Jones <paulej@packetizer.com>
 *
 *  Description:
 *      This module will test the AES Crypt Engine's ability to decrypt
 *      various known plaintext / ciphertext pairs following stream format 0.
 *
 *  Portability Issues:
 *      None.
 */

#include <vector>
#include <string>
#include <terra/aescrypt/engine/decryptor.h>
#include <terra/logger/logger.h>
#include <terra/logger/null_ostream.h>
#include <terra/stf/stf.h>
#include "../string_stream_buffer.h"

using namespace Terra::AESCrypt::Engine;
using namespace Terra::Logger;

// It is assumed that a char and uint8_t are the same size
static_assert(sizeof(char) == sizeof(std::uint8_t));

namespace
{

// Structure to hold plaintext and ciphertext pairs
struct PlaintextCiphertext
{
    const std::string plaintext;
    const std::vector<uint8_t> ciphertext;
};

} // namespace

STF_TEST(TestDecryption, Stream_Format_0)
{
    const std::u8string password = u8"Hello";
    const std::vector<PlaintextCiphertext> tests =
    {
        {
            std::string(""),
            {
                0x41, 0x45, 0x53, 0x00, 0x00, 0x33, 0x64, 0x05,
                0xda, 0xcc, 0x29, 0xe2, 0xb1, 0x10, 0xff, 0xe2,
                0xad, 0x46, 0x90, 0x77, 0xbe, 0xd2, 0xec, 0xdb,
                0x0a, 0x07, 0x61, 0x0a, 0xb0, 0x77, 0x9f, 0x39,
                0xd8, 0xa5, 0x45, 0x2f, 0x24, 0x42, 0x8d, 0xfb,
                0x0d, 0xb9, 0x0b, 0x87, 0x91, 0x57, 0x77, 0x88,
                0x41, 0xae, 0x97, 0xca, 0x75
            }
        },
        {
            std::string("0"),
            {
                0x41, 0x45, 0x53, 0x00, 0x01, 0xa9, 0xa8, 0x48,
                0x14, 0x6a, 0x69, 0x14, 0x62, 0x37, 0x0f, 0x96,
                0xd7, 0xa0, 0x60, 0x34, 0x6c, 0x0d, 0xbf, 0xe1,
                0xb6, 0x1a, 0x83, 0x37, 0xc3, 0x75, 0x66, 0xe3,
                0x83, 0x99, 0xd3, 0x92, 0xb9, 0x0f, 0xad, 0xf1,
                0x00, 0x58, 0x91, 0x19, 0x1a, 0x26, 0xd8, 0xb8,
                0xcf, 0x8c, 0xc4, 0xea, 0x79, 0x0b, 0x07, 0x79,
                0x56, 0x9d, 0xef, 0x6d, 0xa5, 0xdd, 0xab, 0x98,
                0xa0, 0xaa, 0xa3, 0x22, 0x45
            }
        },
        {
            std::string("012"),
            {
                0x41, 0x45, 0x53, 0x00, 0x03, 0x90, 0xad, 0x83,
                0xe2, 0x46, 0xce, 0xfb, 0x46, 0xcc, 0x42, 0x9c,
                0x37, 0x44, 0x29, 0x34, 0x2a, 0x3d, 0x59, 0x55,
                0x20, 0x00, 0xc2, 0x56, 0xe1, 0x1b, 0x87, 0x3c,
                0x2c, 0xf3, 0x20, 0x1c, 0x36, 0x65, 0x5c, 0x60,
                0xa0, 0xcb, 0x3a, 0x37, 0x56, 0xe3, 0x4c, 0x2d,
                0xe2, 0xaf, 0xb3, 0x18, 0xce, 0x43, 0x46, 0x15,
                0x94, 0x47, 0xac, 0x51, 0xcb, 0xd7, 0xe8, 0x1a,
                0x69, 0x7a, 0xef, 0x7e, 0xa9
            }
        },
        {
            std::string("0123456789ABCDE"),
            {
                0x41, 0x45, 0x53, 0x00, 0x0f, 0x73, 0x56, 0x09,
                0x40, 0x2a, 0x2a, 0x29, 0xc5, 0x77, 0x8a, 0xff,
                0xcc, 0xf6, 0x93, 0x8e, 0x71, 0x32, 0xf1, 0xcb,
                0x32, 0x17, 0x48, 0x6d, 0x46, 0xc6, 0x53, 0x3f,
                0x15, 0x92, 0x57, 0xd2, 0xfd, 0x5a, 0xad, 0x5c,
                0xed, 0x78, 0xa5, 0x97, 0x90, 0xfd, 0x48, 0xd1,
                0xcf, 0x59, 0xc0, 0x7f, 0x15, 0x30, 0xf7, 0x1d,
                0xb3, 0x6d, 0xcc, 0xa5, 0xc2, 0xa8, 0xb3, 0xa1,
                0x49, 0x65, 0xf1, 0x2f, 0xea
            }
        },
        {
            std::string("0123456789ABCDEF"),
            {
                0x41, 0x45, 0x53, 0x00, 0x00, 0xb5, 0xcc, 0xe7,
                0x25, 0x81, 0x81, 0x33, 0x9f, 0x66, 0xf9, 0x59,
                0xff, 0x61, 0xbc, 0x11, 0x10, 0x6d, 0xb6, 0xba,
                0x0e, 0x1d, 0x02, 0x20, 0x0b, 0xb4, 0x1f, 0xad,
                0xe2, 0xc8, 0xc7, 0x51, 0x05, 0x2c, 0xb4, 0x96,
                0xa2, 0xf4, 0xd1, 0x70, 0xd9, 0xa5, 0xca, 0xfa,
                0x78, 0xd1, 0x51, 0x1d, 0x1c, 0x44, 0x82, 0xc2,
                0x0d, 0xb0, 0x65, 0x32, 0x27, 0x9d, 0x8c, 0x8a,
                0x73, 0xaa, 0x7b, 0x6a, 0x4e
            }
        },
        {
            std::string("0123456789ABCDEF0"),
            {
                0x41, 0x45, 0x53, 0x00, 0x01, 0x2d, 0x9c, 0x44,
                0xdd, 0x77, 0xba, 0x68, 0x34, 0x74, 0x9d, 0x68,
                0xfa, 0x7e, 0x9b, 0xa2, 0x24, 0xfa, 0x56, 0x88,
                0xc9, 0x88, 0xe8, 0x3b, 0x83, 0x3f, 0xb8, 0xd4,
                0x94, 0x9f, 0x99, 0x9c, 0xc9, 0x25, 0x2e, 0x9e,
                0x0c, 0x5b, 0x19, 0xdb, 0x58, 0x9c, 0x69, 0xf9,
                0xe4, 0xd3, 0xe4, 0x18, 0x68, 0x36, 0x56, 0x00,
                0x75, 0x77, 0x38, 0x12, 0xc4, 0x64, 0x08, 0x6c,
                0x66, 0xdf, 0xf5, 0x8d, 0xff, 0x71, 0x28, 0xc3,
                0x99, 0xac, 0x70, 0x45, 0x3d, 0x51, 0x8b, 0xda,
                0x96, 0xd8, 0x25, 0xda, 0x49
            }
        },
        {
            std::string("0123456789ABCDEF0123456789ABCDE"),
            {
                0x41, 0x45, 0x53, 0x00, 0x0f, 0x21, 0x67, 0x71,
                0x89, 0x9d, 0x05, 0x26, 0xb4, 0xfd, 0xd7, 0x24,
                0x2a, 0xde, 0x09, 0x14, 0xf1, 0x85, 0xb8, 0x50,
                0x3b, 0x32, 0x67, 0x34, 0x9f, 0x20, 0x25, 0x7b,
                0x9f, 0x07, 0xf3, 0x36, 0x23, 0xb4, 0xbe, 0x3e,
                0xa5, 0x90, 0x4a, 0xfd, 0x4c, 0x37, 0x31, 0x2c,
                0x33, 0x43, 0x52, 0x6d, 0x4e, 0x7f, 0x92, 0x9c,
                0x8c, 0x4d, 0xaa, 0x34, 0x91, 0x09, 0x04, 0x39,
                0x86, 0xc6, 0x22, 0x86, 0x70, 0xc6, 0xc2, 0xe4,
                0x43, 0xde, 0x4c, 0xeb, 0x70, 0xc0, 0x8d, 0xf1,
                0x7b, 0x1b, 0xfd, 0x9a, 0x21
            }
        },
        {
            std::string("0123456789ABCDEF0123456789ABCDEF"),
            {
                0x41, 0x45, 0x53, 0x00, 0x00, 0x33, 0xd4, 0xe4,
                0x98, 0xad, 0x2a, 0x1a, 0x4b, 0xbe, 0x6d, 0xcd,
                0xe1, 0x78, 0x51, 0x5b, 0xcf, 0xf4, 0xa1, 0x78,
                0xee, 0x5a, 0x9c, 0xfa, 0xf4, 0x27, 0x4b, 0x0a,
                0x83, 0x8a, 0x0a, 0x5a, 0x49, 0x2d, 0x08, 0xb9,
                0x82, 0x4c, 0x8d, 0xc6, 0x1d, 0xaf, 0xa0, 0xea,
                0xcf, 0xe9, 0x1f, 0x04, 0xed, 0x01, 0x7f, 0x1b,
                0xaa, 0xbf, 0xa5, 0xf7, 0x7c, 0x27, 0xb0, 0x33,
                0x81, 0x34, 0x0b, 0x2d, 0x45, 0xac, 0x10, 0x72,
                0x4b, 0xf9, 0x8d, 0x5f, 0xa8, 0x1b, 0x30, 0x8f,
                0x68, 0x72, 0xe0, 0x87, 0x89
            }
        },
        {
            std::string("0123456789ABCDEF0123456789ABCDEF0"),
            {
                0x41, 0x45, 0x53, 0x00, 0x01, 0x9f, 0xed, 0x9b,
                0x75, 0x27, 0xe2, 0x4a, 0xf2, 0x33, 0xa7, 0x42,
                0x4d, 0x55, 0x12, 0x6e, 0x37, 0xa5, 0x89, 0x1c,
                0x38, 0x0c, 0x8a, 0xc0, 0xcf, 0x47, 0xc7, 0xe8,
                0xd9, 0x8c, 0x12, 0x1b, 0x98, 0xc3, 0xbb, 0xea,
                0x8f, 0x61, 0x3a, 0x7a, 0x4e, 0x3d, 0xc9, 0xcb,
                0xf2, 0xde, 0x8e, 0x2e, 0x3b, 0x92, 0xe2, 0xf0,
                0xa3, 0xdb, 0x59, 0xb9, 0x07, 0x2a, 0xbb, 0xa3,
                0xd5, 0x07, 0x89, 0xe6, 0xe5, 0x98, 0xc6, 0xed,
                0xb4, 0xeb, 0xd1, 0x4d, 0x4e, 0xbc, 0xdf, 0x85,
                0xc0, 0xbc, 0x6d, 0x83, 0x9b, 0x93, 0xa7, 0x94,
                0xfb, 0x38, 0x83, 0x6a, 0x0c, 0xec, 0x2f, 0xd7,
                0x01, 0x27, 0xda, 0xc4, 0xf0
            }
        },
        {
            std::string("0123456789ABCDEF0123456789ABCDEF0123456789ABCDE"),
            {
                0x41, 0x45, 0x53, 0x00, 0x0f, 0x7a, 0xa9, 0xf7,
                0x4f, 0xf4, 0x7d, 0x15, 0x1f, 0x6e, 0x50, 0xf7,
                0x01, 0xd3, 0xed, 0xf8, 0x50, 0x8b, 0x07, 0x87,
                0xe2, 0x61, 0x4f, 0xc1, 0x5c, 0x57, 0x38, 0x3e,
                0x16, 0xc5, 0x57, 0x8e, 0x5b, 0x18, 0x48, 0x58,
                0x46, 0x97, 0x77, 0xca, 0x1a, 0xd4, 0x44, 0x71,
                0x2f, 0xb9, 0xa8, 0xa0, 0x4c, 0xa1, 0x30, 0x70,
                0x5b, 0xd2, 0xf8, 0x9f, 0x22, 0xfd, 0xaa, 0x70,
                0x79, 0x19, 0x30, 0x1d, 0x09, 0x5a, 0xb3, 0x8b,
                0xed, 0xf8, 0x8c, 0xa5, 0x7c, 0x04, 0xee, 0xca,
                0x67, 0xe6, 0xe7, 0xa8, 0x01, 0x11, 0x58, 0x7f,
                0x42, 0xfe, 0x49, 0x89, 0x49, 0xf4, 0x03, 0xba,
                0xee, 0x14, 0x87, 0x6a, 0x1a
            }
        },
        {
            std::string("0123456789ABCDEF0123456789ABCDEF0123456789ABCDEF"),
            {
                0x41, 0x45, 0x53, 0x00, 0x00, 0x54, 0xb1, 0x47,
                0x3f, 0xcc, 0x88, 0x72, 0x79, 0xfb, 0x83, 0x16,
                0xdf, 0x96, 0x22, 0x93, 0xed, 0x63, 0xd6, 0x94,
                0xc2, 0x52, 0x57, 0xe4, 0x0d, 0xf1, 0x71, 0x8b,
                0xb5, 0xc1, 0xb5, 0x6b, 0xfd, 0x09, 0x63, 0xe0,
                0xa2, 0xb8, 0x04, 0xa5, 0xa9, 0x9f, 0x2b, 0xae,
                0xb4, 0x85, 0x2a, 0xd6, 0x20, 0x71, 0xc7, 0x14,
                0x7f, 0xd9, 0x69, 0x91, 0x16, 0x39, 0xd1, 0x74,
                0x58, 0x1f, 0x36, 0x6c, 0xff, 0x08, 0xed, 0x85,
                0x50, 0xb3, 0x9f, 0x6c, 0xe0, 0xe1, 0x6b, 0xed,
                0x78, 0x28, 0x6e, 0x85, 0xd8, 0x30, 0x67, 0xce,
                0x9a, 0xba, 0x3f, 0xd1, 0x3e, 0x35, 0x6f, 0x18,
                0x62, 0x44, 0xf8, 0xde, 0x5b
            }
        },
        {
            std::string("0123456789ABCDEF0123456789ABCDEF0123456789ABCDEF0"),
            {
                0x41, 0x45, 0x53, 0x00, 0x01, 0x3f, 0x84, 0x05,
                0xc6, 0x22, 0x21, 0x69, 0xdc, 0xe7, 0x26, 0xb2,
                0xc0, 0x14, 0xb1, 0x92, 0xf7, 0xae, 0x24, 0x24,
                0x38, 0xda, 0xed, 0x21, 0x44, 0x70, 0x37, 0xe3,
                0x68, 0xaa, 0xba, 0xa6, 0xc0, 0xb3, 0x98, 0xf0,
                0x98, 0x10, 0x9e, 0x89, 0x7b, 0xf7, 0xca, 0x63,
                0xc7, 0x81, 0x2e, 0xb3, 0x87, 0x0f, 0x9e, 0xe2,
                0x79, 0xb8, 0x03, 0x9a, 0xd9, 0xf8, 0x27, 0x95,
                0x30, 0x54, 0x3a, 0x74, 0xee, 0xf3, 0x1f, 0x42,
                0x40, 0x39, 0x05, 0x6f, 0x45, 0x25, 0xaa, 0x8f,
                0x98, 0x64, 0x0a, 0x58, 0x80, 0xfd, 0x42, 0x92,
                0x82, 0x40, 0x2a, 0x45, 0x7d, 0xc7, 0xc8, 0x61,
                0xd1, 0x12, 0xde, 0x7f, 0x8b, 0x22, 0xb3, 0xc2,
                0x1a, 0xe7, 0x75, 0x93, 0xf4, 0xd4, 0x7c, 0x52,
                0xa5, 0x28, 0xbe, 0x1b, 0xd9
            }
        },
        {
            std::string("0123456789ABCDEF0123456789ABCDEF0123456789ABCDEF012345"
                        "6789ABCDE"),
            {
                0x41, 0x45, 0x53, 0x00, 0x0f, 0x13, 0x72, 0x52,
                0x32, 0x41, 0x99, 0xb8, 0x81, 0xe2, 0x1a, 0x7d,
                0xdb, 0x8b, 0x2c, 0x25, 0x10, 0x07, 0xdf, 0xa6,
                0x71, 0xc9, 0xdb, 0xa8, 0x71, 0xe2, 0x4c, 0xf1,
                0x7b, 0xd6, 0x70, 0x00, 0x41, 0x2f, 0x25, 0x49,
                0x4a, 0x40, 0x99, 0xea, 0xd7, 0x9e, 0x25, 0xb8,
                0x74, 0xb3, 0x08, 0x00, 0x0a, 0x8f, 0xed, 0xca,
                0x76, 0x0d, 0xa6, 0x70, 0xac, 0x75, 0x84, 0xed,
                0x9b, 0x57, 0x76, 0x3e, 0x07, 0x52, 0x48, 0xa4,
                0x94, 0xad, 0xde, 0x10, 0x2c, 0x36, 0x54, 0x03,
                0x75, 0xec, 0x59, 0x37, 0x68, 0xac, 0xa8, 0xdb,
                0x0f, 0x20, 0x93, 0x56, 0x62, 0x56, 0x8f, 0x34,
                0x73, 0xa4, 0xad, 0xd4, 0x65, 0xb6, 0x5a, 0x4a,
                0xe7, 0xec, 0xd7, 0x55, 0x5c, 0xd2, 0x86, 0x02,
                0xd6, 0x52, 0xae, 0x57, 0xf7
            }
        },
        {
            std::string("0123456789ABCDEF0123456789ABCDEF0123456789ABCDEF012345"
                        "6789ABCDEF"),
            {
                0x41, 0x45, 0x53, 0x00, 0x00, 0xbe, 0xbd, 0xfa,
                0xcd, 0x57, 0x47, 0x54, 0x90, 0x6c, 0x1b, 0x4a,
                0xc2, 0x88, 0x63, 0x41, 0xb2, 0xc5, 0x9e, 0xd9,
                0x36, 0x00, 0x4f, 0xbb, 0x52, 0xe8, 0xc6, 0x6e,
                0xba, 0x28, 0x11, 0xca, 0x5d, 0xf1, 0xc0, 0x05,
                0x32, 0xc3, 0xa0, 0xa8, 0x8c, 0x75, 0x8e, 0xef,
                0x2c, 0x5f, 0xf2, 0x9e, 0xcd, 0x81, 0x6c, 0xc0,
                0xf4, 0xbb, 0x9b, 0xa4, 0xf0, 0x3f, 0x92, 0x37,
                0x5d, 0x66, 0x4f, 0xa5, 0xf1, 0xb0, 0x19, 0xdd,
                0xb3, 0x3f, 0xdd, 0x0a, 0x81, 0x50, 0x8a, 0x1f,
                0x8d, 0xae, 0xe1, 0x62, 0x02, 0x0c, 0x75, 0x84,
                0x8b, 0xa0, 0xe0, 0x36, 0x4a, 0x68, 0xeb, 0x4a,
                0x52, 0x39, 0xa0, 0x22, 0x23, 0x90, 0xd6, 0x69,
                0x10, 0xc7, 0x24, 0x01, 0x5c, 0xba, 0xce, 0xc8,
                0xb3, 0x96, 0xce, 0x0c, 0x19
            }
        },
        {
            std::string("0123456789ABCDEF0123456789ABCDEF0123456789ABCDEF012345"
                        "6789ABCDEF0"),
            {
                0x41, 0x45, 0x53, 0x00, 0x01, 0xa3, 0x7b, 0xd9,
                0x97, 0xad, 0x72, 0x7e, 0x08, 0xdc, 0x6f, 0xca,
                0x6b, 0x99, 0xcf, 0x9c, 0xdd, 0xec, 0xbf, 0x57,
                0xc7, 0x5c, 0x3a, 0x6b, 0x83, 0xab, 0x4c, 0x85,
                0x9b, 0x3b, 0x70, 0x49, 0x5c, 0x04, 0xc9, 0xa6,
                0x3f, 0xc9, 0x28, 0xcd, 0x5f, 0x42, 0x86, 0x19,
                0x31, 0x90, 0xc3, 0x64, 0x62, 0x99, 0x22, 0x21,
                0xca, 0x2e, 0xb2, 0x24, 0x2c, 0x04, 0xaa, 0x00,
                0x77, 0xf6, 0x15, 0xb2, 0x0e, 0x01, 0xfc, 0x82,
                0x65, 0x77, 0xea, 0x33, 0x6f, 0x0f, 0x05, 0x2d,
                0xcc, 0x78, 0x77, 0x4a, 0x45, 0x30, 0xf4, 0xd2,
                0xd0, 0x31, 0xcf, 0x66, 0x01, 0x4b, 0x28, 0x6a,
                0xe8, 0x44, 0xf3, 0x3d, 0xbc, 0xaa, 0xe9, 0xd8,
                0x89, 0x11, 0x0d, 0x3a, 0xa2, 0x97, 0x09, 0xed,
                0x9e, 0x9a, 0xdc, 0x86, 0x48, 0xd1, 0x73, 0x90,
                0x64, 0x9d, 0x97, 0x1d, 0xf5, 0xb2, 0xe5, 0x62,
                0x53, 0x3b, 0x60, 0x4a, 0x12
            }
        },
        {
            std::string("0123456789ABCDEF0123456789ABCDEF0123456789ABCDEF012345"
                        "6789ABCDEF0123456789ABCDE"),
            {
                0x41, 0x45, 0x53, 0x00, 0x0f, 0x54, 0x29, 0x7f,
                0xec, 0x57, 0x1a, 0x9d, 0x78, 0x52, 0xfd, 0x71,
                0x2a, 0x9c, 0xa2, 0x89, 0xa0, 0x43, 0x66, 0x13,
                0xb2, 0xad, 0xa3, 0xa6, 0x74, 0xc8, 0x0c, 0x94,
                0xe1, 0x71, 0x60, 0x2a, 0x11, 0xf7, 0x0a, 0x99,
                0x66, 0x2a, 0x1e, 0x0d, 0x59, 0x57, 0xa0, 0xe3,
                0x8f, 0xe8, 0x4b, 0x59, 0x3e, 0x05, 0x7e, 0xc2,
                0xe0, 0x2e, 0xf5, 0x6e, 0x7f, 0x1d, 0xa2, 0xc1,
                0x89, 0x8b, 0xd4, 0x48, 0x11, 0x73, 0x04, 0x39,
                0xb6, 0xec, 0xaf, 0xba, 0xee, 0xcb, 0xdc, 0x4b,
                0xa9, 0xb2, 0x6e, 0x26, 0x37, 0xaa, 0x37, 0x93,
                0x6d, 0xb1, 0xda, 0x96, 0xba, 0xb4, 0x43, 0xec,
                0xf3, 0x31, 0xbd, 0x16, 0x2e, 0xd0, 0xcc, 0x2c,
                0xf0, 0xf8, 0xee, 0x02, 0xa5, 0xf2, 0x16, 0xd6,
                0x47, 0xc9, 0xee, 0x1a, 0x03, 0xe2, 0x9d, 0xff,
                0xa6, 0x4c, 0xbb, 0x61, 0x0f, 0x6d, 0xb2, 0x44,
                0x59, 0x15, 0xda, 0x22, 0xf2
            }
        },
        {
            std::string("0123456789ABCDEF0123456789ABCDEF0123456789ABCDEF012345"
                        "6789ABCDEF0123456789ABCDEF"),
            {
                0x41, 0x45, 0x53, 0x00, 0x00, 0x0a, 0xce, 0xc2,
                0x76, 0x6c, 0xdc, 0x1f, 0xe2, 0xb6, 0xe6, 0x76,
                0xd7, 0xb3, 0x47, 0xa7, 0x4c, 0x46, 0x96, 0x13,
                0x84, 0xb2, 0xee, 0xb1, 0xfb, 0x24, 0xbb, 0x35,
                0x91, 0x50, 0xa0, 0x29, 0x63, 0x47, 0x0d, 0x74,
                0xce, 0x69, 0xce, 0xf1, 0x9f, 0x24, 0x28, 0x22,
                0x1c, 0xf6, 0x96, 0x19, 0xaa, 0x92, 0x45, 0x26,
                0x67, 0x53, 0xf0, 0x65, 0x62, 0x34, 0x3e, 0x9f,
                0xaa, 0x74, 0x37, 0xc7, 0xcb, 0xe4, 0xc1, 0x9a,
                0xb8, 0x10, 0x8b, 0x76, 0x37, 0xc3, 0x13, 0xb0,
                0x2f, 0x05, 0xa3, 0xb3, 0xff, 0x7b, 0xcb, 0x6e,
                0x59, 0x1f, 0xd6, 0xba, 0xc3, 0x0b, 0x5b, 0x0d,
                0x95, 0x82, 0x09, 0x5c, 0x9f, 0xb5, 0x43, 0xd4,
                0x93, 0xb6, 0xab, 0x3f, 0x53, 0xd8, 0x01, 0xfc,
                0xca, 0xd8, 0x38, 0x56, 0xdb, 0x97, 0x37, 0x1e,
                0x02, 0x97, 0x68, 0x38, 0xe5, 0x88, 0x21, 0x68,
                0x65, 0xea, 0x95, 0xcf, 0x13
            }
        },
        {
            std::string("0123456789ABCDEF0123456789ABCDEF0123456789ABCDEF012345"
                        "6789ABCDEF0123456789ABCDEF0"),
            {
                0x41, 0x45, 0x53, 0x00, 0x01, 0x7a, 0x24, 0x63,
                0x8c, 0x9a, 0x64, 0x60, 0x75, 0x25, 0x38, 0x0e,
                0xd8, 0xf9, 0x38, 0xfc, 0x62, 0xfd, 0x97, 0x02,
                0x88, 0xe5, 0xaf, 0x90, 0x74, 0x04, 0x02, 0x7b,
                0x62, 0xc2, 0x3e, 0x23, 0x61, 0x13, 0xe1, 0x04,
                0x0f, 0xcd, 0xd9, 0xed, 0x54, 0xa5, 0x9a, 0xc7,
                0x1d, 0xeb, 0x65, 0x7e, 0x2e, 0xd2, 0x1a, 0xf1,
                0xc2, 0xf3, 0x57, 0x3b, 0x8d, 0xf1, 0x25, 0xe1,
                0xbc, 0x06, 0x48, 0x24, 0x9b, 0x11, 0x18, 0xa0,
                0x8b, 0xcf, 0xe9, 0xed, 0x1d, 0x36, 0xc5, 0x73,
                0xba, 0x29, 0x4a, 0xe6, 0x55, 0x7c, 0x5d, 0x29,
                0xec, 0x86, 0x75, 0x65, 0xad, 0xf6, 0x1b, 0x49,
                0x63, 0x04, 0x3a, 0x7f, 0x01, 0xb8, 0x31, 0x8a,
                0x8e, 0xf7, 0xda, 0x4e, 0x61, 0x0d, 0x30, 0x48,
                0xaf, 0xd1, 0x31, 0xf1, 0x84, 0x49, 0xa8, 0xf9,
                0x8e, 0x2a, 0x31, 0xf9, 0x3c, 0x4f, 0xc6, 0x1c,
                0x13, 0x03, 0xbe, 0x3a, 0x07, 0x8f, 0x30, 0x4b,
                0x76, 0x04, 0xa1, 0xc3, 0x3f, 0xd7, 0xf2, 0xd6,
                0x59, 0xab, 0x2c, 0x0d, 0x32
            }
        },
        {
            std::string("0123456789ABCDEF0123456789ABCDEF0123456789ABCDEF012345"
                        "6789ABCDEF0123456789ABCDEF0123456789ABCDEF0123456789AB"
                        "CDEF0123456789ABCDEF0123456789ABCDEF0123456789ABCDEF01"
                        "23456789ABCDEF0123456789ABCDEF0123456789ABCDEF01234567"
                        "89ABCDEF0123456789ABCDEF0123456789ABCDE"),
            {
                0x41, 0x45, 0x53, 0x00, 0x0f, 0x37, 0xd8, 0xc5,
                0x90, 0x74, 0x2c, 0x2b, 0x16, 0xa1, 0x71, 0x68,
                0x84, 0x66, 0x04, 0x13, 0xe8, 0xc9, 0xd9, 0x36,
                0x2e, 0x16, 0xb1, 0x03, 0xd3, 0x47, 0xc6, 0xd0,
                0x12, 0xd0, 0x66, 0xe2, 0x39, 0x8a, 0xa9, 0x27,
                0xef, 0x08, 0x85, 0x5e, 0x8c, 0x82, 0x15, 0x2d,
                0x52, 0x14, 0x7a, 0x7e, 0x10, 0x92, 0xa3, 0xb3,
                0x69, 0xd4, 0x43, 0xad, 0x43, 0xbb, 0x14, 0x65,
                0x23, 0x3c, 0xc9, 0x34, 0xd3, 0xaf, 0xc4, 0x08,
                0x8e, 0x64, 0xcf, 0xe5, 0x9c, 0x20, 0x50, 0x43,
                0xb6, 0x39, 0xcd, 0xf5, 0xe2, 0xaf, 0x14, 0x49,
                0x74, 0x64, 0x39, 0x00, 0xc2, 0x21, 0xf2, 0x99,
                0x60, 0x7b, 0xbf, 0x21, 0x54, 0x61, 0xc8, 0x1a,
                0x89, 0x7a, 0x52, 0xad, 0xa5, 0xed, 0x83, 0x4e,
                0x60, 0xf3, 0xe3, 0x6c, 0x51, 0xc1, 0x31, 0xce,
                0x5c, 0xa3, 0x99, 0x45, 0xdc, 0xbd, 0x6b, 0xb2,
                0xd1, 0xfa, 0x5a, 0x8d, 0x13, 0x9b, 0xe3, 0xd6,
                0x15, 0x62, 0x01, 0x10, 0xfa, 0xcd, 0x90, 0x10,
                0x96, 0x72, 0xd1, 0xb7, 0x5e, 0xd4, 0x28, 0x68,
                0x8d, 0xc1, 0x86, 0x83, 0x5b, 0x49, 0xfd, 0x43,
                0x3d, 0xe0, 0xa4, 0x5b, 0xd9, 0x17, 0x23, 0x84,
                0xb7, 0x6e, 0x9d, 0x5a, 0x55, 0x8d, 0x37, 0x3d,
                0x9c, 0xa2, 0x7e, 0xc4, 0xbd, 0x98, 0x15, 0x25,
                0x96, 0x46, 0xa1, 0x57, 0x8d, 0xcc, 0xb9, 0x51,
                0x6b, 0x47, 0x02, 0x09, 0x66, 0x3d, 0xc0, 0x6c,
                0xb1, 0xc2, 0x37, 0x06, 0x90, 0xc8, 0x56, 0x88,
                0x5d, 0xb2, 0x03, 0x94, 0x6f, 0x79, 0x9d, 0x67,
                0x65, 0x1d, 0x66, 0x28, 0x00, 0x2c, 0xc6, 0xef,
                0x52, 0x54, 0x58, 0x36, 0xf2, 0x63, 0x2c, 0x3c,
                0x89, 0x68, 0xff, 0x7c, 0x13, 0xf0, 0x18, 0x23,
                0xb3, 0xdf, 0xf6, 0xac, 0x25, 0x79, 0xad, 0xe7,
                0xee, 0x1a, 0x36, 0x84, 0x89, 0xda, 0x41, 0xa7,
                0xa7, 0xd6, 0xa6, 0xd2, 0x51, 0x8c, 0x46, 0x73,
                0x1f, 0x14, 0xe9, 0xfc, 0x69, 0xec, 0x05, 0x34,
                0x5c, 0x91, 0x09, 0xbf, 0xe2, 0x7d, 0xf8, 0x7f,
                0x22, 0x71, 0x44, 0x20, 0xc5, 0x7e, 0x1b, 0x37,
                0xaa, 0xa6, 0xc7, 0x1a, 0x27, 0xe1, 0x21, 0x14,
                0x0b, 0x14, 0x16, 0x19, 0x51, 0x0b, 0x37, 0xf7,
                0xc4, 0xd4, 0x7f, 0x62, 0x01
            }
        },
        {
            std::string("0123456789ABCDEF0123456789ABCDEF0123456789ABCDEF012345"
                        "6789ABCDEF0123456789ABCDEF0123456789ABCDEF0123456789AB"
                        "CDEF0123456789ABCDEF0123456789ABCDEF0123456789ABCDEF01"
                        "23456789ABCDEF0123456789ABCDEF0123456789ABCDEF01234567"
                        "89ABCDEF0123456789ABCDEF0123456789ABCDEF"),
            {
                0x41, 0x45, 0x53, 0x00, 0x00, 0x94, 0xeb, 0x8f,
                0x8b, 0x65, 0x1f, 0x22, 0x40, 0x09, 0xfb, 0xe1,
                0x68, 0x56, 0xf4, 0x46, 0xf0, 0x2b, 0xf2, 0x5f,
                0x62, 0x19, 0x2e, 0x9a, 0xb7, 0x9e, 0x57, 0xf2,
                0x8e, 0xfc, 0xc8, 0x6b, 0x84, 0x4c, 0x4c, 0xa3,
                0x6c, 0x74, 0xca, 0x10, 0x54, 0xa9, 0x1b, 0x5c,
                0xfa, 0xd3, 0x5e, 0x4b, 0x65, 0x74, 0x8d, 0xe9,
                0x09, 0xf8, 0x3a, 0xa3, 0xea, 0xbe, 0x5a, 0x9c,
                0x89, 0x41, 0x8d, 0x43, 0x7a, 0xbc, 0x32, 0xef,
                0x69, 0x36, 0x67, 0xd0, 0xcc, 0x53, 0xb7, 0x98,
                0x44, 0xdc, 0xea, 0x13, 0x34, 0x85, 0x9c, 0x6c,
                0xf8, 0x40, 0x13, 0x53, 0xa5, 0xce, 0x05, 0xb1,
                0xe9, 0x7d, 0x14, 0xe8, 0xba, 0x4b, 0xac, 0xd7,
                0x4d, 0xdb, 0x7a, 0x5d, 0xc3, 0x74, 0xa4, 0xc9,
                0x46, 0x89, 0x3e, 0x1e, 0x00, 0xec, 0x44, 0xd9,
                0x5e, 0xae, 0x5e, 0xbc, 0xb0, 0x81, 0x5e, 0x55,
                0xf9, 0x8a, 0x71, 0xa4, 0x05, 0xeb, 0xe5, 0x44,
                0xc1, 0x59, 0x6e, 0x0b, 0x67, 0x40, 0xfb, 0xb2,
                0x2c, 0x30, 0x91, 0x79, 0x06, 0x7d, 0x65, 0x3d,
                0xe2, 0x03, 0x7c, 0xef, 0xde, 0x5f, 0x4f, 0x3d,
                0xe2, 0xec, 0xa7, 0x4a, 0x1b, 0x03, 0x7e, 0x72,
                0xce, 0x3c, 0x02, 0x9b, 0x63, 0x21, 0x25, 0xde,
                0x6b, 0x45, 0xd2, 0xf8, 0x6e, 0x6d, 0xb1, 0x96,
                0x81, 0x08, 0x97, 0xa8, 0x0e, 0x05, 0x7e, 0x3a,
                0xcc, 0xa2, 0xc5, 0x9a, 0x68, 0x1f, 0xc0, 0xa9,
                0x15, 0xe9, 0x9c, 0x5a, 0x65, 0xbe, 0x13, 0x38,
                0x13, 0xf8, 0x3d, 0x66, 0x12, 0x03, 0xbb, 0x94,
                0x1c, 0x49, 0x76, 0xee, 0x76, 0x67, 0xfe, 0x72,
                0x99, 0x24, 0x7d, 0x5f, 0xe4, 0x76, 0xee, 0x02,
                0x72, 0xd5, 0x7f, 0x73, 0x79, 0xb0, 0x66, 0x5f,
                0xf1, 0x06, 0xf5, 0x45, 0x4c, 0xc4, 0xe5, 0xa8,
                0x78, 0xfe, 0xf5, 0x5a, 0xf5, 0xab, 0x50, 0xd3,
                0x8c, 0x26, 0xb2, 0xed, 0xc0, 0x4c, 0xc9, 0xd5,
                0xc6, 0x4b, 0xfe, 0x81, 0x59, 0x85, 0xc3, 0x39,
                0x91, 0xce, 0xdb, 0x9a, 0x5a, 0x5d, 0xd9, 0x02,
                0x29, 0x19, 0x12, 0xc9, 0x59, 0x57, 0xed, 0x27,
                0x30, 0xa8, 0xfb, 0x50, 0x37, 0x37, 0x87, 0xbd,
                0x77, 0x89, 0xc0, 0xe7, 0xea, 0xd6, 0x64, 0x10,
                0x5d, 0x81, 0x25, 0xb6, 0x5b
            }
        },
        {
            std::string("0123456789ABCDEF0123456789ABCDEF0123456789ABCDEF012345"
                        "6789ABCDEF0123456789ABCDEF0123456789ABCDEF0123456789AB"
                        "CDEF0123456789ABCDEF0123456789ABCDEF0123456789ABCDEF01"
                        "23456789ABCDEF0123456789ABCDEF0123456789ABCDEF01234567"
                        "89ABCDEF0123456789ABCDEF0123456789ABCDEF0"),
            {
                0x41, 0x45, 0x53, 0x00, 0x01, 0xd1, 0xa2, 0x12,
                0x40, 0x09, 0x3e, 0xe3, 0x60, 0x1b, 0x7c, 0xd3,
                0x94, 0xc0, 0x8c, 0xdc, 0x67, 0xbf, 0xae, 0xeb,
                0xb0, 0x4d, 0xae, 0xf2, 0x63, 0xa8, 0xfe, 0x17,
                0xea, 0xef, 0xe5, 0x6d, 0x03, 0x46, 0xa8, 0x58,
                0x00, 0x92, 0x2f, 0x07, 0x63, 0x3f, 0xdd, 0x36,
                0xda, 0xb7, 0xb1, 0x13, 0x7a, 0x2c, 0x8c, 0x24,
                0xc6, 0x48, 0xed, 0x58, 0x9e, 0xcf, 0x5d, 0x80,
                0x7a, 0xa0, 0x1a, 0x36, 0x13, 0x2b, 0x6d, 0x80,
                0x03, 0x7c, 0x0e, 0x30, 0x2e, 0x25, 0x1a, 0x22,
                0x7a, 0xd7, 0xae, 0xe9, 0x6c, 0x39, 0x61, 0x97,
                0x10, 0xd2, 0x77, 0x3e, 0x4f, 0x13, 0x46, 0x3c,
                0x2a, 0xc2, 0xbc, 0x80, 0x11, 0xdb, 0xc0, 0x5e,
                0x4a, 0x59, 0x0c, 0xa6, 0xbe, 0xe2, 0xb5, 0xcf,
                0xec, 0x09, 0x92, 0xf1, 0x31, 0xfe, 0x8d, 0x77,
                0xf5, 0x8d, 0x2d, 0x9b, 0xec, 0x78, 0x17, 0x6b,
                0x90, 0x2c, 0x0e, 0x63, 0x42, 0xf5, 0x5f, 0xd0,
                0x46, 0x7f, 0x4b, 0x2f, 0x43, 0x13, 0x2d, 0xff,
                0xb9, 0x53, 0x9c, 0x64, 0xe9, 0x6e, 0xbc, 0xf7,
                0x1d, 0x99, 0x17, 0xf8, 0x73, 0x2c, 0x3e, 0x08,
                0xa8, 0x6a, 0x8c, 0x9c, 0x68, 0xeb, 0x16, 0x3a,
                0xa8, 0x17, 0x1f, 0x36, 0xd0, 0x4c, 0x22, 0x32,
                0xc8, 0xd0, 0x01, 0x82, 0xce, 0xef, 0xa8, 0x14,
                0x4b, 0x7d, 0x49, 0x1e, 0x50, 0xae, 0x79, 0xc0,
                0x97, 0x05, 0xaf, 0x51, 0x66, 0x6f, 0xa0, 0xa5,
                0x71, 0xc9, 0x87, 0x1b, 0xb9, 0x2c, 0x74, 0xe3,
                0xdc, 0x1d, 0x5a, 0x64, 0xde, 0xed, 0x62, 0xbc,
                0xe6, 0xe3, 0xee, 0xe3, 0xbe, 0xd2, 0x7f, 0x20,
                0x95, 0xf2, 0x8f, 0xb6, 0xfe, 0x2f, 0x8f, 0x84,
                0x47, 0xdd, 0x92, 0x36, 0x21, 0xd1, 0x59, 0x3f,
                0xbc, 0xcb, 0xb0, 0x6d, 0x6a, 0x48, 0x6f, 0xb7,
                0x5a, 0x5b, 0x06, 0x19, 0xcb, 0x8e, 0x7b, 0x08,
                0x21, 0x04, 0x00, 0xb8, 0x64, 0x5c, 0xfc, 0xbd,
                0xe8, 0x4f, 0xa5, 0x09, 0x79, 0xc3, 0xf5, 0x8c,
                0x9c, 0x9f, 0xb6, 0x71, 0xc3, 0xa2, 0x84, 0x24,
                0x3f, 0x8a, 0x56, 0x77, 0x39, 0xc1, 0x16, 0xde,
                0x8f, 0x53, 0xa4, 0xf6, 0x4a, 0x34, 0x96, 0xf7,
                0xe6, 0x95, 0x4a, 0xc6, 0xb1, 0x09, 0x79, 0x07,
                0xd7, 0x3d, 0x43, 0xb3, 0xd4, 0x8d, 0xb9, 0x5e,
                0x4b, 0xd7, 0x8c, 0x2a, 0x9e, 0x7f, 0x00, 0x70,
                0xb1, 0xc4, 0x50, 0x6d, 0x57
            }
        }
    };

    // Create a Logger object; chance null_ostream to std::cerr if debugging
    NullOStream null_ostream;
    LoggerPointer logger =
        std::make_shared<Logger>(null_ostream, LogLevel::Debug);

    // Iterate over the test vector
    std::size_t counter = 0;
    for (const auto &[plaintext, ciphertext] : tests)
    {
        // Create a Decryptor object
        Decryptor decryptor(logger, std::to_string(counter++));

        // Define a place to put the decrypted text
        std::ostringstream oss;

        // Create an input stream over the ciphertext
        StringStreamBuffer cipher_stream_buffer(ciphertext);
        std::istream iss(&cipher_stream_buffer);

        // Decrypt the ciphertext
        auto result = decryptor.Decrypt(password, iss, oss, 0);
        STF_ASSERT_EQ(DecryptResult::Success, result);

        // Ensure the output string length is the expected length
        STF_ASSERT_EQ(plaintext.size(), oss.str().size());
    }
}
