"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getCompositeParentOfType = getCompositeParentOfType;
exports.getHostChildren = getHostChildren;
exports.getHostParent = getHostParent;
exports.getHostSelf = getHostSelf;
exports.getHostSelves = getHostSelves;
exports.getHostSiblings = getHostSiblings;
exports.isHostElement = isHostElement;
exports.isHostElementForType = isHostElementForType;
/**
 * Checks if the given element is a host element.
 * @param element The element to check.
 */
function isHostElement(element) {
  return typeof element?.type === 'string';
}

/**
 * Returns first host ancestor for given element.
 * @param element The element start traversing from.
 */
function getHostParent(element) {
  if (element == null) {
    return null;
  }
  let current = element.parent;
  while (current) {
    if (isHostElement(current)) {
      return current;
    }
    current = current.parent;
  }
  return null;
}

/**
 * Returns host children for given element.
 * @param element The element start traversing from.
 */
function getHostChildren(element) {
  if (element == null) {
    return [];
  }
  const hostChildren = [];
  element.children.forEach(child => {
    if (typeof child !== 'object') {
      return;
    }
    if (isHostElement(child)) {
      hostChildren.push(child);
    } else {
      hostChildren.push(...getHostChildren(child));
    }
  });
  return hostChildren;
}

/**
 * Return a single host element that represent the passed host or composite element.
 *
 * @param element The element start traversing from.
 * @throws Error if the passed element is a composite element and has no host children or has more than one host child.
 * @returns If the passed element is a host element, it will return itself, if the passed element is a composite
 * element, it will return a single host descendant.
 */
function getHostSelf(element) {
  const hostSelves = getHostSelves(element);
  if (hostSelves.length === 0) {
    throw new Error(`Expected exactly one host element, but found none.`);
  }
  if (hostSelves.length > 1) {
    throw new Error(`Expected exactly one host element, but found ${hostSelves.length}.`);
  }
  return hostSelves[0];
}

/**
 * Return the array of host elements that represent the passed element.
 *
 * @param element The element start traversing from.
 * @returns If the passed element is a host element, it will return an array containing only that element,
 * if the passed element is a composite element, it will return an array containing its host children (zero, one or many).
 */
function getHostSelves(element) {
  return typeof element?.type === 'string' ? [element] : getHostChildren(element);
}

/**
 * Returns host siblings for given element.
 * @param element The element start traversing from.
 */
function getHostSiblings(element) {
  const hostParent = getHostParent(element);
  const hostSelves = getHostSelves(element);
  return getHostChildren(hostParent).filter(sibling => !hostSelves.includes(sibling));
}
function getCompositeParentOfType(element, type) {
  let current = element.parent;
  while (!isHostElement(current)) {
    // We're at the root of the tree
    if (!current) {
      return null;
    }
    if (current.type === type) {
      return current;
    }
    current = current.parent;
  }
  return null;
}

/**
 * Note: this function should be generally used for core React Native types like `View`, `Text`, `TextInput`, etc.
 */
function isHostElementForType(element, type) {
  // Not a host element
  if (!isHostElement(element)) return false;
  return getCompositeParentOfType(element, type) !== null;
}
//# sourceMappingURL=component-tree.js.map