/**
 * Copyright (c) Meta Platforms, Inc. and affiliates.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 *
 *
 * @format
 * @oncall react_native
 */

"use strict";

const { isAssetTypeAnImage } = require("./Bundler/util");
const AssetPaths = require("./node-haste/lib/AssetPaths");
const crypto = require("crypto");
const denodeify = require("denodeify");
const fs = require("fs");
const imageSize = require("image-size");
const path = require("path");
const readDir = denodeify(fs.readdir);
const readFile = denodeify(fs.readFile);
const hashFiles = denodeify(function hashFilesCb(files, hash, callback) {
  if (!files.length) {
    callback(null);
    return;
  }
  const file = files.shift();
  fs.readFile(file, (err, data) => {
    if (err) {
      callback(err);
      return;
    } else {
      hash.update(data);
      hashFilesCb(files, hash, callback);
    }
  });
});
function buildAssetMap(dir, files, platform) {
  const platforms = new Set(platform != null ? [platform] : []);
  const assets = files.map((file) => AssetPaths.tryParse(file, platforms));
  const map = new Map();
  assets.forEach(function (asset, i) {
    if (asset == null) {
      return;
    }
    const file = files[i];
    const assetKey = getAssetKey(asset.assetName, asset.platform);
    let record = map.get(assetKey);
    if (!record) {
      record = {
        scales: [],
        files: [],
      };
      map.set(assetKey, record);
    }
    let insertIndex;
    const length = record.scales.length;
    for (insertIndex = 0; insertIndex < length; insertIndex++) {
      if (asset.resolution < record.scales[insertIndex]) {
        break;
      }
    }
    record.scales.splice(insertIndex, 0, asset.resolution);
    record.files.splice(insertIndex, 0, path.join(dir, file));
  });
  return map;
}
function getAssetKey(assetName, platform) {
  if (platform != null) {
    return `${assetName} : ${platform}`;
  } else {
    return assetName;
  }
}
async function getAbsoluteAssetRecord(assetPath, platform = null) {
  const filename = path.basename(assetPath);
  const dir = path.dirname(assetPath);
  const files = await readDir(dir);
  const assetData = AssetPaths.parse(
    filename,
    new Set(platform != null ? [platform] : [])
  );
  const map = buildAssetMap(dir, files, platform);
  let record;
  if (platform != null) {
    record =
      map.get(getAssetKey(assetData.assetName, platform)) ||
      map.get(assetData.assetName);
  } else {
    record = map.get(assetData.assetName);
  }
  if (!record) {
    throw new Error(
      `Asset not found: ${assetPath} for platform: ${
        platform !== null && platform !== void 0 ? platform : "(unspecified)"
      }`
    );
  }
  return record;
}
async function getAbsoluteAssetInfo(assetPath, platform = null) {
  const nameData = AssetPaths.parse(
    assetPath,
    new Set(platform != null ? [platform] : [])
  );
  const { name, type } = nameData;
  const { scales, files } = await getAbsoluteAssetRecord(assetPath, platform);
  const hasher = crypto.createHash("md5");
  if (files.length > 0) {
    await hashFiles(Array.from(files), hasher);
  }
  return {
    files,
    hash: hasher.digest("hex"),
    name,
    scales,
    type,
  };
}
async function getAssetData(
  assetPath,
  localPath,
  assetDataPlugins,
  platform = null,
  publicPath
) {
  // If the path of the asset is outside of the projectRoot, we don't want to
  // use `path.join` since this will generate an incorrect URL path. In that
  // case we just concatenate the publicPath with the relative path.
  let assetUrlPath = localPath.startsWith("..")
    ? publicPath.replace(/\/$/, "") + "/" + path.dirname(localPath)
    : path.join(publicPath, path.dirname(localPath));

  // On Windows, change backslashes to slashes to get proper URL path from file path.
  if (path.sep === "\\") {
    assetUrlPath = assetUrlPath.replace(/\\/g, "/");
  }
  const isImage = isAssetTypeAnImage(path.extname(assetPath).slice(1));
  const assetInfo = await getAbsoluteAssetInfo(assetPath, platform);
  const isImageInput = assetInfo.files[0].includes(".zip/")
    ? fs.readFileSync(assetInfo.files[0])
    : assetInfo.files[0];
  const dimensions = isImage ? imageSize(isImageInput) : null;
  const scale = assetInfo.scales[0];
  const assetData = {
    __packager_asset: true,
    fileSystemLocation: path.dirname(assetPath),
    httpServerLocation: assetUrlPath,
    width: dimensions ? dimensions.width / scale : undefined,
    height: dimensions ? dimensions.height / scale : undefined,
    scales: assetInfo.scales,
    files: assetInfo.files,
    hash: assetInfo.hash,
    name: assetInfo.name,
    type: assetInfo.type,
  };
  return await applyAssetDataPlugins(assetDataPlugins, assetData);
}
async function applyAssetDataPlugins(assetDataPlugins, assetData) {
  if (!assetDataPlugins.length) {
    return assetData;
  }
  const [currentAssetPlugin, ...remainingAssetPlugins] = assetDataPlugins;
  // $FlowFixMe: impossible to type a dynamic require.
  const assetPluginFunction = require(currentAssetPlugin);
  const resultAssetData = await assetPluginFunction(assetData);
  return await applyAssetDataPlugins(remainingAssetPlugins, resultAssetData);
}

/**
 * Returns all the associated files (for different resolutions) of an asset.
 **/
async function getAssetFiles(assetPath, platform = null) {
  const assetData = await getAbsoluteAssetRecord(assetPath, platform);
  return assetData.files;
}

/**
 * Return a buffer with the actual image given a request for an image by path.
 * The relativePath can contain a resolution postfix, in this case we need to
 * find that image (or the closest one to it's resolution) in one of the
 * project roots:
 *
 * 1. We first parse the directory of the asset
 * 2. We then build a map of all assets and their scales in this directory
 * 3. Then try to pick platform-specific asset records
 * 4. Then pick the closest resolution (rounding up) to the requested one
 */
async function getAsset(
  relativePath,
  projectRoot,
  watchFolders,
  platform = null,
  assetExts
) {
  const assetData = AssetPaths.parse(
    relativePath,
    new Set(platform != null ? [platform] : [])
  );
  const absolutePath = path.resolve(projectRoot, relativePath);
  if (!assetExts.includes(assetData.type)) {
    throw new Error(
      `'${relativePath}' cannot be loaded as its extension is not registered in assetExts`
    );
  }
  if (!pathBelongsToRoots(absolutePath, [projectRoot, ...watchFolders])) {
    throw new Error(
      `'${relativePath}' could not be found, because it cannot be found in the project root or any watch folder`
    );
  }
  const record = await getAbsoluteAssetRecord(absolutePath, platform);
  for (let i = 0; i < record.scales.length; i++) {
    if (record.scales[i] >= assetData.resolution) {
      return readFile(record.files[i]);
    }
  }
  return readFile(record.files[record.files.length - 1]);
}
function pathBelongsToRoots(pathToCheck, roots) {
  for (const rootFolder of roots) {
    if (pathToCheck.startsWith(path.resolve(rootFolder))) {
      return true;
    }
  }
  return false;
}
module.exports = {
  getAsset,
  getAssetData,
  getAssetFiles,
};
