package com.darkrockstudios.apps.hammer.projects.routes

import com.darkrockstudios.apps.hammer.base.ProjectId
import com.darkrockstudios.apps.hammer.base.http.HEADER_SYNC_ID
import com.darkrockstudios.apps.hammer.project.InvalidSyncIdException
import com.darkrockstudios.apps.hammer.project.ProjectDefinition
import com.darkrockstudios.apps.hammer.projects.ProjectsRepository.ProjectCreatedResult
import com.darkrockstudios.apps.hammer.utilities.SResult
import io.ktor.client.request.*
import io.ktor.http.*
import io.ktor.server.testing.*
import io.mockk.coEvery
import io.mockk.coVerify
import org.junit.jupiter.api.Test
import kotlin.test.assertEquals
import kotlin.test.assertTrue

class ProjectsRoutesCreateProjectTest : ProjectsRoutesBaseTest() {

	@Test
	fun `Projects - Create Project - Success`() = testApplication {
		val projectName = "TestProject"
		val projectId = ProjectId("uuid-1")
		val syncId = "syncId-test"
		val userId = 0L

		coEvery { accountsRepository.checkToken(userId, BEARER_TOKEN) } returns SResult.success(0L)
		coEvery { whiteListRepository.useWhiteList() } returns false
		coEvery {
			projectsRepository.createProject(
				userId = userId,
				syncId = syncId,
				projectName = projectName,
			)
		} returns SResult.success(
			ProjectCreatedResult(
				project = ProjectDefinition(projectName, projectId),
				alreadyExisted = false
			)
		)

		defaultApplication()

		client.get("api/projects/0/TestProject/create") {
			header("Authorization", "Bearer $BEARER_TOKEN")
			header(HEADER_SYNC_ID, syncId)
		}.apply {
			assertTrue(status.isSuccess())
			coVerify {
				projectsRepository.createProject(
					userId = userId,
					syncId = syncId,
					projectName = projectName,
				)
			}
		}
	}

	@Test
	fun `Projects - Create Project - Invalid Request`() = testApplication {
		coEvery { accountsRepository.checkToken(any(), any()) } returns SResult.success(0L)
		coEvery { whiteListRepository.useWhiteList() } returns false

		defaultApplication()

		client.get("api/projects/0/TestProject/create") {
			header("Authorization", "Bearer $BEARER_TOKEN")
		}.apply {
			assertEquals(HttpStatusCode.BadRequest, status)
		}
	}

	@Test
	fun `Projects - Create Project - Failure - Bad SyncId`() = testApplication {
		val projectName = "TestProject"
		val syncId = "syncId-test"
		val userId = 0L

		coEvery { accountsRepository.checkToken(userId, BEARER_TOKEN) } returns SResult.success(0L)
		coEvery { whiteListRepository.useWhiteList() } returns false
		coEvery {
			projectsRepository.createProject(
				userId = userId,
				syncId = syncId,
				projectName = projectName,
			)
		} returns SResult.failure(InvalidSyncIdException())

		defaultApplication()

		client.get("api/projects/0/TestProject/create") {
			header("Authorization", "Bearer $BEARER_TOKEN")
			header(HEADER_SYNC_ID, syncId)
		}.apply {
			assertEquals(HttpStatusCode.BadRequest, status)
		}
	}

	@Test
	fun `Projects - Create Project - Failure - Repository Exception`() = testApplication {
		val projectName = "TestProject"
		val syncId = "syncId-test"
		val userId = 0L

		coEvery { accountsRepository.checkToken(userId, BEARER_TOKEN) } returns SResult.success(0L)
		coEvery { whiteListRepository.useWhiteList() } returns false
		coEvery {
			projectsRepository.createProject(
				userId = userId,
				syncId = syncId,
				projectName = projectName,
			)
		} returns SResult.failure(Exception())

		defaultApplication()

		client.get("api/projects/0/TestProject/create") {
			header("Authorization", "Bearer $BEARER_TOKEN")
			header(HEADER_SYNC_ID, syncId)
		}.apply {
			assertEquals(HttpStatusCode.InternalServerError, status)
		}
	}
}

