package com.darkrockstudios.apps.hammer.e2e

import com.darkrockstudios.apps.hammer.base.http.*
import com.darkrockstudios.apps.hammer.e2e.util.EndToEndTest
import com.darkrockstudios.apps.hammer.e2e.util.TestDataSet1
import io.ktor.client.*
import io.ktor.client.call.*
import io.ktor.client.request.*
import io.ktor.client.request.forms.*
import io.ktor.client.statement.*
import io.ktor.http.*
import io.ktor.utils.io.core.*
import korlibs.io.compression.compress
import korlibs.io.compression.deflate.GZIP
import kotlinx.serialization.json.Json
import org.junit.jupiter.api.BeforeEach
import kotlin.test.assertEquals

abstract class ProjectSyncTestBase : EndToEndTest() {

	lateinit var json: Json

	@BeforeEach
	override fun setup() {
		super.setup()

		json = createJsonSerializer()
	}

	protected suspend fun HttpClient.projectSynchronizationBegan(
		userId: Long,
		authToken: Token,
		clientState: ClientEntityState
	): ProjectSynchronizationBegan {
		val beginSyncResponse = projectSynchronizationBeganRequest(userId, authToken, clientState)

		assertEquals(HttpStatusCode.OK, beginSyncResponse.status)
		val synchronizationBegan = beginSyncResponse.body<ProjectSynchronizationBegan>()
		return synchronizationBegan
	}

	protected suspend fun HttpClient.projectSynchronizationBeganRequest(
		userId: Long,
		authToken: Token,
		clientState: ClientEntityState
	): HttpResponse {
		val beginSyncResponse =
			post(api("project/$userId/${TestDataSet1.project1.name}/begin_sync")) {
				headers {
					append(HAMMER_PROTOCOL_HEADER, HAMMER_PROTOCOL_VERSION.toString())
					append("Authorization", "Bearer ${authToken.auth}")
				}
				contentType(ContentType.Application.OctetStream)
				parameter("projectId", TestDataSet1.project1.uuid.toString())

				val stateJson = json.encodeToString(ClientEntityState.serializer(), clientState)
				val compressed = stateJson.toByteArray().compress(GZIP)

				setBody(compressed)
			}

		return beginSyncResponse
	}

	protected suspend inline fun <reified T : ApiProjectEntity> HttpClient.uploadEntityRequest(
		userId: Long,
		authToken: Token,
		syncId: String,
		entity: T,
		originalhash: String,
		force: Boolean = false,
	): SaveEntityResponse {
		val uploadResponse =
			post(api("project/$userId/${TestDataSet1.project1.name}/upload_entity/${entity.id}")) {
				headers {
					append(HAMMER_PROTOCOL_HEADER, HAMMER_PROTOCOL_VERSION.toString())
					append("Authorization", "Bearer ${authToken.auth}")
					append(HEADER_SYNC_ID, syncId)
					append(HEADER_ORIGINAL_HASH, originalhash)
					append(HEADER_ENTITY_TYPE, entity.type.toString())
				}
				contentType(ContentType.Application.Json)
				url {
					parameters.append("projectId", TestDataSet1.project1.uuid.toString())
					parameters.append("force", force.toString())
				}

				setBody(entity)
			}
		assertEquals(HttpStatusCode.OK, uploadResponse.status)

		val savedEntity = uploadResponse.body<SaveEntityResponse>()
		return savedEntity
	}

	protected suspend inline fun <reified T : ApiProjectEntity> HttpClient.uploadConflictedEntityRequest(
		userId: Long,
		authToken: Token,
		syncId: String,
		entity: T,
		originalhash: String,
		force: Boolean = false,
	): T {
		val uploadResponse =
			post(api("project/$userId/${TestDataSet1.project1.name}/upload_entity/${entity.id}")) {
				headers {
					append(HAMMER_PROTOCOL_HEADER, HAMMER_PROTOCOL_VERSION.toString())
					append("Authorization", "Bearer ${authToken.auth}")
					append(HEADER_SYNC_ID, syncId)
					append(HEADER_ORIGINAL_HASH, originalhash)
					append(HEADER_ENTITY_TYPE, entity.type.toString())
				}
				contentType(ContentType.Application.Json)
				url {
					parameters.append("projectId", TestDataSet1.project1.uuid.toString())
					parameters.append("force", force.toString())
				}

				setBody(entity)
			}
		assertEquals(HttpStatusCode.Conflict, uploadResponse.status)

		val serverEntity = uploadResponse.body<T>()
		return serverEntity
	}

	protected suspend inline fun <reified T : ApiProjectEntity> HttpClient.downloadEntity(
		userId: Long,
		authToken: Token,
		syncId: String,
		entityId: Int,
		entityHash: String?,
	): T {
		// End Sync
		val downloadResponse =
			downloadEntityRequest(userId, authToken, syncId, entityId, entityHash)
		assertEquals(HttpStatusCode.OK, downloadResponse.status)

		val downloadedEntity: T = downloadResponse.body<T>()
		return downloadedEntity
	}

	protected suspend inline fun HttpClient.downloadEntityRequest(
		userId: Long,
		authToken: Token,
		syncId: String,
		entityId: Int,
		entityHash: String?,
	): HttpResponse {
		// End Sync
		val downloadResponse =
			get(api("project/$userId/${TestDataSet1.project1.name}/download_entity/${entityId}")) {
				headers {
					append(HAMMER_PROTOCOL_HEADER, HAMMER_PROTOCOL_VERSION.toString())
					append("Authorization", "Bearer ${authToken.auth}")
					append(HEADER_SYNC_ID, syncId)
					if (entityHash != null) {
						append(HEADER_ENTITY_HASH, entityHash)
					}
				}
				contentType(ContentType.Application.FormUrlEncoded)
				parameter("projectId", TestDataSet1.project1.uuid.toString())
			}
		return downloadResponse
	}

	protected suspend fun HttpClient.endSyncRequest(
		userId: Long,
		authToken: Token,
		synchronizationBegan: ProjectSynchronizationBegan
	) {
		// End Sync
		val endSyncResponse =
			post(api("project/$userId/${TestDataSet1.project1.name}/end_sync")) {
				headers {
					append(HAMMER_PROTOCOL_HEADER, HAMMER_PROTOCOL_VERSION.toString())
					append("Authorization", "Bearer ${authToken.auth}")
					append(HEADER_SYNC_ID, synchronizationBegan.syncId)
				}
				contentType(ContentType.Application.FormUrlEncoded)
				parameter("projectId", TestDataSet1.project1.uuid.toString())

				setBody(
					FormDataContent(
						Parameters.build {
							append("lastSync", synchronizationBegan.lastSync.toString())
							append("lastId", synchronizationBegan.lastId.toString())
						}
					)
				)
			}
		assertEquals(HttpStatusCode.OK, endSyncResponse.status)
	}
}