package com.sshdaemon.sshd;

import static org.hamcrest.CoreMatchers.is;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.containsInAnyOrder;
import static org.junit.jupiter.api.Assertions.assertFalse;
import static org.junit.jupiter.api.Assertions.assertTrue;

import org.junit.jupiter.api.Test;

import java.math.BigInteger;
import java.nio.file.Paths;
import java.security.KeyFactory;
import java.security.interfaces.RSAPublicKey;
import java.security.spec.RSAPublicKeySpec;

class SshPublicKeyAuthenticatorTest {

    private final SshPublicKeyAuthenticator sshPublicKeyAuthenticator = new SshPublicKeyAuthenticator();

    @Test
    void testLoadRSAKey() throws Exception {
        var rsaPublicKey = (RSAPublicKey) SshPublicKeyAuthenticator.readKey("ssh-rsa AAAAB3NzaC1yc2EAAAADAQABAAABAQCrkf5RHFcmmnPFxfOVsVOCdDVfs04dZg+/n808/NEdyOPuyAde4UIvZbzKEjW9brtEvOHCFfxZuXa0TbTIUau9p+4gWTGXIONcarwJ7LtNUlWfJiWYmIWVgyNnpzVftcW3mi8gRGxPbCJM2yVeB7gv452wvWPDe9TFdpgbwhLBqVIRG6EBHC0VBXX8qKNCbFoclYbiXa5DfwMkxYwN2yyKaSu75e0H4FP4BehaqQ6SfBIThqQRVdcx9J9Du3GzTi4ArN0timPAQ+X17pWxgEQ3qNbj49Lnteu+NSmb0PawcrP+Ykd7oy82kXm/hRM6cLjS1GOTsXpGDFf0NevAW8b3 D050150@WDFL34195932A");
        assertThat(rsaPublicKey.getPublicExponent(), is(new BigInteger("65537")));
        assertThat(rsaPublicKey.getModulus(), is(new BigInteger("21658742190318166967712730864679652658650859121969481181201380769435852715982079838796135745206268981260737877360141273622280512537469661232310601414632396577736750997307043633989350470146139654498683603607823966490835477269345553397205866827412911445557084380501015516582017566897110095005407768881980022943053565933828297090533987425102831869390057642704253755269803136323388759627399370507151238064778399477125470941103468997107204954580888346976963732529191611522789249471940599415587667163136427455499142265843852906870573003016543761403915579728832278943756709241709719567708592405407294409003276217649490282231")));
    }

    @Test
    void testLoadED25519Key() throws Exception {
        var publicKey = SshPublicKeyAuthenticator.readKey("ssh-ed25519 AAAAC3NzaC1lZDI1NTE5AAAAIGJ0j5BztROLdZYHf8cpJsJr9jd8gCRUfm6oe9k3Bhh0 @quantenzitrone:matrix.org");
        assertThat(publicKey.getFormat(), is("X.509"));
        assertThat(publicKey.getEncoded(), is(new byte[]{48, 42, 48, 5, 6, 3, 43, 101, 112, 3, 33, 0, 98, 116, -113, -112, 115, -75, 19, -117, 117, -106, 7, 127, -57, 41, 38, -62, 107, -10, 55, 124, -128, 36, 84, 126, 110, -88, 123, -39, 55, 6, 24, 116}));
    }

    @Test
    void testLoadKeyFromFilePath() throws Exception {
        var resourceDirectory = Paths.get("src", "test", "resources");
        var absolutePath = resourceDirectory.toFile().getAbsolutePath() + "/authorized_keys";
        assertTrue(sshPublicKeyAuthenticator.loadKeysFromPath(absolutePath));
        var authorizedKeys = sshPublicKeyAuthenticator.getAuthorizedKeys();
        assertThat(authorizedKeys.size(), is(3));

        var exponent = new BigInteger("65537");
        var firstKey = new RSAPublicKeySpec(new BigInteger("21658742190318166967712730864679652658650859121969481181201380769435852715982079838796135745206268981260737877360141273622280512537469661232310601414632396577736750997307043633989350470146139654498683603607823966490835477269345553397205866827412911445557084380501015516582017566897110095005407768881980022943053565933828297090533987425102831869390057642704253755269803136323388759627399370507151238064778399477125470941103468997107204954580888346976963732529191611522789249471940599415587667163136427455499142265843852906870573003016543761403915579728832278943756709241709719567708592405407294409003276217649490282231"), exponent);
        var secondKey = new RSAPublicKeySpec(new BigInteger("784057767550419878369497798651827476361889178814477573346641631234935186314436782078536135459192115951182846131604763290106347820711735919818466318881966145658619187844260657686465054388415729621256835109072466122751680324465523571218314239699133938274929722422531435916124040593004728158303703638151544229751515190620733194729793182402256827874540802963173001942073095959874409030457157131068008004452131416339302414300154381574660775550756346290523471370004641759457082400056951523140192837676235596014868691294116723696798672826048372197524626777597698825985438359440849049188507660150181938186465442057503356737043772475149570597456464086884083587865261320028768112850655672995490391301788008160746624607620536612729945152345637233101657918767620370276196646289217228948026575176667526067692435995447599542086540447642569281636925038610129227622664311974763550767950513197666055242104878427773497759504733742315824234665981633069516518731571901751350961661458615098275390788530389016622885595867572513280042783815166138155280065655067579686735407066393737630455891385113394433769584091954362175665925155421775122038568449049307648037245144977590866670732267987944408567171290527382426393459497954986775620782288149569534834718157"), exponent);
        var keyFactory = KeyFactory.getInstance("RSA");
        var ed25519Key = SshPublicKeyAuthenticator.readKey("ssh-ed25519 AAAAC3NzaC1lZDI1NTE5AAAAIGJ0j5BztROLdZYHf8cpJsJr9jd8gCRUfm6oe9k3Bhh0 @quantenzitrone:matrix.org");
        assertThat(authorizedKeys, containsInAnyOrder(
                keyFactory.generatePublic(firstKey),
                keyFactory.generatePublic(secondKey),
                ed25519Key));
    }

    @Test
    void testLoadDSAKeyFromFilePath() {
        var resourceDirectory = Paths.get("src", "test", "resources");
        var absolutePath = resourceDirectory.toFile().getAbsolutePath() + "/id_dsa.pub";
        assertFalse(sshPublicKeyAuthenticator.loadKeysFromPath(absolutePath));

        var authorizedKeys = sshPublicKeyAuthenticator.getAuthorizedKeys();
        assertThat(authorizedKeys.isEmpty(), is(true));
    }

    @Test
    void testAuthenticationWithDifferentKeys() throws Exception {
        var resourceDirectory = Paths.get("src", "test", "resources");
        var absolutePath = resourceDirectory.toFile().getAbsolutePath() + "/authorized_keys";
        assertTrue(sshPublicKeyAuthenticator.loadKeysFromPath(absolutePath));

        var exponent = new BigInteger("65537");
        var firstKey = new RSAPublicKeySpec(new BigInteger("21658742190318166967712730864679652658650859121969481181201380769435852715982079838796135745206268981260737877360141273622280512537469661232310601414632396577736750997307043633989350470146139654498683603607823966490835477269345553397205866827412911445557084380501015516582017566897110095005407768881980022943053565933828297090533987425102831869390057642704253755269803136323388759627399370507151238064778399477125470941103468997107204954580888346976963732529191611522789249471940599415587667163136427455499142265843852906870573003016543761403915579728832278943756709241709719567708592405407294409003276217649490282231"), exponent);
        var secondKey = new RSAPublicKeySpec(new BigInteger("784057767550419878369497798651827476361889178814477573346641631234935186314436782078536135459192115951182846131604763290106347820711735919818466318881966145658619187844260657686465054388415729621256835109072466122751680324465523571218314239699133938274929722422531435916124040593004728158303703638151544229751515190620733194729793182402256827874540802963173001942073095959874409030457157131068008004452131416339302414300154381574660775550756346290523471370004641759457082400056951523140192837676235596014868691294116723696798672826048372197524626777597698825985438359440849049188507660150181938186465442057503356737043772475149570597456464086884083587865261320028768112850655672995490391301788008160746624607620536612729945152345637233101657918767620370276196646289217228948026575176667526067692435995447599542086540447642569281636925038610129227622664311974763550767950513197666055242104878427773497759504733742315824234665981633069516518731571901751350961661458615098275390788530389016622885595867572513280042783815166138155280065655067579686735407066393737630455891385113394433769584091954362175665925155421775122038568449049307648037245144977590866670732267987944408567171290527382426393459497954986775620782288149569534834718157"), exponent);
        var thirdKey = new RSAPublicKeySpec(new BigInteger("801927917580758757979472556844251061760106383062627255825844583234477172790822215181804115531524750167110626578753641854839364115183295134199855565705144171967726133540156054269874189394350513031624322275630868353550082435946449319622372805788192807194970449569969122740243642313771155978930558037041288538032621167596518567745650173750903408431586471621670551481448236654798300138271016665710707996714088857315862552285448835022674484517837235751299196272185180808003335278208338200082797063402819475274804059251132953473839761384934730254695012273980048687260755946494082294536504365854421042209747262215825867288895826525289860017356087972232374369407427514550781201588459581132834605833136171102931440301693064676022315949585452254976583732273279685241548589989552958501585967916294792046822598474933776631974472781228885380669240991739694442254830793044605848889551686184197217760849351416181286067575078914864978102172921573494121943905751707671961364840577714192853806567108935276748552313084033867404574544169457199433681276796221829523529342719064559112634681882823253467088788554399120329634693450227027912364246515879972958896483379084082357407915577538153708816300929136449401980058929992848710222259418850185043358239451"), exponent);
        var keyFactory = KeyFactory.getInstance("RSA");

        var ed25519Key = SshPublicKeyAuthenticator.readKey("ssh-ed25519 AAAAC3NzaC1lZDI1NTE5AAAAIGJ0j5BztROLdZYHf8cpJsJr9jd8gCRUfm6oe9k3Bhh0 @quantenzitrone:matrix.org");

        assertThat(sshPublicKeyAuthenticator.authenticate(null, keyFactory.generatePublic(firstKey), null), is(true));
        assertThat(sshPublicKeyAuthenticator.authenticate(null, keyFactory.generatePublic(secondKey), null), is(true));
        assertThat(sshPublicKeyAuthenticator.authenticate(null, keyFactory.generatePublic(thirdKey), null), is(false));
        assertThat(sshPublicKeyAuthenticator.authenticate(null, ed25519Key, null), is(true));
    }

    @Test
    void testLoadInvalidKeyContent() throws Exception {
        var resourceDirectory = Paths.get("src", "test", "resources");
        var absolutePath = resourceDirectory.toFile().getAbsolutePath() + "/invalid_authorized_keys";
        assertFalse(sshPublicKeyAuthenticator.loadKeysFromPath(absolutePath));
        var authorizedKeys = sshPublicKeyAuthenticator.getAuthorizedKeys();
        assertThat(authorizedKeys.isEmpty(), is(true));
    }
}