package com.commit451.gitlab.activity

import android.animation.ArgbEvaluator
import android.animation.ObjectAnimator
import android.content.Context
import android.content.Intent
import android.graphics.Color
import android.os.Bundle
import androidx.palette.graphics.Palette
import coil.load
import com.commit451.addendum.themeAttrColor
import com.commit451.alakazam.navigationBarColorAnimator
import com.commit451.gitlab.R
import com.commit451.gitlab.databinding.ActivityUserBinding
import com.commit451.gitlab.extension.feedUrl
import com.commit451.gitlab.fragment.FeedFragment
import com.commit451.gitlab.image.PaletteImageViewTarget
import com.commit451.gitlab.model.api.User
import com.commit451.gitlab.util.ImageUtil

/**
 * User activity, which shows the user!
 */
class UserActivity : BaseActivity() {

    companion object {

        private const val KEY_USER = "user"

        fun newIntent(context: Context, user: User): Intent {
            val intent = Intent(context, UserActivity::class.java)
            intent.putExtra(KEY_USER, user)
            return intent
        }
    }

    private lateinit var binding: ActivityUserBinding
    private lateinit var user: User

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        binding = ActivityUserBinding.inflate(layoutInflater)
        setContentView(binding.root)
        user = intent.getParcelableExtra(KEY_USER)!!

        // Default content and scrim colors
        binding.collapsingToolbarLayout.setCollapsedTitleTextColor(Color.WHITE)
        binding.collapsingToolbarLayout.setExpandedTitleColor(Color.WHITE)

        binding.toolbar.setNavigationIcon(R.drawable.ic_back_24dp)
        binding.toolbar.setNavigationOnClickListener { onBackPressed() }
        binding.toolbar.title = user.username
        val url = ImageUtil.getAvatarUrl(
            user,
            resources.getDimensionPixelSize(R.dimen.user_header_image_size)
        )

        val paletteImageViewTarget = PaletteImageViewTarget(binding.backdrop) {
            bindPalette(it)
        }
        binding.backdrop.load(url) {
            allowHardware(false)
            target(paletteImageViewTarget)
        }

        if (savedInstanceState == null) {
            val fragmentTransaction = supportFragmentManager.beginTransaction()
            fragmentTransaction
                .add(R.id.user_feed, FeedFragment.newInstance(user.feedUrl))
                .commit()
        }
    }

    override fun onBackPressed() {
        supportFinishAfterTransition()
    }

    private fun bindPalette(palette: Palette) {
        val animationTime = 1000
        val vibrantColor = palette.getVibrantColor(this.themeAttrColor(R.attr.colorPrimary))

        window.navigationBarColorAnimator(vibrantColor)
            .setDuration(animationTime.toLong())
            .start()
        window.statusBarColor = vibrantColor

        ObjectAnimator.ofObject(
            binding.collapsingToolbarLayout, "contentScrimColor", ArgbEvaluator(),
            this.themeAttrColor(R.attr.colorPrimary), vibrantColor
        )
            .setDuration(animationTime.toLong())
            .start()

        ObjectAnimator.ofObject(
            binding.collapsingToolbarLayout, "statusBarScrimColor", ArgbEvaluator(),
            this.themeAttrColor(R.attr.colorPrimaryDark), vibrantColor
        )
            .setDuration(animationTime.toLong())
            .start()

        ObjectAnimator.ofObject(
            binding.toolbar, "titleTextColor", ArgbEvaluator(),
            Color.WHITE, palette.getDarkMutedColor(Color.BLACK)
        )
            .setDuration(animationTime.toLong())
            .start()
    }
}
