package com.commit451.gitlab.activity

import android.content.Context
import android.content.Intent
import android.os.Bundle
import android.view.MenuItem
import android.view.View
import android.view.ViewGroup
import android.widget.Toast
import androidx.appcompat.widget.Toolbar
import androidx.viewpager.widget.ViewPager
import butterknife.BindView
import butterknife.ButterKnife
import butterknife.OnClick
import com.commit451.gitlab.App
import com.commit451.gitlab.R
import com.commit451.gitlab.adapter.IssuePagerAdapter
import com.commit451.gitlab.event.IssueChangedEvent
import com.commit451.gitlab.event.IssueReloadEvent
import com.commit451.gitlab.extension.getUrl
import com.commit451.gitlab.extension.with
import com.commit451.gitlab.model.api.Issue
import com.commit451.gitlab.model.api.Project
import com.commit451.gitlab.navigation.Navigator
import com.commit451.gitlab.util.IntentUtil
import com.commit451.teleprinter.Teleprinter
import com.google.android.material.snackbar.Snackbar
import com.google.android.material.tabs.TabLayout
import io.reactivex.Single
import org.greenrobot.eventbus.Subscribe
import timber.log.Timber

/**
 * Shows off an issue like a bar of gold
 */
class IssueActivity : BaseActivity() {

    companion object {

        private const val EXTRA_PROJECT = "extra_project"
        private const val EXTRA_SELECTED_ISSUE = "extra_selected_issue"

        fun newIntent(context: Context, project: Project, issue: Issue): Intent {
            val intent = Intent(context, IssueActivity::class.java)
            intent.putExtra(EXTRA_PROJECT, project)
            intent.putExtra(EXTRA_SELECTED_ISSUE, issue)
            return intent
        }
    }

    @BindView(R.id.root)
    lateinit var root: ViewGroup
    @BindView(R.id.toolbar)
    lateinit var toolbar: Toolbar
    @BindView(R.id.tabs)
    lateinit var tabLayout: TabLayout
    @BindView(R.id.pager)
    lateinit var viewPager: ViewPager
    @BindView(R.id.progress)
    lateinit var progress: View

    private lateinit var menuItemOpenClose: MenuItem
    private lateinit var teleprinter: Teleprinter

    lateinit var project: Project
    lateinit var issue: Issue

    private val onMenuItemClickListener = Toolbar.OnMenuItemClickListener { item ->
        when (item.itemId) {
            R.id.action_share -> {
                IntentUtil.share(root, issue.getUrl(project))
                return@OnMenuItemClickListener true
            }
            R.id.action_close -> {
                closeOrOpenIssue()
                return@OnMenuItemClickListener true
            }
            R.id.action_delete -> {
                App.get().gitLab.deleteIssue(project.id, issue.iid)
                        .with(this)
                        .subscribe({
                            App.bus().post(IssueReloadEvent())
                            Toast.makeText(this@IssueActivity, R.string.issue_deleted, Toast.LENGTH_SHORT)
                                    .show()
                            finish()
                        }, {
                            Timber.e(it)
                            Snackbar.make(root, getString(R.string.failed_to_delete_issue), Snackbar.LENGTH_SHORT)
                                    .show()
                        })
                return@OnMenuItemClickListener true
            }
        }
        false
    }

    @OnClick(R.id.fab_edit_issue)
    fun onEditIssueClick() {
        val project = project
        val issue = issue
        Navigator.navigateToEditIssue(this@IssueActivity, project, issue)
    }

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        setContentView(R.layout.activity_issue)
        ButterKnife.bind(this)
        teleprinter = Teleprinter(this)
        App.bus().register(this)

        project = intent.getParcelableExtra(EXTRA_PROJECT)!!
        issue = intent.getParcelableExtra(EXTRA_SELECTED_ISSUE)!!

        toolbar.setNavigationIcon(R.drawable.ic_back_24dp)
        toolbar.setNavigationOnClickListener { onBackPressed() }
        toolbar.inflateMenu(R.menu.share)
        toolbar.inflateMenu(R.menu.close)
        toolbar.inflateMenu(R.menu.delete)
        menuItemOpenClose = toolbar.menu.findItem(R.id.action_close)
        toolbar.setOnMenuItemClickListener(onMenuItemClickListener)

        val sectionsPagerAdapter = IssuePagerAdapter(
                this,
                supportFragmentManager,
                project,
                issue)

        viewPager.adapter = sectionsPagerAdapter
        tabLayout.setupWithViewPager(viewPager)
        bindIssue()
    }

    override fun onDestroy() {
        App.bus().unregister(this)
        super.onDestroy()
    }

    private fun bindIssue() {
        setOpenCloseMenuStatus()
        toolbar.title = getString(R.string.issue_number, issue.iid)
        toolbar.subtitle = project.nameWithNamespace
    }

    private fun closeOrOpenIssue() {
        progress.visibility = View.VISIBLE
        if (issue.state == Issue.STATE_CLOSED) {
            updateIssueStatus(App.get().gitLab.updateIssueStatus(project.id, issue.iid, Issue.STATE_REOPEN))
        } else {
            updateIssueStatus(App.get().gitLab.updateIssueStatus(project.id, issue.iid, Issue.STATE_CLOSE))
        }
    }

    private fun updateIssueStatus(observable: Single<Issue>) {
        observable
                .with(this)
                .subscribe({
                    progress.visibility = View.GONE
                    this@IssueActivity.issue = it
                    App.bus().post(IssueChangedEvent(issue))
                    App.bus().post(IssueReloadEvent())
                    setOpenCloseMenuStatus()
                }, {
                    Timber.e(it)
                    progress.visibility = View.GONE
                    Snackbar.make(root, getString(R.string.error_changing_issue), Snackbar.LENGTH_SHORT)
                            .show()
                })
    }

    private fun setOpenCloseMenuStatus() {
        menuItemOpenClose.setTitle(if (issue.state == Issue.STATE_CLOSED) R.string.reopen else R.string.close)
    }

    @Subscribe
    fun onEvent(event: IssueChangedEvent) {
        if (issue.id == event.issue.id) {
            issue = event.issue
            bindIssue()
        }
    }
}
