package org.session.libsession.messaging.sending_receiving.link_preview;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonProperty;

import org.session.libsession.messaging.sending_receiving.attachments.Attachment;
import org.session.libsession.messaging.sending_receiving.attachments.AttachmentId;
import org.session.libsession.messaging.sending_receiving.attachments.DatabaseAttachment;
import org.session.libsignal.utilities.JsonUtil;
import org.session.libsignal.utilities.guava.Optional;

import java.io.IOException;
import java.util.Objects;

public class LinkPreview {

  @JsonProperty
  private final String       url;

  @JsonProperty
  private final String       title;

  @JsonProperty
  private final AttachmentId attachmentId;

  @JsonIgnore
  public Optional<Attachment> thumbnail;

  public LinkPreview(@NonNull String url, @NonNull String title, @NonNull DatabaseAttachment thumbnail) {
    this.url          = url;
    this.title        = title;
    this.thumbnail    = Optional.of(thumbnail);
    this.attachmentId = thumbnail.getAttachmentId();
  }

  public LinkPreview(@NonNull String url, @NonNull String title, @NonNull Optional<Attachment> thumbnail) {
    this.url          = url;
    this.title        = title;
    this.thumbnail    = thumbnail;
    this.attachmentId = null;
  }

  public LinkPreview(@JsonProperty("url")          @NonNull  String url,
                     @JsonProperty("title")        @NonNull  String title,
                     @JsonProperty("attachmentId") @Nullable AttachmentId attachmentId)
  {
    this.url          = url;
    this.title        = title;
    this.attachmentId = attachmentId;
    this.thumbnail    = Optional.absent();
  }

  public String getUrl() {
    return url;
  }

  public String getTitle() {
    return title;
  }

  public Optional<Attachment> getThumbnail() {
    return thumbnail;
  }

  public @Nullable AttachmentId getAttachmentId() {
    return attachmentId;
  }

  public String serialize() throws IOException {
    return JsonUtil.toJsonThrows(this);
  }

  public static LinkPreview deserialize(@NonNull String serialized) throws IOException {
    return JsonUtil.fromJson(serialized, LinkPreview.class);
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) return true;
    if (o == null || getClass() != o.getClass()) return false;
    LinkPreview that = (LinkPreview) o;
    return Objects.equals(url, that.url) && Objects.equals(title, that.title) && Objects.equals(attachmentId, that.attachmentId) && Objects.equals(thumbnail, that.thumbnail);
  }

  @Override
  public int hashCode() {
    return Objects.hash(url, title, attachmentId, thumbnail);
  }
}
