/*
 * Copyright (C) 2012 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

package com.android.talkback.menurules;

import android.content.Context;
import android.support.v4.view.accessibility.AccessibilityNodeInfoCompat;
import com.google.android.marvin.talkback.TalkBackService;
import com.android.talkback.contextmenu.ContextMenuItem;
import com.android.talkback.contextmenu.ContextMenuItemBuilder;
import com.android.talkback.contextmenu.RadialMenuItem;

import java.util.List;

interface NodeMenuRule {
    /**
     * Determines whether this rule should process the specified node.
     *
     * @param service The parent service.
     * @param node The node to filter.
     * @return {@code true} if this rule should process the node.
     */
    public boolean accept(TalkBackService service, AccessibilityNodeInfoCompat node);

    /**
     * Processes the specified node and returns a {@link List} of relevant local
     * {@link ContextMenuItem}s for that node.
     * <p>
     * Note: The validity of the node is guaranteed only within the scope of
     * this method. If this node is used for future actions, as in an
     * onItemClicked callback, a copy of this node must be retained and recycled
     * by the rule.
     * <p>
     *
     * @param service The parent service.
     * @param menuItemBuilder builder to create menu items
     * @param node The node to process.
     */
    public List<ContextMenuItem> getMenuItemsForNode(
            TalkBackService service, ContextMenuItemBuilder menuItemBuilder,
            AccessibilityNodeInfoCompat node);

    /**
     * Provides the menu rule processor with a potentially user-visible name for
     * the submenu the {@link RadialMenuItem}s returned by this rule appear in.
     * If more than one rule is applied to a node, each rule's menu items
     * returned from {@link #getMenuItemsForNode(TalkBackService,
     * ContextMenuItemBuilder menuItemBuilder, AccessibilityNodeInfoCompat)} will appear in
     * a submenu with the label as returned from this method.
     *
     * @return The user-visible label for the menu item grouping generated by
     *         this rule.
     */
    public CharSequence getUserFriendlyMenuName(Context context);

    /**
     * @return Whether this menu can be collapsed and its children moved up to
     *         the parent menu if the parent contains no other menus.
     */
    public boolean canCollapseMenu();
}
