package com.google.android.libraries.accessibility.utils.screencapture;

import android.app.Activity;
import android.content.Intent;
import android.media.projection.MediaProjectionManager;
import android.os.Bundle;
import android.os.Parcelable;
import androidx.localbroadcastmanager.content.LocalBroadcastManager;
import com.google.android.libraries.accessibility.utils.log.LogUtils;
import com.google.android.libraries.accessibility.utils.screencapture.ScreenCaptureController.ScreenshotAuthorizationReceiver;

/**
 * Short-lived proxy activity without UI for launching {@link MediaProjectionManager}'s user consent
 * flow for obtaining a screen capture token.
 * <p>
 * {@link ScreenCaptureController} starts this {@link Activity} via an {@link Intent} containing a
 * {@link Parcelable} {@code extra} with the key defined by
 * {@link #INTENT_EXTRA_SCREEN_CAPTURE_INTENT}. The extra is another {@code Intent} generated by
 * {@link MediaProjectionManager#createScreenCaptureIntent()} and is used to start a consent UI for
 * obtaining screen capture data. This activity starts the system-generated {@code Intent} and waits
 * for a result. When the user authorizes or cancels the request, the result is passed to
 * {@link #onActivityResult(int, int, Intent)}. This class then uses {@link LocalBroadcastManager}
 * to pass the result back to a listener held by {@code ScreenCaptureController}.
 */
public class ScreenshotAuthProxyActivity extends Activity {

  public static final String INTENT_EXTRA_SCREEN_CAPTURE_INTENT =
      "com.google.android.libraries.accessibility.utils.screencapture.EXTRA_SCREEN_CAPTURE_INTENT";

  private static final String TAG = "ScreenshotAuthProxyActivity";

  private static final int INTENT_REQUEST_CODE_SCREEN_CAPTURE_AUTH = 1000;

  private LocalBroadcastManager broadcastManager;

  @Override
  protected void onCreate(Bundle savedInstanceState) {
    super.onCreate(savedInstanceState);
    broadcastManager = LocalBroadcastManager.getInstance(this);
    Intent launchIntent = getIntent();
    Intent systemIntent =
        (Intent) launchIntent.getParcelableExtra(INTENT_EXTRA_SCREEN_CAPTURE_INTENT);
    if (systemIntent == null) {
      // A system-generated screen capture authorization Intent must be passed to this activity as
      // an Intent extra with the key INTENT_EXTRA_SCREEN_CAPTURE_INTENT. This Intent is obtained
      // from MediaProjectionManager and should be added as a parcelable to the Intent that launches
      // this activity.
      LogUtils.e(TAG, "Could not start authorization as no MediaProjection intent was provided.");
      finish();
    }
    startActivityForResult(systemIntent, INTENT_REQUEST_CODE_SCREEN_CAPTURE_AUTH);
  }

  @Override
  protected void onActivityResult(int requestCode, int resultCode, Intent data) {
    super.onActivityResult(requestCode, resultCode, data);
    if (requestCode != INTENT_REQUEST_CODE_SCREEN_CAPTURE_AUTH) {
      LogUtils.e(TAG, "Incorrect request code for activity result");
      return;
    }

    if (resultCode == Activity.RESULT_OK) {
      Intent wrapperIntent =
          new Intent(ScreenshotAuthorizationReceiver.ACTION_SCREEN_CAPTURE_AUTHORIZED);
      wrapperIntent.putExtra(
          ScreenshotAuthorizationReceiver.INTENT_EXTRA_SCREEN_CAPTURE_AUTH_INTENT, data);
      broadcastManager.sendBroadcast(wrapperIntent);
    } else {
      broadcastManager.sendBroadcast(
          new Intent(ScreenshotAuthorizationReceiver.ACTION_SCREEN_CAPTURE_NOT_AUTHORIZED));
    }

    finish();
  }
}
