/*
 * Copyright (C) 2012 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

package com.google.android.accessibility.talkback.menurules;

import android.content.Context;
import android.content.SharedPreferences;
import androidx.core.view.accessibility.AccessibilityNodeInfoCompat;
import com.google.android.accessibility.utils.SharedPreferencesUtils;

/** Rule to create node menu for the {@link AccessibilityNodeInfoCompat}. */
public abstract class NodeMenuRule implements NodeMenu {

  private final int showPrefKeyResId;
  private final int showPrefKeyResIdDefault;

  NodeMenuRule(int showPrefKeyResId, int showPrefKeyResIdDefault) {
    this.showPrefKeyResId = showPrefKeyResId;
    this.showPrefKeyResIdDefault = showPrefKeyResIdDefault;
  }

  /**
   * Determines whether this rule is enabled. The node menu is allowed to be shown if this rule is
   * enabled.
   *
   * @param context The parent context.
   * @return {@code true} if it is enabled.
   */
  boolean isEnabled(Context context) {
    SharedPreferences prefs = SharedPreferencesUtils.getSharedPreferences(context);
    return prefs.getBoolean(
        context.getString(showPrefKeyResId),
        context.getResources().getBoolean(showPrefKeyResIdDefault));
  }

  /**
   * Provides the menu rule processor with a potentially user-visible name for the submenu. If more
   * than one rule is applied to a node, each rule's menu items returned from {@link
   * #getMenuItemsForNode(Context, AccessibilityNodeInfoCompat, boolean)} will appear in a submenu
   * with the label as returned from this method.
   *
   * @return The user-visible label for the menu item grouping generated by this rule.
   */
  abstract CharSequence getUserFriendlyMenuName(Context context);

  /** @return Whether this NodeMenuRule is used for sub menu */
  boolean isSubMenu() {
    return true;
  }
}
