/*
 * Copyright (C) 2017 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

package com.google.android.accessibility.talkback.compositor;

import android.content.Context;
import androidx.annotation.Nullable;
import com.google.android.accessibility.talkback.compositor.parsetree.ParseTree;
import com.google.android.accessibility.utils.output.SpeechCleanupUtils;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;

/** A VariableDelegate that maps data from EventInterpretation. */
class InterpretationVariables implements ParseTree.VariableDelegate {

  private static final String TAG = "InterpretationVariables";

  // IDs of variables.
  private static final int EVENT_TEXT_OR_DESCRIPTION = 5000;
  private static final int EVENT_REMOVED_TEXT = 5001;
  private static final int EVENT_ADDED_TEXT = 5002;
  private static final int EVENT_TRAVERSED_TEXT = 5003;
  private static final int EVENT_DESELECTED_TEXT = 5004;
  private static final int EVENT_SELECTED_TEXT = 5005;
  private static final int EVENT_LAST_WORD = 5006;
  private static final int EVENT_IS_CUT = 5007;
  private static final int EVENT_IS_PASTE = 5008;
  private static final int EVENT_HINT_TYPE = 5009;
  private static final int EVENT_HINT_FORCE_FEEDBACK_EVEN_IF_AUDIO_PLAYBACK_ACTIVE = 5010;
  private static final int EVENT_HINT_FORCE_MICROPHONE_ACTIVE = 5011;
  private static final int EVENT_HINT_TEXT = 5012;
  private static final int EVENT_FORCE_FEEDBACK_EVEN_IF_AUDIO_PLAYBACK_ACTIVE = 5013;
  private static final int EVENT_FORCE_FEEDBACK_EVEN_IF_MICROPHONE_ACTIVE = 5014;
  private static final int EVENT_FORCE_FEEDBACK_EVEN_IF_SSB_ACTIVE = 5015;
  private static final int EVENT_IS_INITIAL_FOCUS = 5017;
  private static final int EVENT_IS_USER_NAVIGATION = 5018;

  // IDs of enums.
  private static final int ENUM_HINT_TYPE = 5200;

  private final Context mContext;
  private final ParseTree.VariableDelegate mParent;
  private final EventInterpretation mEventInterpretation;
  @Nullable private final Locale mUserPreferredLocale;

  /**
   * Constructs InterpretationVariables, which contains context variables to help generate feedback
   * for an accessibility event.
   */
  InterpretationVariables(
      Context context,
      ParseTree.VariableDelegate parent,
      EventInterpretation eventInterpreted,
      GlobalVariables globalVariables) {
    mUserPreferredLocale = globalVariables.getUserPreferredLocale();
    mContext = context;
    mParent = parent;
    mEventInterpretation = eventInterpreted;
    // TODO: We should have a variable mLocale like we have in NodeVariables to
    // capture the locale if the event is from IME. gBoard uses Nodevariables to wrap the locale
    // for keyboard input. Currently a few IMEs do not send the source with the event.
    // So it is not clear how to detect if the event was generated by IME.
    // The captured locale should be wrapped around event text and event description.
  }

  @Override
  public boolean getBoolean(int variableId) {
    switch (variableId) {
      case EVENT_IS_CUT:
        return AccessibilityInterpretationFeedbackUtils.safeTextInterpretation(mEventInterpretation)
            .getIsCutAction();
      case EVENT_IS_PASTE:
        return AccessibilityInterpretationFeedbackUtils.safeTextInterpretation(mEventInterpretation)
            .getIsPasteAction();
      case EVENT_HINT_FORCE_FEEDBACK_EVEN_IF_AUDIO_PLAYBACK_ACTIVE:
        {
          @Nullable HintEventInterpretation hintInterp = mEventInterpretation.getHint();
          return (hintInterp != null) && hintInterp.getForceFeedbackEvenIfAudioPlaybackActive();
        }
      case EVENT_HINT_FORCE_MICROPHONE_ACTIVE:
        {
          @Nullable HintEventInterpretation hintInterp = mEventInterpretation.getHint();
          return (hintInterp != null) && hintInterp.getForceFeedbackEvenIfMicrophoneActive();
        }
      case EVENT_FORCE_FEEDBACK_EVEN_IF_AUDIO_PLAYBACK_ACTIVE:
        {
          return AccessibilityInterpretationFeedbackUtils.safeAccessibilityFocusInterpretation(
                  mEventInterpretation)
              .getForceFeedbackEvenIfAudioPlaybackActive();
        }
      case EVENT_FORCE_FEEDBACK_EVEN_IF_MICROPHONE_ACTIVE:
        {
          return AccessibilityInterpretationFeedbackUtils.safeAccessibilityFocusInterpretation(
                  mEventInterpretation)
              .getForceFeedbackEvenIfMicrophoneActive();
        }
      case EVENT_FORCE_FEEDBACK_EVEN_IF_SSB_ACTIVE:
        {
          return AccessibilityInterpretationFeedbackUtils.safeAccessibilityFocusInterpretation(
                  mEventInterpretation)
              .getForceFeedbackEvenIfSsbActive();
        }
      case EVENT_IS_INITIAL_FOCUS:
        {
          return AccessibilityInterpretationFeedbackUtils.safeAccessibilityFocusInterpretation(
                  mEventInterpretation)
              .getIsInitialFocusAfterScreenStateChange();
        }
      case EVENT_IS_USER_NAVIGATION:
        {
          return AccessibilityInterpretationFeedbackUtils.safeAccessibilityFocusInterpretation(
                  mEventInterpretation)
              .getIsNavigateByUser();
        }
      default:
        return mParent.getBoolean(variableId);
    }
  }

  @Override
  public int getInteger(int variableId) {
    return mParent.getInteger(variableId);
  }

  @Override
  public double getNumber(int variableId) {
    return mParent.getNumber(variableId);
  }

  @Override
  @Nullable
  public CharSequence getString(int variableId) {
    // TODO: Remove collapseRepeatedCharactersAndCleanUp() from VariableDelegate classes. Instead,
    // apply collapseRepeatedCharactersAndCleanUp() to Compositor ttsOutput result whenever
    // Compositor output ttsOutputClean returns true (default is true).
    // TODO: Use spans to mark which parts of composed text are already clean (or should never be
    // cleaned).
    CharSequence text = getStringInternal(variableId);
    if (text == null) {
      return mParent.getString(variableId);
    } else {
      return SpeechCleanupUtils.collapseRepeatedCharactersAndCleanUp(mContext, text);
    }
  }

  @Nullable
  private CharSequence getStringInternal(int variableId) {
    switch (variableId) {
      case EVENT_TEXT_OR_DESCRIPTION:
        return AccessibilityInterpretationFeedbackUtils.safeTextInterpretation(mEventInterpretation)
            .getTextOrDescription();
      case EVENT_REMOVED_TEXT:
        return AccessibilityInterpretationFeedbackUtils.safeTextInterpretation(mEventInterpretation)
            .getRemovedText();
      case EVENT_ADDED_TEXT:
        return AccessibilityInterpretationFeedbackUtils.safeTextInterpretation(mEventInterpretation)
            .getAddedText();
      case EVENT_TRAVERSED_TEXT:
        return AccessibilityInterpretationFeedbackUtils.getEventTraversedText(
            mEventInterpretation, mUserPreferredLocale);
      case EVENT_DESELECTED_TEXT:
        return AccessibilityInterpretationFeedbackUtils.safeTextInterpretation(mEventInterpretation)
            .getDeselectedText();
      case EVENT_SELECTED_TEXT:
        return AccessibilityInterpretationFeedbackUtils.safeTextInterpretation(mEventInterpretation)
            .getSelectedText();
      case EVENT_LAST_WORD:
        return AccessibilityInterpretationFeedbackUtils.safeTextInterpretation(mEventInterpretation)
            .getInitialWord();
      case EVENT_HINT_TEXT:
        {
          @Nullable HintEventInterpretation hintInterp = mEventInterpretation.getHint();
          return (hintInterp == null) ? "" : hintInterp.getText();
        }
      default:
        return null;
    }
  }

  @Override
  public int getEnum(int variableId) {
    switch (variableId) {
      case EVENT_HINT_TYPE:
        {
          @Nullable HintEventInterpretation hintInterp = mEventInterpretation.getHint();
          return (hintInterp == null)
              ? HintEventInterpretation.HINT_TYPE_NONE
              : hintInterp.getHintType();
        }
      default:
        return mParent.getEnum(variableId);
    }
  }

  @Override
  @Nullable
  public ParseTree.VariableDelegate getReference(int variableId) {
    return mParent.getReference(variableId);
  }

  @Override
  public int getArrayLength(int variableId) {
    return mParent.getArrayLength(variableId);
  }

  @Override
  @Nullable
  public CharSequence getArrayStringElement(int variableId, int index) {
    return mParent.getArrayStringElement(variableId, index);
  }

  @Override
  @Nullable
  public ParseTree.VariableDelegate getArrayChildElement(int variableId, int index) {
    return mParent.getArrayChildElement(variableId, index);
  }

  static void declareVariables(ParseTree parseTree) {

    // Enum values for hint type.
    Map<Integer, String> hintTypes = new HashMap<>();
    hintTypes.put(HintEventInterpretation.HINT_TYPE_NONE, "none");
    hintTypes.put(HintEventInterpretation.HINT_TYPE_ACCESSIBILITY_FOCUS, "access_focus");
    hintTypes.put(HintEventInterpretation.HINT_TYPE_INPUT_FOCUS, "input_focus");
    hintTypes.put(HintEventInterpretation.HINT_TYPE_SCREEN, "screen");
    hintTypes.put(HintEventInterpretation.HINT_TYPE_SELECTOR, "selector");
    hintTypes.put(HintEventInterpretation.HINT_TYPE_TEXT_SUGGESTION, "text_suggestion");
    hintTypes.put(HintEventInterpretation.HINT_TYPE_TYPO, "typo");
    parseTree.addEnum(ENUM_HINT_TYPE, hintTypes);

    parseTree.addStringVariable("event.textOrDescription", EVENT_TEXT_OR_DESCRIPTION);
    parseTree.addStringVariable("event.removedText", EVENT_REMOVED_TEXT);
    parseTree.addStringVariable("event.addedText", EVENT_ADDED_TEXT);
    parseTree.addStringVariable("event.traversedText", EVENT_TRAVERSED_TEXT);
    parseTree.addStringVariable("event.deselectedText", EVENT_DESELECTED_TEXT);
    parseTree.addStringVariable("event.selectedText", EVENT_SELECTED_TEXT);
    parseTree.addStringVariable("event.initialWord", EVENT_LAST_WORD);
    parseTree.addBooleanVariable("event.isCut", EVENT_IS_CUT);
    parseTree.addBooleanVariable("event.isPaste", EVENT_IS_PASTE);
    parseTree.addEnumVariable("event.hintType", EVENT_HINT_TYPE, ENUM_HINT_TYPE);
    parseTree.addBooleanVariable(
        "event.hintForceFeedbackEvenIfAudioPlaybackActive",
        EVENT_HINT_FORCE_FEEDBACK_EVEN_IF_AUDIO_PLAYBACK_ACTIVE);
    parseTree.addBooleanVariable(
        "event.hintForceFeedackEvenIfMicrophoneActive", EVENT_HINT_FORCE_MICROPHONE_ACTIVE);
    parseTree.addStringVariable("event.hintText", EVENT_HINT_TEXT);
    parseTree.addBooleanVariable(
        "event.forceFeedbackEvenIfAudioPlaybackActive",
        EVENT_FORCE_FEEDBACK_EVEN_IF_AUDIO_PLAYBACK_ACTIVE);
    parseTree.addBooleanVariable(
        "event.forceFeedbackEvenIfMicrophoneActive",
        EVENT_FORCE_FEEDBACK_EVEN_IF_MICROPHONE_ACTIVE);
    parseTree.addBooleanVariable(
        "event.forceFeedbackEvenIfSsbActive", EVENT_FORCE_FEEDBACK_EVEN_IF_SSB_ACTIVE);
    parseTree.addBooleanVariable("event.isInitialFocus", EVENT_IS_INITIAL_FOCUS);
    parseTree.addBooleanVariable("event.isNavigateByUser", EVENT_IS_USER_NAVIGATION);
  }
}
