/*
 * Copyright (C) 2012 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

package com.google.android.accessibility.talkback.menurules;

import android.accessibilityservice.AccessibilityService;
import android.content.Context;
import android.content.SharedPreferences;
import androidx.core.view.accessibility.AccessibilityNodeInfoCompat;
import com.google.android.accessibility.talkback.contextmenu.ContextMenuItem;
import com.google.android.accessibility.utils.SharedPreferencesUtils;
import java.util.List;

abstract class NodeMenuRule {

  private final int showPrefKeyResId;
  private final int showPrefKeyResIdDefault;

  NodeMenuRule(int showPrefKeyResId, int showPrefKeyResIdDefault) {
    this.showPrefKeyResId = showPrefKeyResId;
    this.showPrefKeyResIdDefault = showPrefKeyResIdDefault;
  }

  /**
   * Determines whether this rule should process the specified node.
   *
   * @param service The parent service.
   * @param node The node to filter.
   * @return {@code true} if this rule should process the node.
   */
  abstract boolean accept(AccessibilityService service, AccessibilityNodeInfoCompat node);

  /**
   * Determines whether this rule is enabled. The node menu is allowed to be shown if this rule is
   * enabled.
   *
   * @param context The parent context.
   * @return {@code true} if it is enabled.
   */
  boolean isEnabled(Context context) {
    SharedPreferences prefs = SharedPreferencesUtils.getSharedPreferences(context);
    return prefs.getBoolean(
        context.getString(showPrefKeyResId),
        context.getResources().getBoolean(showPrefKeyResIdDefault));
  }

  /**
   * Processes the specified node and returns a {@link List} of relevant local {@link
   * ContextMenuItem}s for that node.
   *
   * <p>Note: The validity of the node is guaranteed only within the scope of this method. If this
   * node is used for future actions, as in an onItemClicked callback, a copy of this node must be
   * retained and recycled by the rule.
   *
   * <p>
   *
   * @param service The parent service.
   * @param node The node to process
   * @param includeAncestors sets to {@code false} to find menu items from the node itself only.
   *     Sets to {@code true} may find menu items from its ancestors.
   */
  abstract List<ContextMenuItem> getMenuItemsForNode(
      AccessibilityService service, AccessibilityNodeInfoCompat node, boolean includeAncestors);

  /**
   * Provides the menu rule processor with a potentially user-visible name for the submenu. If more
   * than one rule is applied to a node, each rule's menu items returned from {@link
   * #getMenuItemsForNode(AccessibilityService, AccessibilityNodeInfoCompat, boolean)} will appear
   * in a submenu with the label as returned from this method.
   *
   * @return The user-visible label for the menu item grouping generated by this rule.
   */
  abstract CharSequence getUserFriendlyMenuName(Context context);

  /** @return Whether this NodeMenuRule is used for sub menu */
  boolean isSubMenu() {
    return true;
  }
}
