import { DOCUMENT } from '@angular/common';
import { Inject, Injectable } from '@angular/core';
import { getPlatforms, isPlatform } from '@ionic/core';
import { Subject } from 'rxjs';
import * as i0 from "@angular/core";
export class Platform {
    constructor(doc, zone) {
        this.doc = doc;
        /**
         * @hidden
         */
        this.backButton = new Subject();
        /**
         * The keyboardDidShow event emits when the
         * on-screen keyboard is presented.
         */
        this.keyboardDidShow = new Subject();
        /**
         * The keyboardDidHide event emits when the
         * on-screen keyboard is hidden.
         */
        this.keyboardDidHide = new Subject();
        /**
         * The pause event emits when the native platform puts the application
         * into the background, typically when the user switches to a different
         * application. This event would emit when a Cordova app is put into
         * the background, however, it would not fire on a standard web browser.
         */
        this.pause = new Subject();
        /**
         * The resume event emits when the native platform pulls the application
         * out from the background. This event would emit when a Cordova app comes
         * out from the background, however, it would not fire on a standard web browser.
         */
        this.resume = new Subject();
        /**
         * The resize event emits when the browser window has changed dimensions. This
         * could be from a browser window being physically resized, or from a device
         * changing orientation.
         */
        this.resize = new Subject();
        zone.run(() => {
            var _a;
            this.win = doc.defaultView;
            this.backButton.subscribeWithPriority = function (priority, callback) {
                return this.subscribe((ev) => {
                    return ev.register(priority, (processNextHandler) => zone.run(() => callback(processNextHandler)));
                });
            };
            proxyEvent(this.pause, doc, 'pause');
            proxyEvent(this.resume, doc, 'resume');
            proxyEvent(this.backButton, doc, 'ionBackButton');
            proxyEvent(this.resize, this.win, 'resize');
            proxyEvent(this.keyboardDidShow, this.win, 'ionKeyboardDidShow');
            proxyEvent(this.keyboardDidHide, this.win, 'ionKeyboardDidHide');
            let readyResolve;
            this._readyPromise = new Promise((res) => {
                readyResolve = res;
            });
            if ((_a = this.win) === null || _a === void 0 ? void 0 : _a['cordova']) {
                doc.addEventListener('deviceready', () => {
                    readyResolve('cordova');
                }, { once: true });
            }
            else {
                // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
                readyResolve('dom');
            }
        });
    }
    /**
     * @returns returns true/false based on platform.
     * @description
     * Depending on the platform the user is on, `is(platformName)` will
     * return `true` or `false`. Note that the same app can return `true`
     * for more than one platform name. For example, an app running from
     * an iPad would return `true` for the platform names: `mobile`,
     * `ios`, `ipad`, and `tablet`. Additionally, if the app was running
     * from Cordova then `cordova` would be true, and if it was running
     * from a web browser on the iPad then `mobileweb` would be `true`.
     *
     * ```
     * import { Platform } from 'ionic-angular';
     *
     * @Component({...})
     * export MyPage {
     *   constructor(public platform: Platform) {
     *     if (this.platform.is('ios')) {
     *       // This will only print when on iOS
     *       console.log('I am an iOS device!');
     *     }
     *   }
     * }
     * ```
     *
     * | Platform Name   | Description                        |
     * |-----------------|------------------------------------|
     * | android         | on a device running Android.       |
     * | capacitor       | on a device running Capacitor.     |
     * | cordova         | on a device running Cordova.       |
     * | ios             | on a device running iOS.           |
     * | ipad            | on an iPad device.                 |
     * | iphone          | on an iPhone device.               |
     * | phablet         | on a phablet device.               |
     * | tablet          | on a tablet device.                |
     * | electron        | in Electron on a desktop device.   |
     * | pwa             | as a PWA app.                      |
     * | mobile          | on a mobile device.                |
     * | mobileweb       | on a mobile device in a browser.   |
     * | desktop         | on a desktop device.               |
     * | hybrid          | is a cordova or capacitor app.     |
     *
     */
    is(platformName) {
        return isPlatform(this.win, platformName);
    }
    /**
     * @returns the array of platforms
     * @description
     * Depending on what device you are on, `platforms` can return multiple values.
     * Each possible value is a hierarchy of platforms. For example, on an iPhone,
     * it would return `mobile`, `ios`, and `iphone`.
     *
     * ```
     * import { Platform } from 'ionic-angular';
     *
     * @Component({...})
     * export MyPage {
     *   constructor(public platform: Platform) {
     *     // This will print an array of the current platforms
     *     console.log(this.platform.platforms());
     *   }
     * }
     * ```
     */
    platforms() {
        return getPlatforms(this.win);
    }
    /**
     * Returns a promise when the platform is ready and native functionality
     * can be called. If the app is running from within a web browser, then
     * the promise will resolve when the DOM is ready. When the app is running
     * from an application engine such as Cordova, then the promise will
     * resolve when Cordova triggers the `deviceready` event.
     *
     * The resolved value is the `readySource`, which states which platform
     * ready was used. For example, when Cordova is ready, the resolved ready
     * source is `cordova`. The default ready source value will be `dom`. The
     * `readySource` is useful if different logic should run depending on the
     * platform the app is running from. For example, only Cordova can execute
     * the status bar plugin, so the web should not run status bar plugin logic.
     *
     * ```
     * import { Component } from '@angular/core';
     * import { Platform } from 'ionic-angular';
     *
     * @Component({...})
     * export MyApp {
     *   constructor(public platform: Platform) {
     *     this.platform.ready().then((readySource) => {
     *       console.log('Platform ready from', readySource);
     *       // Platform now ready, execute any required native code
     *     });
     *   }
     * }
     * ```
     */
    ready() {
        return this._readyPromise;
    }
    /**
     * Returns if this app is using right-to-left language direction or not.
     * We recommend the app's `index.html` file already has the correct `dir`
     * attribute value set, such as `<html dir="ltr">` or `<html dir="rtl">`.
     * [W3C: Structural markup and right-to-left text in HTML](http://www.w3.org/International/questions/qa-html-dir)
     */
    get isRTL() {
        return this.doc.dir === 'rtl';
    }
    /**
     * Get the query string parameter
     */
    getQueryParam(key) {
        return readQueryParam(this.win.location.href, key);
    }
    /**
     * Returns `true` if the app is in landscape mode.
     */
    isLandscape() {
        return !this.isPortrait();
    }
    /**
     * Returns `true` if the app is in portrait mode.
     */
    isPortrait() {
        var _a, _b;
        return (_b = (_a = this.win).matchMedia) === null || _b === void 0 ? void 0 : _b.call(_a, '(orientation: portrait)').matches;
    }
    testUserAgent(expression) {
        const nav = this.win.navigator;
        return !!((nav === null || nav === void 0 ? void 0 : nav.userAgent) && nav.userAgent.indexOf(expression) >= 0);
    }
    /**
     * Get the current url.
     */
    url() {
        return this.win.location.href;
    }
    /**
     * Gets the width of the platform's viewport using `window.innerWidth`.
     */
    width() {
        return this.win.innerWidth;
    }
    /**
     * Gets the height of the platform's viewport using `window.innerHeight`.
     */
    height() {
        return this.win.innerHeight;
    }
}
/** @nocollapse */ Platform.ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "12.2.10", ngImport: i0, type: Platform, deps: [{ token: DOCUMENT }, { token: i0.NgZone }], target: i0.ɵɵFactoryTarget.Injectable });
/** @nocollapse */ Platform.ɵprov = i0.ɵɵngDeclareInjectable({ minVersion: "12.0.0", version: "12.2.10", ngImport: i0, type: Platform, providedIn: 'root' });
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "12.2.10", ngImport: i0, type: Platform, decorators: [{
            type: Injectable,
            args: [{
                    providedIn: 'root',
                }]
        }], ctorParameters: function () { return [{ type: undefined, decorators: [{
                    type: Inject,
                    args: [DOCUMENT]
                }] }, { type: i0.NgZone }]; } });
const readQueryParam = (url, key) => {
    key = key.replace(/[[\]\\]/g, '\\$&');
    const regex = new RegExp('[\\?&]' + key + '=([^&#]*)');
    const results = regex.exec(url);
    return results ? decodeURIComponent(results[1].replace(/\+/g, ' ')) : null;
};
const proxyEvent = (emitter, el, eventName) => {
    if (el) {
        el.addEventListener(eventName, (ev) => {
            // ?? cordova might emit "null" events
            emitter.next(ev != null ? ev.detail : undefined);
        });
    }
};
//# sourceMappingURL=data:application/json;base64,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