/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { ChangeDetectionStrategy } from '../change_detection/constants';
import { compileComponent, compileDirective } from '../render3/jit/directive';
import { compilePipe } from '../render3/jit/pipe';
import { makeDecorator, makePropDecorator } from '../util/decorators';
/**
 * Type of the Directive metadata.
 *
 * @publicApi
 */
export const Directive = makeDecorator('Directive', (dir = {}) => dir, undefined, undefined, (type, meta) => compileDirective(type, meta));
/**
 * Component decorator and metadata.
 *
 * @Annotation
 * @publicApi
 */
export const Component = makeDecorator('Component', (c = {}) => ({ changeDetection: ChangeDetectionStrategy.Default, ...c }), Directive, undefined, (type, meta) => compileComponent(type, meta));
/**
 * @Annotation
 * @publicApi
 */
export const Pipe = makeDecorator('Pipe', (p) => ({ pure: true, ...p }), undefined, undefined, (type, meta) => compilePipe(type, meta));
/**
 * @Annotation
 * @publicApi
 */
export const Input = makePropDecorator('Input', (bindingPropertyName) => ({ bindingPropertyName }));
/**
 * @Annotation
 * @publicApi
 */
export const Output = makePropDecorator('Output', (bindingPropertyName) => ({ bindingPropertyName }));
/**
 * @Annotation
 * @publicApi
 */
export const HostBinding = makePropDecorator('HostBinding', (hostPropertyName) => ({ hostPropertyName }));
/**
 * Decorator that binds a DOM event to a host listener and supplies configuration metadata.
 * Angular invokes the supplied handler method when the host element emits the specified event,
 * and updates the bound element with the result.
 *
 * If the handler method returns false, applies `preventDefault` on the bound element.
 *
 * @usageNotes
 *
 * The following example declares a directive
 * that attaches a click listener to a button and counts clicks.
 *
 * ```ts
 * @Directive({selector: 'button[counting]'})
 * class CountClicks {
 *   numberOfClicks = 0;
 *
 *   @HostListener('click', ['$event.target'])
 *   onClick(btn) {
 *     console.log('button', btn, 'number of clicks:', this.numberOfClicks++);
 *   }
 * }
 *
 * @Component({
 *   selector: 'app',
 *   template: '<button counting>Increment</button>',
 * })
 * class App {}
 *
 * ```
 *
 * The following example registers another DOM event handler that listens for `Enter` key-press
 * events on the global `window`.
 * ``` ts
 * import { HostListener, Component } from "@angular/core";
 *
 * @Component({
 *   selector: 'app',
 *   template: `<h1>Hello, you have pressed enter {{counter}} number of times!</h1> Press enter key
 * to increment the counter.
 *   <button (click)="resetCounter()">Reset Counter</button>`
 * })
 * class AppComponent {
 *   counter = 0;
 *   @HostListener('window:keydown.enter', ['$event'])
 *   handleKeyDown(event: KeyboardEvent) {
 *     this.counter++;
 *   }
 *   resetCounter() {
 *     this.counter = 0;
 *   }
 * }
 * ```
 * The list of valid key names for `keydown` and `keyup` events
 * can be found here:
 * https://www.w3.org/TR/DOM-Level-3-Events-key/#named-key-attribute-values
 *
 * Note that keys can also be combined, e.g. `@HostListener('keydown.shift.a')`.
 *
 * The global target names that can be used to prefix an event name are
 * `document:`, `window:` and `body:`.
 *
 * @Annotation
 * @publicApi
 */
export const HostListener = makePropDecorator('HostListener', (eventName, args) => ({ eventName, args }));
//# sourceMappingURL=data:application/json;base64,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