/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { ERROR_DETAILS_PAGE_BASE_URL } from './error_details_base_url';
/**
 * Class that represents a runtime error.
 * Formats and outputs the error message in a consistent way.
 *
 * Example:
 * ```
 *  throw new RuntimeError(
 *    RuntimeErrorCode.INJECTOR_ALREADY_DESTROYED,
 *    ngDevMode && 'Injector has already been destroyed.');
 * ```
 *
 * Note: the `message` argument contains a descriptive error message as a string in development
 * mode (when the `ngDevMode` is defined). In production mode (after tree-shaking pass), the
 * `message` argument becomes `false`, thus we account for it in the typings and the runtime logic.
 */
export class RuntimeError extends Error {
    constructor(code, message) {
        super(formatRuntimeError(code, message));
        this.code = code;
    }
}
/**
 * Called to format a runtime error.
 * See additional info on the `message` argument type in the `RuntimeError` class description.
 */
export function formatRuntimeError(code, message) {
    // Error code might be a negative number, which is a special marker that instructs the logic to
    // generate a link to the error details page on angular.io.
    // We also prepend `0` to non-compile-time errors.
    const fullCode = `NG0${Math.abs(code)}`;
    let errorMessage = `${fullCode}${message ? ': ' + message.trim() : ''}`;
    if (ngDevMode && code < 0) {
        const addPeriodSeparator = !errorMessage.match(/[.,;!?]$/);
        const separator = addPeriodSeparator ? '.' : '';
        errorMessage =
            `${errorMessage}${separator} Find more at ${ERROR_DETAILS_PAGE_BASE_URL}/${fullCode}`;
    }
    return errorMessage;
}
//# sourceMappingURL=data:application/json;base64,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