"use strict";
/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.isJobHandler = exports.JobState = exports.JobOutboundMessageKind = exports.JobInboundMessageKind = void 0;
/**
 * Messages that can be sent TO a job. The job needs to listen to those.
 */
var JobInboundMessageKind;
(function (JobInboundMessageKind) {
    JobInboundMessageKind["Ping"] = "ip";
    JobInboundMessageKind["Stop"] = "is";
    // Channel specific messages.
    JobInboundMessageKind["Input"] = "in";
    // Input channel does not allow completion / error. Erroring this will just close the Subject
    // but not notify the job.
})(JobInboundMessageKind = exports.JobInboundMessageKind || (exports.JobInboundMessageKind = {}));
/**
 * Kind of messages that can be outputted from a job.
 */
var JobOutboundMessageKind;
(function (JobOutboundMessageKind) {
    // Lifecycle specific messages.
    JobOutboundMessageKind["OnReady"] = "c";
    JobOutboundMessageKind["Start"] = "s";
    JobOutboundMessageKind["End"] = "e";
    JobOutboundMessageKind["Pong"] = "p";
    // Feedback messages.
    JobOutboundMessageKind["Output"] = "o";
    // Channel specific messages.
    JobOutboundMessageKind["ChannelCreate"] = "cn";
    JobOutboundMessageKind["ChannelMessage"] = "cm";
    JobOutboundMessageKind["ChannelError"] = "ce";
    JobOutboundMessageKind["ChannelComplete"] = "cc";
})(JobOutboundMessageKind = exports.JobOutboundMessageKind || (exports.JobOutboundMessageKind = {}));
/**
 * The state of a job. These are changed as the job reports a new state through its messages.
 */
var JobState;
(function (JobState) {
    /**
     * The job was queued and is waiting to start.
     */
    JobState["Queued"] = "queued";
    /**
     * The job description was found, its dependencies (see "Synchronizing and Dependencies")
     * are done running, and the job's argument is validated and the job's code will be executed.
     */
    JobState["Ready"] = "ready";
    /**
     * The job has been started. The job implementation is expected to send this as soon as its
     * work is starting.
     */
    JobState["Started"] = "started";
    /**
     * The job has ended and is done running.
     */
    JobState["Ended"] = "ended";
    /**
     * An error occured and the job stopped because of internal state.
     */
    JobState["Errored"] = "errored";
})(JobState = exports.JobState || (exports.JobState = {}));
function isJobHandler(value) {
    const job = value;
    return (typeof job == 'function' && typeof job.jobDescription == 'object' && job.jobDescription !== null);
}
exports.isJobHandler = isJobHandler;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYXBpLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vLi4vLi4vLi4vLi4vLi4vLi4vLi4vcGFja2FnZXMvYW5ndWxhcl9kZXZraXQvYXJjaGl0ZWN0L3NyYy9qb2JzL2FwaS50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiO0FBQUE7Ozs7OztHQU1HOzs7QUFzREg7O0dBRUc7QUFDSCxJQUFZLHFCQVFYO0FBUkQsV0FBWSxxQkFBcUI7SUFDL0Isb0NBQVcsQ0FBQTtJQUNYLG9DQUFXLENBQUE7SUFFWCw2QkFBNkI7SUFDN0IscUNBQVksQ0FBQTtJQUNaLDZGQUE2RjtJQUM3RiwwQkFBMEI7QUFDNUIsQ0FBQyxFQVJXLHFCQUFxQixHQUFyQiw2QkFBcUIsS0FBckIsNkJBQXFCLFFBUWhDO0FBa0REOztHQUVHO0FBQ0gsSUFBWSxzQkFlWDtBQWZELFdBQVksc0JBQXNCO0lBQ2hDLCtCQUErQjtJQUMvQix1Q0FBYSxDQUFBO0lBQ2IscUNBQVcsQ0FBQTtJQUNYLG1DQUFTLENBQUE7SUFDVCxvQ0FBVSxDQUFBO0lBRVYscUJBQXFCO0lBQ3JCLHNDQUFZLENBQUE7SUFFWiw2QkFBNkI7SUFDN0IsOENBQW9CLENBQUE7SUFDcEIsK0NBQXFCLENBQUE7SUFDckIsNkNBQW1CLENBQUE7SUFDbkIsZ0RBQXNCLENBQUE7QUFDeEIsQ0FBQyxFQWZXLHNCQUFzQixHQUF0Qiw4QkFBc0IsS0FBdEIsOEJBQXNCLFFBZWpDO0FBaUlEOztHQUVHO0FBQ0gsSUFBWSxRQXVCWDtBQXZCRCxXQUFZLFFBQVE7SUFDbEI7O09BRUc7SUFDSCw2QkFBaUIsQ0FBQTtJQUNqQjs7O09BR0c7SUFDSCwyQkFBZSxDQUFBO0lBQ2Y7OztPQUdHO0lBQ0gsK0JBQW1CLENBQUE7SUFDbkI7O09BRUc7SUFDSCwyQkFBZSxDQUFBO0lBQ2Y7O09BRUc7SUFDSCwrQkFBbUIsQ0FBQTtBQUNyQixDQUFDLEVBdkJXLFFBQVEsR0FBUixnQkFBUSxLQUFSLGdCQUFRLFFBdUJuQjtBQWlKRCxTQUFnQixZQUFZLENBQzFCLEtBQWM7SUFFZCxNQUFNLEdBQUcsR0FBRyxLQUE0QixDQUFDO0lBRXpDLE9BQU8sQ0FDTCxPQUFPLEdBQUcsSUFBSSxVQUFVLElBQUksT0FBTyxHQUFHLENBQUMsY0FBYyxJQUFJLFFBQVEsSUFBSSxHQUFHLENBQUMsY0FBYyxLQUFLLElBQUksQ0FDakcsQ0FBQztBQUNKLENBQUM7QUFSRCxvQ0FRQyIsInNvdXJjZXNDb250ZW50IjpbIi8qKlxuICogQGxpY2Vuc2VcbiAqIENvcHlyaWdodCBHb29nbGUgTExDIEFsbCBSaWdodHMgUmVzZXJ2ZWQuXG4gKlxuICogVXNlIG9mIHRoaXMgc291cmNlIGNvZGUgaXMgZ292ZXJuZWQgYnkgYW4gTUlULXN0eWxlIGxpY2Vuc2UgdGhhdCBjYW4gYmVcbiAqIGZvdW5kIGluIHRoZSBMSUNFTlNFIGZpbGUgYXQgaHR0cHM6Ly9hbmd1bGFyLmlvL2xpY2Vuc2VcbiAqL1xuXG5pbXBvcnQgeyBKc29uT2JqZWN0LCBKc29uVmFsdWUsIHNjaGVtYSB9IGZyb20gJ0Bhbmd1bGFyLWRldmtpdC9jb3JlJztcbmltcG9ydCB7IE9ic2VydmFibGUsIE9ic2VydmVyIH0gZnJvbSAncnhqcyc7XG5pbXBvcnQgeyBEZWVwUmVhZG9ubHkgfSBmcm9tICcuL3R5cGVzJztcblxuLyoqXG4gKiBBIGpvYiBuYW1lIGlzIGp1c3QgYSBzdHJpbmcgKG5lZWRzIHRvIGJlIHNlcmlhbGl6YWJsZSkuXG4gKi9cbmV4cG9ydCB0eXBlIEpvYk5hbWUgPSBzdHJpbmc7XG5cbi8qKlxuICogVGhlIGpvYiBoYW5kbGVyIGZ1bmN0aW9uLCB3aGljaCBpcyBhIG1ldGhvZCB0aGF0J3MgZXhlY3V0ZWQgZm9yIHRoZSBqb2IuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgSm9iSGFuZGxlcjxcbiAgQXJnVCBleHRlbmRzIEpzb25WYWx1ZSxcbiAgSW5wdXRUIGV4dGVuZHMgSnNvblZhbHVlLFxuICBPdXRwdXRUIGV4dGVuZHMgSnNvblZhbHVlLFxuPiB7XG4gIChhcmd1bWVudDogQXJnVCwgY29udGV4dDogSm9iSGFuZGxlckNvbnRleHQ8QXJnVCwgSW5wdXRULCBPdXRwdXRUPik6IE9ic2VydmFibGU8XG4gICAgSm9iT3V0Ym91bmRNZXNzYWdlPE91dHB1dFQ+XG4gID47XG5cbiAgam9iRGVzY3JpcHRpb246IFBhcnRpYWw8Sm9iRGVzY3JpcHRpb24+O1xufVxuXG4vKipcbiAqIFRoZSBjb250ZXh0IGluIHdoaWNoIHRoZSBqb2IgaXMgcnVuLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIEpvYkhhbmRsZXJDb250ZXh0PFxuICBNaW5pbXVtQXJndW1lbnRWYWx1ZVQgZXh0ZW5kcyBKc29uVmFsdWUgPSBKc29uVmFsdWUsXG4gIE1pbmltdW1JbnB1dFZhbHVlVCBleHRlbmRzIEpzb25WYWx1ZSA9IEpzb25WYWx1ZSxcbiAgTWluaW11bU91dHB1dFZhbHVlVCBleHRlbmRzIEpzb25WYWx1ZSA9IEpzb25WYWx1ZSxcbj4ge1xuICByZWFkb25seSBkZXNjcmlwdGlvbjogSm9iRGVzY3JpcHRpb247XG4gIHJlYWRvbmx5IHNjaGVkdWxlcjogU2NoZWR1bGVyPEpzb25WYWx1ZSwgSnNvblZhbHVlLCBKc29uVmFsdWU+O1xuXG4gIC8vIEluIHRoaXMgY29udGV4dCwgSnNvblZhbHVlIGlzIGNvbXBhcmFibGUgdG8gYGFueWAuXG4gIHJlYWRvbmx5IGRlcGVuZGVuY2llczogSm9iPEpzb25WYWx1ZSwgSnNvblZhbHVlLCBKc29uVmFsdWU+W107XG5cbiAgcmVhZG9ubHkgaW5ib3VuZEJ1czogT2JzZXJ2YWJsZTxKb2JJbmJvdW5kTWVzc2FnZTxNaW5pbXVtSW5wdXRWYWx1ZVQ+Pjtcbn1cblxuLyoqXG4gKiBNZXRhZGF0YSBhc3NvY2lhdGVkIHdpdGggYSBqb2IuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgSm9iRGVzY3JpcHRpb24gZXh0ZW5kcyBKc29uT2JqZWN0IHtcbiAgcmVhZG9ubHkgbmFtZTogSm9iTmFtZTtcblxuICByZWFkb25seSBhcmd1bWVudDogRGVlcFJlYWRvbmx5PHNjaGVtYS5Kc29uU2NoZW1hPjtcbiAgcmVhZG9ubHkgaW5wdXQ6IERlZXBSZWFkb25seTxzY2hlbWEuSnNvblNjaGVtYT47XG4gIHJlYWRvbmx5IG91dHB1dDogRGVlcFJlYWRvbmx5PHNjaGVtYS5Kc29uU2NoZW1hPjtcbn1cblxuLyoqXG4gKiBNZXNzYWdlcyB0aGF0IGNhbiBiZSBzZW50IFRPIGEgam9iLiBUaGUgam9iIG5lZWRzIHRvIGxpc3RlbiB0byB0aG9zZS5cbiAqL1xuZXhwb3J0IGVudW0gSm9iSW5ib3VuZE1lc3NhZ2VLaW5kIHtcbiAgUGluZyA9ICdpcCcsXG4gIFN0b3AgPSAnaXMnLFxuXG4gIC8vIENoYW5uZWwgc3BlY2lmaWMgbWVzc2FnZXMuXG4gIElucHV0ID0gJ2luJyxcbiAgLy8gSW5wdXQgY2hhbm5lbCBkb2VzIG5vdCBhbGxvdyBjb21wbGV0aW9uIC8gZXJyb3IuIEVycm9yaW5nIHRoaXMgd2lsbCBqdXN0IGNsb3NlIHRoZSBTdWJqZWN0XG4gIC8vIGJ1dCBub3Qgbm90aWZ5IHRoZSBqb2IuXG59XG5cbi8qKiBCYXNlIGludGVyZmFjZSBmb3IgdGhlIGFsbCBqb2IgaW5ib3VuZCBtZXNzYWdlcy4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgSm9iSW5ib3VuZE1lc3NhZ2VCYXNlIGV4dGVuZHMgSnNvbk9iamVjdCB7XG4gIC8qKlxuICAgKiBUaGUga2luZCBvZiBtZXNzYWdlIHRoaXMgaXMuXG4gICAqL1xuICByZWFkb25seSBraW5kOiBKb2JJbmJvdW5kTWVzc2FnZUtpbmQ7XG59XG5cbi8qKlxuICogQSBwaW5nIHRvIHRoZSBqb2IuIFRoZSBqb2Igc2hvdWxkIHJlcGx5IHdpdGggYSBwb25nIGFzIHNvb24gYXMgcG9zc2libGUuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgSm9iSW5ib3VuZE1lc3NhZ2VQaW5nIGV4dGVuZHMgSm9iSW5ib3VuZE1lc3NhZ2VCYXNlIHtcbiAgcmVhZG9ubHkga2luZDogSm9iSW5ib3VuZE1lc3NhZ2VLaW5kLlBpbmc7XG5cbiAgLyoqXG4gICAqIEFuIElEIHRoYXQgc2hvdWxkIGJlIHJldHVybmVkIGluIHRoZSBjb3JyZXNwb25kaW5nIFBvbmcuXG4gICAqL1xuICByZWFkb25seSBpZDogbnVtYmVyO1xufVxuXG4vKipcbiAqIFN0b3AgdGhlIGpvYi4gVGhpcyBpcyBoYW5kbGVkIGJ5IHRoZSBqb2IgaXRzZWxmIGFuZCBqb2JzIG1pZ2h0IG5vdCBoYW5kbGUgaXQuIEl0IHdpbGwgYWxzb1xuICogdW5zdWJzY3JpYmUgZnJvbSB0aGUgT2JzZXJ2YWJsZTw+LlxuICogVGhpcyBpcyBlcXVpdmFsZW50IHRvIFNJR1RFUk0uXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgSm9iSW5ib3VuZE1lc3NhZ2VTdG9wIGV4dGVuZHMgSm9iSW5ib3VuZE1lc3NhZ2VCYXNlIHtcbiAgcmVhZG9ubHkga2luZDogSm9iSW5ib3VuZE1lc3NhZ2VLaW5kLlN0b3A7XG59XG5cbi8qKlxuICogQSBKb2Igd2FudHMgdG8gc2VuZCBhIG1lc3NhZ2UgdG8gYSBjaGFubmVsLiBUaGlzIGNhbiBiZSBtYXJzaGFsZWQsIGFuZCB0aGUgSm9iIG9iamVjdFxuICogaGFzIGhlbHBlcnMgdG8gdHJhbnNmb3JtIHRoaXMgaW50byBhbiBvYnNlcnZhYmxlLiBUaGUgY29udGV4dCBhbHNvIGNhbiBjcmVhdGUgUnhKUyBzdWJqZWN0cyB0aGF0XG4gKiBtYXJzaGFsbCBtZXNzYWdlcyB0aHJvdWdoIGEgY2hhbm5lbC5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBKb2JJbmJvdW5kTWVzc2FnZUlucHV0PElucHV0VCBleHRlbmRzIEpzb25WYWx1ZT4gZXh0ZW5kcyBKb2JJbmJvdW5kTWVzc2FnZUJhc2Uge1xuICByZWFkb25seSBraW5kOiBKb2JJbmJvdW5kTWVzc2FnZUtpbmQuSW5wdXQ7XG5cbiAgLyoqXG4gICAqIFRoZSBpbnB1dCBiZWluZyBzZW50IHRvIHRoZSBqb2IuXG4gICAqL1xuICByZWFkb25seSB2YWx1ZTogSW5wdXRUO1xufVxuXG5leHBvcnQgdHlwZSBKb2JJbmJvdW5kTWVzc2FnZTxJbnB1dFQgZXh0ZW5kcyBKc29uVmFsdWU+ID1cbiAgfCBKb2JJbmJvdW5kTWVzc2FnZVBpbmdcbiAgfCBKb2JJbmJvdW5kTWVzc2FnZVN0b3BcbiAgfCBKb2JJbmJvdW5kTWVzc2FnZUlucHV0PElucHV0VD47XG5cbi8qKlxuICogS2luZCBvZiBtZXNzYWdlcyB0aGF0IGNhbiBiZSBvdXRwdXR0ZWQgZnJvbSBhIGpvYi5cbiAqL1xuZXhwb3J0IGVudW0gSm9iT3V0Ym91bmRNZXNzYWdlS2luZCB7XG4gIC8vIExpZmVjeWNsZSBzcGVjaWZpYyBtZXNzYWdlcy5cbiAgT25SZWFkeSA9ICdjJyxcbiAgU3RhcnQgPSAncycsXG4gIEVuZCA9ICdlJyxcbiAgUG9uZyA9ICdwJyxcblxuICAvLyBGZWVkYmFjayBtZXNzYWdlcy5cbiAgT3V0cHV0ID0gJ28nLFxuXG4gIC8vIENoYW5uZWwgc3BlY2lmaWMgbWVzc2FnZXMuXG4gIENoYW5uZWxDcmVhdGUgPSAnY24nLFxuICBDaGFubmVsTWVzc2FnZSA9ICdjbScsXG4gIENoYW5uZWxFcnJvciA9ICdjZScsXG4gIENoYW5uZWxDb21wbGV0ZSA9ICdjYycsXG59XG5cbi8qKiBCYXNlIGludGVyZmFjZSBmb3IgdGhlIGFsbCBqb2IgbWVzc2FnZXMuICovXG5leHBvcnQgaW50ZXJmYWNlIEpvYk91dGJvdW5kTWVzc2FnZUJhc2Uge1xuICAvKipcbiAgICogVGhlIGpvYiBkZXNjcmlwdGlvbi5cbiAgICovXG4gIHJlYWRvbmx5IGRlc2NyaXB0aW9uOiBKb2JEZXNjcmlwdGlvbjtcblxuICAvKipcbiAgICogVGhlIGtpbmQgb2YgbWVzc2FnZSB0aGlzIGlzLlxuICAgKi9cbiAgcmVhZG9ubHkga2luZDogSm9iT3V0Ym91bmRNZXNzYWdlS2luZDtcbn1cblxuLyoqXG4gKiBUaGUgam9iIGhhcyBiZWVuIGNyZWF0ZWQgYW5kIHdpbGwgdmFsaWRhdGUgaXRzIGlucHV0LlxuICovXG5leHBvcnQgaW50ZXJmYWNlIEpvYk91dGJvdW5kTWVzc2FnZU9uUmVhZHkgZXh0ZW5kcyBKb2JPdXRib3VuZE1lc3NhZ2VCYXNlIHtcbiAgcmVhZG9ubHkga2luZDogSm9iT3V0Ym91bmRNZXNzYWdlS2luZC5PblJlYWR5O1xufVxuXG4vKipcbiAqIFRoZSBqb2Igc3RhcnRlZC4gVGhpcyBpcyBkb25lIGJ5IHRoZSBqb2IgaXRzZWxmLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIEpvYk91dGJvdW5kTWVzc2FnZVN0YXJ0IGV4dGVuZHMgSm9iT3V0Ym91bmRNZXNzYWdlQmFzZSB7XG4gIHJlYWRvbmx5IGtpbmQ6IEpvYk91dGJvdW5kTWVzc2FnZUtpbmQuU3RhcnQ7XG59XG5cbi8qKlxuICogQW4gb3V0cHV0IHZhbHVlIGlzIGF2YWlsYWJsZS5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBKb2JPdXRib3VuZE1lc3NhZ2VPdXRwdXQ8T3V0cHV0VCBleHRlbmRzIEpzb25WYWx1ZT5cbiAgZXh0ZW5kcyBKb2JPdXRib3VuZE1lc3NhZ2VCYXNlIHtcbiAgcmVhZG9ubHkga2luZDogSm9iT3V0Ym91bmRNZXNzYWdlS2luZC5PdXRwdXQ7XG5cbiAgLyoqXG4gICAqIFRoZSBtZXNzYWdlIGJlaW5nIG91dHB1dHRlZCBmcm9tIHRoZSBqb2IuXG4gICAqL1xuICByZWFkb25seSB2YWx1ZTogT3V0cHV0VDtcbn1cblxuLyoqXG4gKiBCYXNlIGludGVyZmFjZSBmb3IgYWxsIGpvYiBtZXNzYWdlIHJlbGF0ZWQgdG8gY2hhbm5lbHMuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgSm9iT3V0Ym91bmRNZXNzYWdlQ2hhbm5lbEJhc2UgZXh0ZW5kcyBKb2JPdXRib3VuZE1lc3NhZ2VCYXNlIHtcbiAgLyoqXG4gICAqIFRoZSBuYW1lIG9mIHRoZSBjaGFubmVsLlxuICAgKi9cbiAgcmVhZG9ubHkgbmFtZTogc3RyaW5nO1xufVxuXG4vKipcbiAqIEEgam9iIHdhbnRzIHRvIHNlbmQgYSBtZXNzYWdlIHRvIGEgY2hhbm5lbC4gVGhpcyBjYW4gYmUgbWFyc2hhbGVkLCBhbmQgdGhlIEpvYiBvYmplY3RcbiAqIGhhcyBoZWxwZXJzIHRvIHRyYW5zZm9ybSB0aGlzIGludG8gYW4gb2JzZXJ2YWJsZS4gVGhlIGNvbnRleHQgYWxzbyBjYW4gY3JlYXRlIFJ4SlMgc3ViamVjdHMgdGhhdFxuICogbWFyc2hhbGwgbWVzc2FnZXMgdGhyb3VnaCBhIGNoYW5uZWwuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgSm9iT3V0Ym91bmRNZXNzYWdlQ2hhbm5lbE1lc3NhZ2UgZXh0ZW5kcyBKb2JPdXRib3VuZE1lc3NhZ2VDaGFubmVsQmFzZSB7XG4gIHJlYWRvbmx5IGtpbmQ6IEpvYk91dGJvdW5kTWVzc2FnZUtpbmQuQ2hhbm5lbE1lc3NhZ2U7XG5cbiAgLyoqXG4gICAqIFRoZSBtZXNzYWdlIGJlaW5nIHNlbnQgdG8gdGhlIGNoYW5uZWwuXG4gICAqL1xuICByZWFkb25seSBtZXNzYWdlOiBKc29uVmFsdWU7XG59XG5cbi8qKlxuICogQSBqb2Igd2FudHMgdG8gc2VuZCBhbiBlcnJvciB0byBvbmUgb2YgaXRzIGNoYW5uZWwuIFRoaXMgaXMgdGhlIGVxdWl2YWxlbnQgb2YgdGhyb3dpbmcgdGhyb3VnaFxuICogYW4gT2JzZXJ2YWJsZS4gVGhlIHNpZGUgY2hhbm5lbCB3aWxsIG5vdCByZWNlaXZlIGFueSBtb3JlIG1lc3NhZ2VzIGFmdGVyIHRoaXMsIGFuZCB3aWxsIG5vdFxuICogY29tcGxldGUuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgSm9iT3V0Ym91bmRNZXNzYWdlQ2hhbm5lbEVycm9yIGV4dGVuZHMgSm9iT3V0Ym91bmRNZXNzYWdlQ2hhbm5lbEJhc2Uge1xuICByZWFkb25seSBraW5kOiBKb2JPdXRib3VuZE1lc3NhZ2VLaW5kLkNoYW5uZWxFcnJvcjtcblxuICAvKipcbiAgICogVGhlIGVycm9yIG1lc3NhZ2UgYmVpbmcgc2VudCB0byB0aGUgY2hhbm5lbC5cbiAgICovXG4gIHJlYWRvbmx5IGVycm9yOiBKc29uVmFsdWU7XG59XG5cbi8qKlxuICogQSBqb2Igd2FudHMgdG8gY3JlYXRlIGEgbmV3IGNoYW5uZWwuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgSm9iT3V0Ym91bmRNZXNzYWdlQ2hhbm5lbENyZWF0ZSBleHRlbmRzIEpvYk91dGJvdW5kTWVzc2FnZUNoYW5uZWxCYXNlIHtcbiAgcmVhZG9ubHkga2luZDogSm9iT3V0Ym91bmRNZXNzYWdlS2luZC5DaGFubmVsQ3JlYXRlO1xufVxuXG4vKipcbiAqIEEgam9iIHdhbnRzIHRvIGNsb3NlIHRoZSBjaGFubmVsLCBhcyBjb21wbGV0ZWQuIFRoaXMgaXMgZG9uZSBhdXRvbWF0aWNhbGx5IHdoZW4gdGhlIGpvYiBlbmRzLFxuICogb3IgY2FuIGJlIGRvbmUgZnJvbSB0aGUgam9iIHRvIGNsb3NlIGl0LiBBIGNsb3NlZCBjaGFubmVsIG1pZ2h0IGJlIHJlb3BlbmVkLCBidXQgdGhlIHVzZXJcbiAqIG5lZWQgdG8gcmVjYWxsIGdldENoYW5uZWwoKS5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBKb2JPdXRib3VuZE1lc3NhZ2VDaGFubmVsQ29tcGxldGUgZXh0ZW5kcyBKb2JPdXRib3VuZE1lc3NhZ2VDaGFubmVsQmFzZSB7XG4gIHJlYWRvbmx5IGtpbmQ6IEpvYk91dGJvdW5kTWVzc2FnZUtpbmQuQ2hhbm5lbENvbXBsZXRlO1xufVxuXG4vKipcbiAqIE9uRW5kIG9mIHRoZSBqb2IgcnVuLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIEpvYk91dGJvdW5kTWVzc2FnZUVuZCBleHRlbmRzIEpvYk91dGJvdW5kTWVzc2FnZUJhc2Uge1xuICByZWFkb25seSBraW5kOiBKb2JPdXRib3VuZE1lc3NhZ2VLaW5kLkVuZDtcbn1cblxuLyoqXG4gKiBBIHBvbmcgcmVzcG9uc2UgZnJvbSBhIHBpbmcgaW5wdXQuIFRoZSBpZCBpcyB0aGUgc2FtZSBhcyB0aGUgb25lIHBhc3NlZCBpbi5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBKb2JPdXRib3VuZE1lc3NhZ2VQb25nIGV4dGVuZHMgSm9iT3V0Ym91bmRNZXNzYWdlQmFzZSB7XG4gIHJlYWRvbmx5IGtpbmQ6IEpvYk91dGJvdW5kTWVzc2FnZUtpbmQuUG9uZztcblxuICAvKipcbiAgICogVGhlIElEIHRoYXQgd2FzIHBhc3NlZCBpbiB0aGUgYFBpbmdgIG1lc3NhZ2VzLlxuICAgKi9cbiAgcmVhZG9ubHkgaWQ6IG51bWJlcjtcbn1cblxuLyoqXG4gKiBHZW5lcmljIG1lc3NhZ2UgdHlwZS5cbiAqL1xuZXhwb3J0IHR5cGUgSm9iT3V0Ym91bmRNZXNzYWdlPE91dHB1dFQgZXh0ZW5kcyBKc29uVmFsdWU+ID1cbiAgfCBKb2JPdXRib3VuZE1lc3NhZ2VPblJlYWR5XG4gIHwgSm9iT3V0Ym91bmRNZXNzYWdlU3RhcnRcbiAgfCBKb2JPdXRib3VuZE1lc3NhZ2VPdXRwdXQ8T3V0cHV0VD5cbiAgfCBKb2JPdXRib3VuZE1lc3NhZ2VDaGFubmVsQ3JlYXRlXG4gIHwgSm9iT3V0Ym91bmRNZXNzYWdlQ2hhbm5lbE1lc3NhZ2VcbiAgfCBKb2JPdXRib3VuZE1lc3NhZ2VDaGFubmVsRXJyb3JcbiAgfCBKb2JPdXRib3VuZE1lc3NhZ2VDaGFubmVsQ29tcGxldGVcbiAgfCBKb2JPdXRib3VuZE1lc3NhZ2VFbmRcbiAgfCBKb2JPdXRib3VuZE1lc3NhZ2VQb25nO1xuXG4vKipcbiAqIFRoZSBzdGF0ZSBvZiBhIGpvYi4gVGhlc2UgYXJlIGNoYW5nZWQgYXMgdGhlIGpvYiByZXBvcnRzIGEgbmV3IHN0YXRlIHRocm91Z2ggaXRzIG1lc3NhZ2VzLlxuICovXG5leHBvcnQgZW51bSBKb2JTdGF0ZSB7XG4gIC8qKlxuICAgKiBUaGUgam9iIHdhcyBxdWV1ZWQgYW5kIGlzIHdhaXRpbmcgdG8gc3RhcnQuXG4gICAqL1xuICBRdWV1ZWQgPSAncXVldWVkJyxcbiAgLyoqXG4gICAqIFRoZSBqb2IgZGVzY3JpcHRpb24gd2FzIGZvdW5kLCBpdHMgZGVwZW5kZW5jaWVzIChzZWUgXCJTeW5jaHJvbml6aW5nIGFuZCBEZXBlbmRlbmNpZXNcIilcbiAgICogYXJlIGRvbmUgcnVubmluZywgYW5kIHRoZSBqb2IncyBhcmd1bWVudCBpcyB2YWxpZGF0ZWQgYW5kIHRoZSBqb2IncyBjb2RlIHdpbGwgYmUgZXhlY3V0ZWQuXG4gICAqL1xuICBSZWFkeSA9ICdyZWFkeScsXG4gIC8qKlxuICAgKiBUaGUgam9iIGhhcyBiZWVuIHN0YXJ0ZWQuIFRoZSBqb2IgaW1wbGVtZW50YXRpb24gaXMgZXhwZWN0ZWQgdG8gc2VuZCB0aGlzIGFzIHNvb24gYXMgaXRzXG4gICAqIHdvcmsgaXMgc3RhcnRpbmcuXG4gICAqL1xuICBTdGFydGVkID0gJ3N0YXJ0ZWQnLFxuICAvKipcbiAgICogVGhlIGpvYiBoYXMgZW5kZWQgYW5kIGlzIGRvbmUgcnVubmluZy5cbiAgICovXG4gIEVuZGVkID0gJ2VuZGVkJyxcbiAgLyoqXG4gICAqIEFuIGVycm9yIG9jY3VyZWQgYW5kIHRoZSBqb2Igc3RvcHBlZCBiZWNhdXNlIG9mIGludGVybmFsIHN0YXRlLlxuICAgKi9cbiAgRXJyb3JlZCA9ICdlcnJvcmVkJyxcbn1cblxuLyoqXG4gKiBBIEpvYiBpbnN0YW5jZSwgcmV0dXJuZWQgZnJvbSBzY2hlZHVsaW5nIGEgam9iLiBBIEpvYiBpbnN0YW5jZSBpcyBfbm90XyBzZXJpYWxpemFibGUuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgSm9iPFxuICBBcmd1bWVudFQgZXh0ZW5kcyBKc29uVmFsdWUgPSBKc29uVmFsdWUsXG4gIElucHV0VCBleHRlbmRzIEpzb25WYWx1ZSA9IEpzb25WYWx1ZSxcbiAgT3V0cHV0VCBleHRlbmRzIEpzb25WYWx1ZSA9IEpzb25WYWx1ZSxcbj4ge1xuICAvKipcbiAgICogRGVzY3JpcHRpb24gb2YgdGhlIGpvYi4gUmVzb2x2aW5nIHRoZSBqb2IncyBkZXNjcmlwdGlvbiBjYW4gYmUgZG9uZSBhc3luY2hyb25vdXNseSwgc28gdGhpc1xuICAgKiBpcyBhbiBvYnNlcnZhYmxlIHRoYXQgd2lsbCByZXNvbHZlIHdoZW4gaXQncyByZWFkeS5cbiAgICovXG4gIHJlYWRvbmx5IGRlc2NyaXB0aW9uOiBPYnNlcnZhYmxlPEpvYkRlc2NyaXB0aW9uPjtcblxuICAvKipcbiAgICogQXJndW1lbnQgc2VudCB3aGVuIHNjaGVkdWxpbmcgdGhlIGpvYi4gVGhpcyBpcyBhIGNvcHkgb2YgdGhlIGFyZ3VtZW50LlxuICAgKi9cbiAgcmVhZG9ubHkgYXJndW1lbnQ6IEFyZ3VtZW50VDtcblxuICAvKipcbiAgICogVGhlIGlucHV0IHRvIHRoZSBqb2IuIFRoaXMgZ29lcyB0aHJvdWdoIHRoZSBpbnB1dCBjaGFubmVsIGFzIG1lc3NhZ2VzLlxuICAgKi9cbiAgcmVhZG9ubHkgaW5wdXQ6IE9ic2VydmVyPElucHV0VD47XG5cbiAgLyoqXG4gICAqIE91dHB1dHMgb2YgdGhpcyBqb2IuXG4gICAqL1xuICByZWFkb25seSBvdXRwdXQ6IE9ic2VydmFibGU8T3V0cHV0VD47XG5cbiAgLyoqXG4gICAqIFRoZSBjdXJyZW50IHN0YXRlIG9mIHRoZSBqb2IuXG4gICAqL1xuICByZWFkb25seSBzdGF0ZTogSm9iU3RhdGU7XG5cbiAgLyoqXG4gICAqIEdldCBhIGNoYW5uZWwgdGhhdCB2YWxpZGF0ZXMgYWdhaW5zdCB0aGUgc2NoZW1hLiBNZXNzYWdlcyB3aWxsIGJlIGZpbHRlcmVkIGJ5IHRoZSBzY2hlbWEuXG4gICAqIEBwYXJhbSBuYW1lIFRoZSBuYW1lIG9mIHRoZSBjaGFubmVsLlxuICAgKiBAcGFyYW0gc2NoZW1hIEEgc2NoZW1hIHRvIHVzZSB0byB2YWxpZGF0ZSBtZXNzYWdlcy5cbiAgICovXG4gIGdldENoYW5uZWw8VCBleHRlbmRzIEpzb25WYWx1ZT4obmFtZTogc3RyaW5nLCBzY2hlbWE/OiBzY2hlbWEuSnNvblNjaGVtYSk6IE9ic2VydmFibGU8VD47XG5cbiAgLyoqXG4gICAqIFBpbmdzIHRoZSBqb2IgYW5kIHdhaXQgZm9yIHRoZSByZXN1bHRpbmcgUG9uZyBiZWZvcmUgY29tcGxldGluZy5cbiAgICovXG4gIHBpbmcoKTogT2JzZXJ2YWJsZTxuZXZlcj47XG5cbiAgLyoqXG4gICAqIFN0b3BzIHRoZSBqb2IgZnJvbSBydW5uaW5nLiBUaGlzIGlzIGRpZmZlcmVudCB0aGFuIHVuc3Vic2NyaWJpbmcgZnJvbSB0aGUgb3V0cHV0IGFzIGluIGl0XG4gICAqIHNlbmRzIHRoZSBKb2JJbmJvdW5kTWVzc2FnZUtpbmQuU3RvcCByYXcgaW5wdXQgdG8gdGhlIGpvYi5cbiAgICovXG4gIHN0b3AoKTogdm9pZDtcblxuICAvKipcbiAgICogVGhlIEpvYkluYm91bmRNZXNzYWdlIG1lc3NhZ2VzIFRPIHRoZSBqb2IuXG4gICAqL1xuICByZWFkb25seSBpbmJvdW5kQnVzOiBPYnNlcnZlcjxKb2JJbmJvdW5kTWVzc2FnZTxJbnB1dFQ+PjtcblxuICAvKipcbiAgICogVGhlIEpvYk91dGJvdW5kTWVzc2FnZSBGUk9NIHRoZSBqb2IuXG4gICAqL1xuICByZWFkb25seSBvdXRib3VuZEJ1czogT2JzZXJ2YWJsZTxKb2JPdXRib3VuZE1lc3NhZ2U8T3V0cHV0VD4+O1xufVxuXG4vKipcbiAqIE9wdGlvbnMgZm9yIHNjaGVkdWxpbmcgam9icy5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBTY2hlZHVsZUpvYk9wdGlvbnMge1xuICAvKipcbiAgICogSm9icyB0aGF0IG5lZWQgdG8gZmluaXNoIGJlZm9yZSBzY2hlZHVsaW5nIHRoaXMgam9iLiBUaGVzZSBkZXBlbmRlbmNpZXMgd2lsbCBiZSBwYXNzZWRcbiAgICogdG8gdGhlIGpvYiBpdHNlbGYgaW4gaXRzIGNvbnRleHQuXG4gICAqL1xuICBkZXBlbmRlbmNpZXM/OiBKb2IgfCBKb2JbXTtcbn1cblxuZXhwb3J0IGludGVyZmFjZSBSZWdpc3RyeTxcbiAgTWluaW11bUFyZ3VtZW50VmFsdWVUIGV4dGVuZHMgSnNvblZhbHVlID0gSnNvblZhbHVlLFxuICBNaW5pbXVtSW5wdXRWYWx1ZVQgZXh0ZW5kcyBKc29uVmFsdWUgPSBKc29uVmFsdWUsXG4gIE1pbmltdW1PdXRwdXRWYWx1ZVQgZXh0ZW5kcyBKc29uVmFsdWUgPSBKc29uVmFsdWUsXG4+IHtcbiAgLyoqXG4gICAqIEdldCBhIGpvYiBoYW5kbGVyLlxuICAgKiBAcGFyYW0gbmFtZSBUaGUgbmFtZSBvZiB0aGUgam9iIHRvIGdldCBhIGhhbmRsZXIgZnJvbS5cbiAgICovXG4gIGdldDxBIGV4dGVuZHMgTWluaW11bUFyZ3VtZW50VmFsdWVULCBJIGV4dGVuZHMgTWluaW11bUlucHV0VmFsdWVULCBPIGV4dGVuZHMgTWluaW11bU91dHB1dFZhbHVlVD4oXG4gICAgbmFtZTogSm9iTmFtZSxcbiAgKTogT2JzZXJ2YWJsZTxKb2JIYW5kbGVyPEEsIEksIE8+IHwgbnVsbD47XG59XG5cbi8qKlxuICogQW4gaW50ZXJmYWNlIHRoYXQgY2FuIHNjaGVkdWxlIGpvYnMuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgU2NoZWR1bGVyPFxuICBNaW5pbXVtQXJndW1lbnRWYWx1ZVQgZXh0ZW5kcyBKc29uVmFsdWUgPSBKc29uVmFsdWUsXG4gIE1pbmltdW1JbnB1dFZhbHVlVCBleHRlbmRzIEpzb25WYWx1ZSA9IEpzb25WYWx1ZSxcbiAgTWluaW11bU91dHB1dFZhbHVlVCBleHRlbmRzIEpzb25WYWx1ZSA9IEpzb25WYWx1ZSxcbj4ge1xuICAvKipcbiAgICogR2V0IGEgam9iIGRlc2NyaXB0aW9uIGZvciBhIG5hbWVkIGpvYi5cbiAgICpcbiAgICogQHBhcmFtIG5hbWUgVGhlIG5hbWUgb2YgdGhlIGpvYi5cbiAgICogQHJldHVybnMgQSBkZXNjcmlwdGlvbiwgb3IgbnVsbCBpZiBubyBkZXNjcmlwdGlvbiBpcyBhdmFpbGFibGUgZm9yIHRoaXMgam9iLlxuICAgKi9cbiAgZ2V0RGVzY3JpcHRpb24obmFtZTogSm9iTmFtZSk6IE9ic2VydmFibGU8Sm9iRGVzY3JpcHRpb24gfCBudWxsPjtcblxuICAvKipcbiAgICogUmV0dXJucyB0cnVlIGlmIHRoZSBqb2IgbmFtZSBoYXMgYmVlbiByZWdpc3RlcmVkLlxuICAgKiBAcGFyYW0gbmFtZSBUaGUgbmFtZSBvZiB0aGUgam9iLlxuICAgKiBAcmV0dXJucyBUcnVlIGlmIHRoZSBqb2IgZXhpc3RzLCBmYWxzZSBvdGhlcndpc2UuXG4gICAqL1xuICBoYXMobmFtZTogSm9iTmFtZSk6IE9ic2VydmFibGU8Ym9vbGVhbj47XG5cbiAgLyoqXG4gICAqIFBhdXNlIHRoZSBzY2hlZHVsZXIsIHRlbXBvcmFyeSBxdWV1ZWluZyBfbmV3XyBqb2JzLiBSZXR1cm5zIGEgcmVzdW1lIGZ1bmN0aW9uIHRoYXQgc2hvdWxkIGJlXG4gICAqIHVzZWQgdG8gcmVzdW1lIGV4ZWN1dGlvbi4gSWYgbXVsdGlwbGUgYHBhdXNlKClgIHdlcmUgY2FsbGVkLCBhbGwgdGhlaXIgcmVzdW1lIGZ1bmN0aW9ucyBtdXN0XG4gICAqIGJlIGNhbGxlZCBiZWZvcmUgdGhlIFNjaGVkdWxlciBhY3R1YWxseSBzdGFydHMgbmV3IGpvYnMuIEFkZGl0aW9uYWwgY2FsbHMgdG8gdGhlIHNhbWUgcmVzdW1lXG4gICAqIGZ1bmN0aW9uIHdpbGwgaGF2ZSBubyBlZmZlY3QuXG4gICAqXG4gICAqIEpvYnMgYWxyZWFkeSBydW5uaW5nIGFyZSBOT1QgcGF1c2VkLiBUaGlzIGlzIHBhdXNpbmcgdGhlIHNjaGVkdWxlciBvbmx5LlxuICAgKlxuICAgKiBAcmV0dXJucyBBIGZ1bmN0aW9uIHRoYXQgY2FuIGJlIHJ1biB0byByZXN1bWUgdGhlIHNjaGVkdWxlci4gSWYgbXVsdGlwbGUgYHBhdXNlKClgIGNhbGxzXG4gICAqICAgICAgICAgIHdlcmUgbWFkZSwgYWxsIHRoZWlyIHJldHVybiBmdW5jdGlvbiBtdXN0IGJlIGNhbGxlZCAoaW4gYW55IG9yZGVyKSBiZWZvcmUgdGhlXG4gICAqICAgICAgICAgIHNjaGVkdWxlciBjYW4gcmVzdW1lLlxuICAgKi9cbiAgcGF1c2UoKTogKCkgPT4gdm9pZDtcblxuICAvKipcbiAgICogU2NoZWR1bGUgYSBqb2IgdG8gYmUgcnVuLCB1c2luZyBpdHMgbmFtZS5cbiAgICogQHBhcmFtIG5hbWUgVGhlIG5hbWUgb2Ygam9iIHRvIGJlIHJ1bi5cbiAgICogQHBhcmFtIGFyZ3VtZW50IFRoZSBhcmd1bWVudCB0byBzZW5kIHRvIHRoZSBqb2Igd2hlbiBzdGFydGluZyBpdC5cbiAgICogQHBhcmFtIG9wdGlvbnMgU2NoZWR1bGluZyBvcHRpb25zLlxuICAgKiBAcmV0dXJucyBUaGUgam9iIGJlaW5nIHJ1bi5cbiAgICovXG4gIHNjaGVkdWxlPFxuICAgIEEgZXh0ZW5kcyBNaW5pbXVtQXJndW1lbnRWYWx1ZVQsXG4gICAgSSBleHRlbmRzIE1pbmltdW1JbnB1dFZhbHVlVCxcbiAgICBPIGV4dGVuZHMgTWluaW11bU91dHB1dFZhbHVlVCxcbiAgPihcbiAgICBuYW1lOiBKb2JOYW1lLFxuICAgIGFyZ3VtZW50OiBBLFxuICAgIG9wdGlvbnM/OiBTY2hlZHVsZUpvYk9wdGlvbnMsXG4gICk6IEpvYjxBLCBJLCBPPjtcbn1cblxuZXhwb3J0IGZ1bmN0aW9uIGlzSm9iSGFuZGxlcjxBIGV4dGVuZHMgSnNvblZhbHVlLCBJIGV4dGVuZHMgSnNvblZhbHVlLCBPIGV4dGVuZHMgSnNvblZhbHVlPihcbiAgdmFsdWU6IHVua25vd24sXG4pOiB2YWx1ZSBpcyBKb2JIYW5kbGVyPEEsIEksIE8+IHtcbiAgY29uc3Qgam9iID0gdmFsdWUgYXMgSm9iSGFuZGxlcjxBLCBJLCBPPjtcblxuICByZXR1cm4gKFxuICAgIHR5cGVvZiBqb2IgPT0gJ2Z1bmN0aW9uJyAmJiB0eXBlb2Ygam9iLmpvYkRlc2NyaXB0aW9uID09ICdvYmplY3QnICYmIGpvYi5qb2JEZXNjcmlwdGlvbiAhPT0gbnVsbFxuICApO1xufVxuIl19