// SPDX-FileCopyrightText: Amolith <amolith@secluded.site>
//
// SPDX-License-Identifier: GPL-3.0-or-later

package com.adresilo.android

import android.content.Context
import android.content.Intent
import android.os.Bundle
import android.util.Log
import androidx.activity.ComponentActivity
import androidx.activity.compose.setContent
import androidx.annotation.StringRes
import androidx.compose.foundation.Image
import androidx.compose.foundation.clickable
import androidx.compose.foundation.isSystemInDarkTheme
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.consumeWindowInsets
import androidx.compose.foundation.layout.fillMaxHeight
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.imePadding
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.size
import androidx.compose.foundation.layout.width
import androidx.compose.foundation.lazy.LazyColumn
import androidx.compose.foundation.lazy.items
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.automirrored.filled.ArrowBack
import androidx.compose.material.icons.automirrored.rounded.OpenInNew
import androidx.compose.material.icons.filled.Settings
import androidx.compose.material.icons.rounded.Cancel
import androidx.compose.material.icons.rounded.Code
import androidx.compose.material.icons.rounded.ContentCopy
import androidx.compose.material.icons.rounded.Keyboard
import androidx.compose.material.icons.rounded.Search
import androidx.compose.material.icons.rounded.Share
import androidx.compose.material3.AlertDialog
import androidx.compose.material3.Button
import androidx.compose.material3.ButtonDefaults
import androidx.compose.material3.Card
import androidx.compose.material3.CircularProgressIndicator
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.FloatingActionButton
import androidx.compose.material3.Icon
import androidx.compose.material3.IconButton
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Scaffold
import androidx.compose.material3.SearchBar
import androidx.compose.material3.Text
import androidx.compose.material3.TextButton
import androidx.compose.material3.TextField
import androidx.compose.material3.TopAppBar
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.rememberCoroutineScope
import androidx.compose.runtime.saveable.rememberSaveable
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.focus.FocusRequester
import androidx.compose.ui.focus.focusRequester
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.res.painterResource
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.style.TextOverflow
import androidx.compose.ui.tooling.preview.Preview
import androidx.compose.ui.unit.dp
import androidx.core.content.ContextCompat.startActivity
import androidx.core.net.toUri
import androidx.datastore.core.DataStore
import androidx.datastore.preferences.core.Preferences
import androidx.datastore.preferences.core.edit
import androidx.datastore.preferences.core.stringPreferencesKey
import androidx.datastore.preferences.core.stringSetPreferencesKey
import androidx.datastore.preferences.preferencesDataStore
import androidx.navigation.NavHostController
import androidx.navigation.compose.NavHost
import androidx.navigation.compose.composable
import androidx.navigation.compose.currentBackStackEntryAsState
import androidx.navigation.compose.rememberNavController
import com.adresilo.android.ui.theme.AdresiloTheme
import com.google.openlocationcode.OpenLocationCode
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.first
import kotlinx.coroutines.flow.firstOrNull
import kotlinx.coroutines.flow.flow
import kotlinx.coroutines.flow.map
import kotlinx.coroutines.launch
import kotlinx.coroutines.runBlocking
import kotlinx.coroutines.withContext
import okhttp3.Call
import okhttp3.Callback
import okhttp3.HttpUrl.Companion.toHttpUrl
import okhttp3.OkHttpClient
import okhttp3.Request
import okhttp3.Response
import org.json.JSONArray
import java.io.IOException
import java.net.URLDecoder
import java.nio.charset.StandardCharsets

class GeoIntentData(val lat: Double, val lon: Double, val query: String) {
    override fun toString(): String {
        return if (query.isEmpty()) {
            "geo:$lat,$lon"
        } else {
            "geo:$lat,$lon?q=$query"
        }
    }
}

class MainActivity : ComponentActivity() {
    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        val api = API(applicationContext)
        
        val initialQuery = extractGeoIntentData(intent)
        
        setContent {
            AdresiloTheme {
                Adresilo(
                    modifier = Modifier,
                    api = api,
                    initialQuery = initialQuery
                )
            }
        }
    }

    private fun extractGeoIntentData(intent: Intent?): GeoIntentData? {
        val data: String? = intent?.data?.toString()
        if (data != null && data.startsWith("geo:")) {
            var query = ""
            val uri = data.replace("geo:", "geo://").toUri()
            if (uri.getQueryParameter("q") != null && uri.getQueryParameter("q") != "") {
                // ?: "" is to give smartcast a String type hint
                query = uri.getQueryParameter("q") ?: ""
            }
            val coordinatePair = uri.host
            if (coordinatePair != null && coordinatePair.contains(",")) {
                val lat = coordinatePair.substringBefore(",").trim().toDoubleOrNull() ?: 0.0
                val lon = coordinatePair.substringAfter(",").trim().toDoubleOrNull() ?: 0.0
                return GeoIntentData(lat, lon, query)
            }
        }
        return null
    }
}

enum class Screen(@StringRes val title: Int) {
    Start(title = R.string.app_name),
    Settings(title = R.string.settings),
}

val Context.dataStore: DataStore<Preferences> by preferencesDataStore(name = "settings")
val BASE_URL = stringPreferencesKey("base_url")
val KEYS = stringSetPreferencesKey("keys")
val TRIAL_KEY = stringPreferencesKey("trial_key")

@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun Adresilo(
    modifier: Modifier = Modifier,
    api: API,
    navController: NavHostController = rememberNavController(),
    initialQuery: GeoIntentData?,
) {

    val context = LocalContext.current

    var query by rememberSaveable { mutableStateOf("") }
    
    val places by api.places.collectAsState()
    val requestState by api.requestState.collectAsState()
    val searchFocusRequester = remember { FocusRequester() }

    val backStackEntry by navController.currentBackStackEntryAsState()
    val currentScreen = Screen.valueOf(
        backStackEntry?.destination?.route ?: Screen.Start.name
    )

    val scope = rememberCoroutineScope()
    var baseUrl by rememberSaveable { mutableStateOf("") }

    if (initialQuery != null) {
        LaunchedEffect(initialQuery) {
            val lat = initialQuery.lat
            val lon = initialQuery.lon
            val intentQuery = initialQuery.query
            if (lat != 0.0 && lon != 0.0) {
                val olc = OpenLocationCode(lat, lon).toString()
                if (intentQuery.isNotBlank()) {
                    val place = Place(
                        name = intentQuery,
                        address = "$lat, $lon",
                        olc = olc,
                        olcURI = "geo:$lat,$lon?q=$olc".replace("+", "%2B"),
                        lat = lat,
                        lon = lon,
                        coordsURI = "geo:$lat,$lon?q=$intentQuery",
                        source = "intent"
                    )
                    api.places.value = listOf(place)
                } else {
                    val place = Place(
                        name = "Coordinates from another app",
                        address = "$lat, $lon",
                        olc = olc,
                        olcURI = "geo:$lat,$lon?q=$olc".replace("+", "%2B"),
                        lat = lat,
                        lon = lon,
                        coordsURI = "geo:$lat,$lon",
                        source = "intent"
                    )
                    api.places.value = listOf(place)
                }
            } else if (lat == 0.0 && lon == 0.0) {
                if (intentQuery.isNotBlank()) {
                    query = intentQuery
                    api.search(query)
                } else {
                    api.requestState.value = RequestState("error", "No coordinates or query provided.")
                }
            }
        }
    }

    LaunchedEffect(Unit) {
        context.dataStore.data
            .map { preferences ->
                preferences[BASE_URL] ?: context.getString(R.string.base_url)
            }
            .collect { value ->
                baseUrl = value
            }
    }

    var newKey by rememberSaveable { mutableStateOf("") }
    var keys by rememberSaveable { mutableStateOf(setOf<String>()) }
    var keyStatuses by remember { mutableStateOf<Map<String, Int?>>(emptyMap()) }

    LaunchedEffect(keys) {
        val newStatuses = mutableMapOf<String, Int?>()
        keys.forEach { key ->
            newStatuses[key] = null // Mark as loading
            keyStatuses = newStatuses.toMap()

            try {
                val httpUrl = baseUrl.toHttpUrl()
                val count = api.keyStatus(httpUrl, key).first()
                newStatuses[key] = count
                keyStatuses = newStatuses.toMap()
            } catch (e: Exception) {
                newStatuses[key] = -1 // Mark as error
                keyStatuses = newStatuses.toMap()
            }
        }
    }

    LaunchedEffect(Unit) {
        context.dataStore.data
            .map { preferences ->
                preferences[KEYS] ?: setOf()
            }
            .collect { theKeys ->
                keys = theKeys
            }
    }

    Scaffold(
        modifier = modifier
            .fillMaxSize(),
        topBar = {
            TopAppBar(
                title = { Text(stringResource(currentScreen.title)) },
                navigationIcon = {
                    if (navController.previousBackStackEntry != null) {
                        IconButton(onClick = { navController.navigateUp() }) {
                            Icon(
                                imageVector = Icons.AutoMirrored.Filled.ArrowBack,
                                contentDescription = "Back button"
                            )
                        }
                    }
                },
                actions = {
                    if (navController.previousBackStackEntry == null) {
                        IconButton(onClick = { navController.navigate(Screen.Settings.name) }) {
                            Icon(
                                imageVector = Icons.Filled.Settings,
                                contentDescription = stringResource(R.string.settings)
                            )
                        }
                    }
                }
            )
        },
        content = { innerPadding ->
            NavHost(
                navController = navController,
                    startDestination = Screen.Start.name,
                modifier = Modifier
                    .fillMaxSize()
            ) {
                composable(route = Screen.Start.name) {
                    Column (
                        modifier = Modifier
                            .fillMaxSize()
                            .padding(innerPadding)
                            .imePadding()
                    ) {
                        SearchBar(
                            query = query,
                            onQueryChange = { query = it },
                            onSearch = { api.search(query) },
                            active = false,
                            onActiveChange = {},
                            placeholder = { Text("Search for a place…") },
                            leadingIcon = {
                                IconButton(
                                    onClick = { api.search(query) }
                                ) {
                                    Icon(
                                        Icons.Rounded.Search,
                                        contentDescription = "Search for a place"
                                    )
                                }
                            },
                            trailingIcon = {
                                IconButton(
                                    onClick = {
                                        query = ""
                                        api.places.value = emptyList()
                                        api.requestState.value = RequestState("idle", null)
                                    }
                                ) {
                                    Icon(
                                        Icons.Rounded.Cancel,
                                        contentDescription = "Clear search field and results"
                                    )
                                }
                            },
                            modifier = Modifier
                                .fillMaxWidth()
                                .padding(horizontal = 16.dp)
                                .focusRequester(searchFocusRequester)
                        ) {}
                        if (requestState.state == "idle" && places.isEmpty()) {
                            Box (
                                contentAlignment = Alignment.Center,
                                modifier = Modifier
                                    .fillMaxSize()
                                    .padding(innerPadding)
                            ) {
                                Column (
                                    horizontalAlignment = Alignment.CenterHorizontally,
                                    modifier = Modifier.padding(0.dp, 0.dp, 0.dp, 200.dp)
                                ) {
                                    Row(
                                        verticalAlignment = Alignment.CenterVertically,
                                        horizontalArrangement = Arrangement.Center,
                                        modifier = Modifier
                                            .padding(16.dp, 16.dp, 16.dp, 16.dp)
                                    ) {
                                        Text(
                                            text = "By the team behind",
                                            modifier = Modifier.padding(0.dp, 0.dp, 8.dp, 0.dp)
                                        )
                                        Image(
                                            painter = painterResource(id = R.drawable.jmp),
                                            contentDescription = "JMP logo",
                                            modifier = Modifier
                                                .size(64.dp)
                                                .fillMaxWidth()
                                                .padding(0.dp)
                                                .clickable {
                                                    val intent = Intent(
                                                        Intent.ACTION_VIEW,
                                                        "https://jmp.chat".toUri()
                                                    )
                                                    context.startActivity(intent)
                                                }
                                        )
                                    }
                                    Button(
                                        onClick = {
                                            val intent = Intent(
                                                Intent.ACTION_VIEW,
                                                "https://git.sr.ht/~amolith/adresilo".toUri()
                                            )
                                            context.startActivity(intent)
                                        }
                                    ) {
                                        Icon(
                                            imageVector = Icons.Rounded.Code,
                                            contentDescription = "Code icon",
                                            modifier = Modifier.size(ButtonDefaults.IconSize)
                                        )
                                        Spacer(modifier = Modifier.size(ButtonDefaults.IconSpacing))
                                        Text("Source code")
                                    }
                                }
                            }
                        } else if (requestState.state == "loading") {
                            Box(
                                modifier = Modifier
                                    .fillMaxWidth()
                                    .padding(16.dp, 200.dp),
                                contentAlignment = Alignment.Center
                            ) {
                                CircularProgressIndicator(
                                    modifier = Modifier.width(128.dp),
                                    color = MaterialTheme.colorScheme.secondary,
                                    trackColor = MaterialTheme.colorScheme.surfaceVariant,
                                )
                            }
                        } else if (requestState.state == "error") {
                            Box(
                                modifier = Modifier
                                    .fillMaxWidth()
                                    .padding(16.dp, 16.dp, 16.dp, 200.dp)
                            ) {
                                Text(
                                    text = "Error: ${requestState.message}",
                                    modifier = Modifier
                                        .fillMaxWidth()
                                        .padding(16.dp)
                                )
                            }
                        } else if (places.isNotEmpty()) {
                            LazyColumn(
                                modifier = Modifier.consumeWindowInsets(innerPadding),
                            ) {
                                items(places) { place ->
                                    SearchCard(place)
                                }
                                if (places.any { it.source == "gmaps" }) {
                                    item {
                                        Spacer(modifier = Modifier.fillMaxHeight())
                                        Row(
                                            modifier = Modifier
                                                .fillMaxWidth()
                                                .padding(16.dp)
                                        ) {
                                            Text(
                                                text = "Results from ",
                                                style = MaterialTheme.typography.bodyMedium,
                                                modifier = Modifier.align(Alignment.CenterVertically)
                                            )
                                            Spacer(modifier = Modifier.width(8.dp))
                                            Image(
                                                painter = painterResource(
                                                    id = if (isSystemInDarkTheme()) R.drawable.google_on_non_white
                                                    else R.drawable.google_on_white
                                                ),
                                                contentDescription = "Google",
                                                modifier = Modifier.size(60.dp)
                                            )
                                        }
                                    }
                                }
                            }
                        } else {
                            Box(
                                modifier = Modifier
                                    .fillMaxWidth()
                                    .padding(16.dp, 16.dp, 16.dp, 200.dp)
                            ) {
                                Text(
                                    text = "If you can see this screen, something's borked.",
                                    modifier = Modifier
                                        .fillMaxWidth()
                                        .padding(16.dp)
                                )
                            }
                        }
                    }
                }

                composable(route = Screen.Settings.name) {
                    Column(
                        modifier = Modifier
                            .fillMaxSize()
                            .padding(innerPadding)
                            .imePadding(),
                        verticalArrangement = Arrangement.spacedBy(16.dp)
                    ) {
                        TextField(
                            value = baseUrl,
                            onValueChange = { baseUrl = it },
                            modifier = Modifier
                                .fillMaxWidth()
                                .padding(horizontal = 16.dp),
                            label = { Text("Server URL") }
                        )

                        Button(
                            onClick = {
                                scope.launch {
                                    context.dataStore.edit { preferences ->
                                        preferences[BASE_URL] = baseUrl
                                    }
                                }
                            },
                            modifier = Modifier
                                .fillMaxWidth()
                                .padding(horizontal = 16.dp),
                        ) {
                            Text("Save")
                        }

                        TextField(
                            value = newKey,
                            onValueChange = { newKey = it },
                            modifier = Modifier
                                .fillMaxWidth()
                                .padding(horizontal = 16.dp),
                            label = { Text("Enter a Key") }
                        )

                        Button(
                            onClick = {
                                if (newKey.isNotBlank()) {
                                    scope.launch {
                                        context.dataStore.edit { preferences ->
                                            val updatedKeys = (preferences[KEYS] ?: setOf()) + newKey
                                            preferences[KEYS] = updatedKeys
                                            newKey = ""
                                        }
                                    }
                                }
                            },
                            modifier = Modifier
                                .fillMaxWidth()
                                .padding(horizontal = 16.dp),
                        ) {
                            Text("Add")
                        }

                        LazyColumn(
                            modifier = Modifier
                                .fillMaxWidth()
                                .padding(horizontal = 16.dp),
                        ) {
                            items(keys.toList()) { key ->
                                var keyToRemove by remember { mutableStateOf<String?>(null) }
                                
                                Row(
                                    modifier = Modifier.padding(8.dp),
                                    verticalAlignment = Alignment.CenterVertically
                                ) {
                                    Text(
                                        text = key,
                                        style = MaterialTheme.typography.bodyLarge,
                                        modifier = Modifier.weight(1f)
                                    )
                                    when (val status = keyStatuses[key]) {
                                        null -> CircularProgressIndicator(modifier = Modifier.size(16.dp))
                                        -1 -> Text("Error fetching status")
                                        else -> Text("$status queries left")
                                    }
                                    IconButton(
                                        onClick = { keyToRemove = key }
                                    ) {
                                        Icon(
                                            Icons.Rounded.Cancel,
                                            contentDescription = "Remove key",
                                        )
                                    }
                                }

                                keyToRemove?.let { key ->
                                    AlertDialog(
                                        onDismissRequest = { keyToRemove = null },
                                        title = { Text("Confirm Key Removal") },
                                        text = { Text("Permanently remove key from app? There is no recovery mechanism, so it will be lost unless you've recorded it somewhere else, like a password manager.") },
                                        dismissButton = {
                                            TextButton(
                                                onClick = { keyToRemove = null }
                                            ) {
                                                Text("Cancel")
                                            }
                                        },
                                        confirmButton = {
                                            Button(
                                                colors = ButtonDefaults.buttonColors(
                                                    containerColor = MaterialTheme.colorScheme.errorContainer,
                                                    contentColor = MaterialTheme.colorScheme.error
                                                ),
                                                onClick = {
                                                    scope.launch {
                                                        context.dataStore.edit { preferences ->
                                                            val updatedKeys = (preferences[KEYS] ?: setOf()) - key
                                                            preferences[KEYS] = updatedKeys
                                                        }
                                                        keyToRemove = null
                                                    }
                                                }
                                            ) {
                                                Text("Remove key")
                                            }
                                        }
                                    )
                                }
                            }
                        }
                    }
                }
            }
        },
        floatingActionButton = {
            FloatingActionButton(
                modifier = Modifier
                    .imePadding(),
                onClick = { searchFocusRequester.requestFocus() },
                containerColor = MaterialTheme.colorScheme.secondaryContainer,
                contentColor = MaterialTheme.colorScheme.secondary
            ) {
                Icon(Icons.Rounded.Keyboard, contentDescription = "Focus search field")
            }
        },
    )
}

@Composable
fun SearchCard(place: Place) {
    val context = LocalContext.current
    Card(
        onClick = { context.open(place) },
        modifier = Modifier
            .padding(
                start = 16.dp, top = 16.dp, end = 16.dp, bottom = 0.dp
            )
            .fillMaxWidth()
    ) {
        Box(
            modifier = Modifier
                .fillMaxSize()
                .padding(16.dp)
        ) {
            Row (
                verticalAlignment = Alignment.CenterVertically,
                modifier = Modifier.fillMaxWidth(),
                horizontalArrangement = Arrangement.SpaceBetween
            ) {
                Column (
                    modifier = Modifier.weight(1f)
                ){
                    Text(
                        place.name,
                        fontSize = MaterialTheme.typography.titleMedium.fontSize
                    )
                    Spacer(modifier = Modifier.height(8.dp))
                    Text(
                        place.address,
                        fontSize = MaterialTheme.typography.bodySmall.fontSize,
                        maxLines = 2,
                        overflow = TextOverflow.Ellipsis
                    )
                }
                IconButton(
                    onClick = { context.copyToClipboard("${place.lat}, ${place.lon}") },
                ) {
                    Icon(
                        Icons.Rounded.ContentCopy,
                        contentDescription = "Copy coordinates",
                        Modifier.size(24.dp)
                    )
                }
                IconButton(
                    onClick = { context.share(place) },
                ) {
                    Icon(
                        Icons.Rounded.Share,
                        contentDescription = "Share location",
                        Modifier.size(24.dp)
                    )
                }
                IconButton(
                    onClick = { context.open(place) }
                ) {
                    Icon(
                        Icons.AutoMirrored.Rounded.OpenInNew,
                        contentDescription = "Open in routing app",
                        Modifier.size(24.dp)
                    )
                }
            }
        }
    }
}

fun Context.share(place: Place) {
    val message = "${place.name}\n${place.address}\nOpen Location Code: ${place.olc}\nCoordinates: ${place.lat}, ${place.lon}"
    val sendIntent: Intent = Intent().apply {
        action = Intent.ACTION_SEND
        putExtra(Intent.EXTRA_TEXT, message)
        type = "text/plain"
    }
    val shareIntent = Intent.createChooser(sendIntent, null)
    try {
        startActivity(shareIntent)
    } catch (e: Exception) {
        // TODO: Show a toast saying the user needs to install
        //  an app that supports sharing
        Log.e("Adresilo", "Error sharing: $e")
    }
}

fun Context.open(place: Place) {
    val uri = place.coordsURI.toUri()
    val intent = Intent(Intent.ACTION_VIEW)
    intent.data = uri
    try {
        startActivity(this, intent, null)
    } catch (e: Exception) {
        // TODO: Show a toast saying the user needs to install
        //  an app that supports navigation
        Log.e("Adresilo", "Error opening map: $e")
    }
}

fun Context.copyToClipboard(text: String) {
    val clipboard = getSystemService(Context.CLIPBOARD_SERVICE) as android.content.ClipboardManager
    val clip = android.content.ClipData.newPlainText("Copied text", text)
    clipboard.setPrimaryClip(clip)
}

class Place(val name: String, val address: String, val olc: String, private val olcURI: String, val lat: Double, val lon: Double, val coordsURI: String, val source: String) {
    override fun toString(): String {
        return "Place(name='$name', address='$address', olc='$olc', olcURI='${olcURI.replace("+", "%2B")}', lat=$lat, lon=$lon, coordsURI='$coordsURI', source='$source')"
    }
}

class RequestState(var state: String, var message: String?) {
    override fun toString(): String {
        return "RequestState(state='$state', message='$message')"
    }
}

class API(private val context: Context) {
    // Immutable reference to a mutable object
    val places = MutableStateFlow(listOf<Place>())

    val requestState = MutableStateFlow(RequestState("idle", null))

    fun search(query: String) {
        if (query.isEmpty()) {
            places.value = emptyList()
            requestState.value = RequestState("idle", null)
            return
        }

         val request = runBlocking { flow {
            val baseUrl = context.dataStore.data.map { prefs -> prefs[BASE_URL] ?: context.getString(R.string.base_url) }.first().toHttpUrl()
            val keys = context.dataStore.data.map { prefs -> prefs[KEYS] }.first() ?: setOf()
            val key = firstValidKey(baseUrl, keys + setOf(trialKey().first())).firstOrNull() ?: trialKey().first()
            emit(
                Request.Builder()
                .url(baseUrl.newBuilder().addPathSegment("search").addQueryParameter("q", query).build())
                .header("Authorization", "Bearer $key")
                .get()
                .build()
            )
        }.firstOrNull() }

        if (request == null) return

        requestState.value = RequestState("loading", null)
        val client = OkHttpClient()
        client.newCall(request).enqueue(object : Callback {
            override fun onFailure(call: Call, e: IOException) {
                if (e.message.orEmpty() == "") {
                    requestState.value = RequestState("error", "unknown error. Please contact the developer.")
                    return
                }
                requestState.value = RequestState("error", e.message)
                e.printStackTrace()
            }

            override fun onResponse(call: Call, response: Response) {
                response.use {
                    if (!response.isSuccessful) {
                        requestState.value = RequestState("error", response.body!!.string())
                        throw IOException("Unexpected code $response")
                    }

                    if (response.body == null) {
                        requestState.value = RequestState("error", "no results returned from data source. Try a different query.")
                        return
                    }

                    val responseBody = response.body!!.string()

                    val list: List<Place>
                    try {
                        list = responseToList(responseBody)
                    } catch (e: Exception) {
                        requestState.value = RequestState("error", e.message)
                        return
                    }

                    places.value = list
                    requestState.value = RequestState("idle", null)
                }
            }
        })
    }

    private fun firstValidKey(baseUrl: okhttp3.HttpUrl, keys: Set<String>): Flow<String> = flow {
        for (key in keys) {
            if (keyStatus(baseUrl, key).firstOrNull { it > 0 } != null) {
                emit(key)
                return@flow
            }
        }
    }

    fun keyStatus(baseUrl: okhttp3.HttpUrl, key: String): Flow<Int> = flow {
        val result = withContext(Dispatchers.IO) {
            val client = OkHttpClient()

            val request = Request.Builder()
                .url(baseUrl.newBuilder().addPathSegment("key-status").build())
                .header("Authorization", "Bearer $key")
                .get()
                .build()

            val response: Response = client.newCall(request).execute()

            if (response.isSuccessful) {
                response.body?.string()?.trim()?.toIntOrNull() ?: 0
            } else {
                0
            }
        }

        emit(result)
    }

    private fun trialKey(): Flow<String> = flow {
        val trialKey = context.dataStore.data.map { prefs -> prefs[TRIAL_KEY] }.firstOrNull()
        if (trialKey == null) {
            val letters = ('a'..'z').toList()
            val newTrialKey = "~" + (1..4).map { letters.random() }.joinToString("")
            context.dataStore.edit { prefs -> prefs[TRIAL_KEY] = newTrialKey }
            emit(newTrialKey)
        } else {
            emit(trialKey)
        }
    }

    fun responseToList(response: String): List<Place> {
        val placesJSON = JSONArray(response)
        val pList = mutableListOf<Place>()
        for (i in 0 until placesJSON.length()) {
            val place = placesJSON.getJSONObject(i)
            val name = place.getString("Name")
            val address = place.getString("Address")
            val lat = place.optDouble("Lat", 0.0)
            val lon = place.optDouble("Lon", 0.0)
            if (lat == 0.0 || lon == 0.0) {
                continue
            }
            val olc = place.getString("OLC") // OpenLocationCode(lat, lon).toString()
            val olcURI = "geo:$lat,$lon?q=$olc".replace("+", "%2B")
            val coordsURI = "geo:$lat,$lon?q=$lat,$lon"
            val source = place.getString("Source")
            pList.add(Place(name, address, olc, olcURI, lat, lon, coordsURI, source))
        }
        if (pList.isEmpty()) {
            throw Exception("no usable results. Try a different query.")
        }
        return pList
    }
}

@Preview(showBackground = true)
@Composable
fun PreviewMain() {
    AdresiloTheme {
        Adresilo(
            modifier = Modifier,
            api = API(LocalContext.current),
            initialQuery = null
        )
    }
}
