/*
 * Copyright (c) 2018 Gnock
 * Copyright (c) 2018-2019 The Masari Project
 * Copyright (c) 2018-2020 The Karbo developers
 * Copyright (c) 2018-2023 Conceal Community, Conceal.Network & Conceal Devs
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following disclaimer in the documentation and/or other materials provided with the distribution.
 *
 * 3. Neither the name of the copyright holder nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

const MSB = 0x80;
const REST = 0x7f;
const MSBALL = ~REST;
const INT = 2 ** 31;
const TWO_POWER_SEVEN = 2 ** 7;

export const encode = ((num: number, out: number[] | Uint8Array = [], offset = 0): number[] | Uint8Array => {
  if (Number.MAX_SAFE_INTEGER && num > Number.MAX_SAFE_INTEGER) {
    encode.bytes = 0;
    throw new RangeError('Could not encode varint');
  }
  const oldOffset = offset;

  while (num >= INT) {
    out[offset++] = (num & 0xff) | MSB;
    num /= 128;
  }
  while (num & MSBALL) {
    out[offset++] = (num & 0xff) | MSB;
    num >>>= 7;
  }
  out[offset] = num | 0;

  encode.bytes = offset - oldOffset + 1;

  return out;
}) as ((num: number, out?: number[] | Uint8Array, offset?: number) => number[] | Uint8Array) & { bytes: number };

export const decode = ((buf: ArrayLike<number>, offset = 0): number => {
  let res = 0,
    shift = 1,
    counter = offset,
    b;
  const l = TWO_POWER_SEVEN ** (buf.length - offset < 8 ? (buf.length - offset) * 7 : 49);

  do {
    if (shift > l) {
      decode.bytes = 0;
      throw new RangeError('Could not decode varint');
    }
    b = buf[counter++];
    res += (b & REST) * shift;
    shift = shift * TWO_POWER_SEVEN;
  } while (b >= MSB);

  decode.bytes = counter - offset;

  return res;
}) as ((buf: ArrayLike<number>, offset?: number) => number) & { bytes: number };
