import { Ionicons } from '@expo/vector-icons';
import { createBottomTabNavigator } from '@react-navigation/bottom-tabs';
import { Platform, View } from 'react-native';
import { useSafeAreaInsets } from 'react-native-safe-area-context';
import { useTheme } from '../contexts/ThemeContext';
import HomeScreen from '../screens/HomeScreen';
import SettingsScreen from '../screens/SettingsScreen';
import WalletScreen from '../screens/WalletScreen';

const Tab = createBottomTabNavigator();

// Custom tab icon component with hover effect
function TabIcon({
  name,
  size,
  color,
  focused,
  theme,
}: {
  name: keyof typeof Ionicons.glyphMap;
  size: number;
  color: string;
  focused: boolean;
  theme: any;
}) {
  if (Platform.OS === 'web') {
    return (
      <div
        style={{
          display: 'flex',
          alignItems: 'center',
          justifyContent: 'center',
          transition: 'all 0.2s ease-in-out',
          filter: focused && theme.isDark ? 'drop-shadow(0 0 1px #FFA500)' : 'none',
        }}
        onMouseEnter={(e) => {
          if (theme.isDark) {
            const target = e.currentTarget as HTMLDivElement;
            target.style.filter = 'drop-shadow(0 0 2px #FFA500)';
            target.style.transform = 'scale(1.1)';
          }
        }}
        onMouseLeave={(e) => {
          if (theme.isDark) {
            const target = e.currentTarget as HTMLDivElement;
            target.style.filter = focused ? 'drop-shadow(0 0 2px #FFA500)' : 'none';
            target.style.transform = 'scale(1)';
          }
        }}
      >
        <Ionicons name={name} size={size} color={color} />
      </div>
    );
  }

  return <Ionicons name={name} size={size} color={color} />;
}

export default function TabNavigator() {
  const { theme } = useTheme();
  const insets = useSafeAreaInsets();

  return (
    <View className="flex-1">
      <Tab.Navigator
        screenOptions={({ route }) => ({
          tabBarIcon: ({ focused, color, size }) => {
            let iconName: keyof typeof Ionicons.glyphMap;

            if (route.name === 'Home') {
              iconName = focused ? 'shield-checkmark' : 'shield-checkmark-outline';
            } else if (route.name === 'Wallet') {
              iconName = focused ? 'wallet' : 'wallet-outline';
            } else if (route.name === 'Settings') {
              iconName = focused ? 'settings' : 'settings-outline';
            } else {
              iconName = 'help-outline';
            }

            return <TabIcon name={iconName} size={size} color={color} focused={focused} theme={theme} />;
          },
          tabBarActiveTintColor: theme.colors.tabBarActive,
          tabBarInactiveTintColor: theme.colors.tabBarInactive,
          tabBarStyle: {
            backgroundColor: theme.colors.tabBar,
            borderTopWidth: 1,
            borderTopColor: theme.colors.border,
            paddingTop: 8,
            paddingBottom: Math.max(13, insets.bottom), // Ensure tab bar is above system navigation
            height: 65 + insets.bottom, // Add bottom inset to total height
          },
          headerShown: false,
        })}
        initialRouteName="Home"
      >
        <Tab.Screen
          name="Home"
          component={HomeScreen}
          options={{
            title: 'Authenticator',
            tabBarLabelStyle: {
              fontFamily: 'Poppins-Light',
              fontSize: 11,
            },
          }}
        />
        <Tab.Screen
          name="Wallet"
          component={WalletScreen}
          options={{
            title: 'Wallet',
            tabBarLabelStyle: {
              fontFamily: 'Poppins-Light',
              fontSize: 11,
            },
          }}
        />
        <Tab.Screen
          name="Settings"
          component={SettingsScreen}
          options={{
            title: 'Settings',
            tabBarLabelStyle: {
              fontFamily: 'Poppins-Light',
              fontSize: 11,
            },
          }}
        />
      </Tab.Navigator>
    </View>
  );
}
