package com.ngrob.android.bluemoon.features.symptom.screens

import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import com.ngrob.android.bluemoon.core.data.repository.BleedingRepository
import com.ngrob.android.bluemoon.core.data.repository.SexRepository
import com.ngrob.android.bluemoon.core.database.model.BleedingStrength
import com.ngrob.android.bluemoon.core.database.model.ContraceptiveEnum
import com.ngrob.android.bluemoon.core.database.model.SexType
import com.ngrob.android.bluemoon.features.calendar.components.TrackingTypes
import dagger.hilt.android.lifecycle.HiltViewModel
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.asStateFlow
import kotlinx.coroutines.flow.update
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import java.time.LocalDate
import javax.inject.Inject

interface Option<T> {
    val question: String
    val options: List<T>
}

data class SingleChoiceOption<T>(
    override val question: String,
    override val options: List<T>,
    var selection: T?,
) : Option<T>{
    fun copyWithSelection(selection: T?): SingleChoiceOption<T> {
        return SingleChoiceOption(question, options, selection)
    }

}
data class MultipleChoiceOption<T>(
    override val question: String,
    override val options: List<T>,
    var selection: List<T>,
) : Option<T>{
    fun copyWithSelection(selection: List<T>): MultipleChoiceOption<T> {
        return MultipleChoiceOption(question, options, selection)
    }
}


//sealed class InputOptions(
//    val symptom: TrackingTypes,
//    var input: List<Option<Any>>,
//) {
//    data object Bleeding : InputOptions(
//        TrackingTypes.BLEEDING, listOf(
//            SingleChoiceOption(
//                question = "How heavy is the bleeding?",
//                options = BleedingStrength.entries.toList(),
//                selection = null
//            )
//        )
//    )
//    data object Sex : InputOptions(
//        TrackingTypes.SEX, listOf(
//            SingleChoiceOption(
//                "Where you sexually active?",
//                SexType.entries.toList(),
//                selection = null,
//            ),
//            MultipleChoiceOption(
//                "Did you use contraceptives?",
//                options = ContraceptiveEnum.entries.toList(),
//                selection = listOf()
//            )
//        )
//    )
//}


data class AddSymptomUiState(
    val currentSymptom: TrackingTypes = TrackingTypes.BLEEDING,
    val symptomInputMap: Map<TrackingTypes, List<Option<Any>>> = mapOf(
        TrackingTypes.BLEEDING to listOf(
            SingleChoiceOption(
                question = "How heavy is the bleeding?",
                options = BleedingStrength.entries.toList(),
                selection = null
            )
        ),
        TrackingTypes.SEX to listOf(
            SingleChoiceOption(
                "Where you sexually active?",
                SexType.entries.toList(),
                selection = null,
            ),
            MultipleChoiceOption(
                "Did you use contraceptives?",
                options = ContraceptiveEnum.entries.toList(),
                selection = listOf()
            )
        ),
        TrackingTypes.MEDICATION to listOf(),
        TrackingTypes.MOOD to listOf(),
    ),
    val saveButtonEnabled: Boolean = false,
)

@HiltViewModel
class AddSymptomViewModel @Inject constructor(
    private val bleedingRepository: BleedingRepository,
    private val sexRepository: SexRepository,
) : ViewModel() {
    private val _uiState = MutableStateFlow(AddSymptomUiState())
    val uiState = _uiState.asStateFlow()

    fun selectTrackingType(newTrackingType: TrackingTypes) {
        _uiState.update { currentState ->
            val updatedSaveButtonEnabled = this.allQuestionAnswered(currentState.symptomInputMap[newTrackingType]!!)
            currentState.copy(
                currentSymptom = newTrackingType,
                saveButtonEnabled = updatedSaveButtonEnabled
            )
        }
    }

    fun answerQuestion(questionIdx: Int, selectedOption: Any) {
        _uiState.update { currentState ->
            val updatedSymptomInputMap = currentState.symptomInputMap.toMutableMap()
            val inputOption = currentState.symptomInputMap[currentState.currentSymptom]?.get(questionIdx)
            val updatedInput = currentState.symptomInputMap[currentState.currentSymptom]?.toMutableList()
            if (inputOption is SingleChoiceOption && updatedInput != null) {
                updatedInput[questionIdx] = inputOption.copyWithSelection(selectedOption)
                updatedSymptomInputMap[currentState.currentSymptom] = updatedInput.toList()
            }
            if (inputOption is MultipleChoiceOption && updatedInput != null) {
                val updatedSelection = inputOption.selection.toMutableList()
                updatedSelection.add(selectedOption)
                updatedInput[questionIdx] = inputOption.copyWithSelection(updatedSelection)
                updatedSymptomInputMap[currentState.currentSymptom] = updatedInput.toList()
            }
            val updatedSaveButtonEnabled = if (updatedInput != null) this.allQuestionAnswered(updatedInput) else false

            currentState.copy(symptomInputMap = updatedSymptomInputMap, saveButtonEnabled = updatedSaveButtonEnabled)
        }
    }

    fun deselectOption(questionIdx: Int, deselectedOption: Any) {
        _uiState.update { currentState ->
            val updatedSymptomInputMap = currentState.symptomInputMap.toMutableMap()
            val inputOption = currentState.symptomInputMap[currentState.currentSymptom]?.get(questionIdx)
            val updatedInput = currentState.symptomInputMap[currentState.currentSymptom]?.toMutableList()
            if (inputOption is SingleChoiceOption && updatedInput != null) {
                updatedInput[questionIdx] = inputOption.copyWithSelection(null)
                updatedSymptomInputMap[currentState.currentSymptom] = updatedInput.toList()
            }
            if (inputOption is MultipleChoiceOption && updatedInput != null) {
                val updatedSelection = inputOption.selection.toMutableList()
                updatedSelection.remove(deselectedOption)
                updatedInput[questionIdx] = inputOption.copyWithSelection(updatedSelection)
                updatedSymptomInputMap[currentState.currentSymptom] = updatedInput.toList()
            }
            val updatedSaveButtonEnabled = if (updatedInput != null) this.allQuestionAnswered(updatedInput) else false
            currentState.copy(symptomInputMap = updatedSymptomInputMap, saveButtonEnabled = updatedSaveButtonEnabled)
        }
    }
    private fun allQuestionAnswered(questions: List<Option<Any>>): Boolean{
        var allQuestionsAnswered = true
        questions.forEach { option ->
            if (option is SingleChoiceOption && option.selection == null){
                allQuestionsAnswered = false
                return@forEach
            }
            if (option is MultipleChoiceOption && option.selection.isEmpty()){
                allQuestionsAnswered = false
                return@forEach
            }
        }
        return allQuestionsAnswered
    }

    fun saveData(currentDate: LocalDate){
        viewModelScope.launch {
            if(_uiState.value.currentSymptom == TrackingTypes.SEX){
                val inputs = _uiState.value.symptomInputMap[TrackingTypes.SEX]!!
                val sexType = (inputs[0] as SingleChoiceOption).selection as SexType
                val contraceptives = (inputs[1] as MultipleChoiceOption).selection as List<ContraceptiveEnum>
                withContext(Dispatchers.IO) {
                    sexRepository.addSex(sexType, contraceptives, currentDate)
                }
            }
            if(_uiState.value.currentSymptom == TrackingTypes.BLEEDING){
                val inputs = _uiState.value.symptomInputMap[TrackingTypes.BLEEDING]!!
                val bleedingStrength = (inputs[0] as SingleChoiceOption).selection as BleedingStrength
                withContext(Dispatchers.IO){
                    bleedingRepository.updateBleeding(bleedingStrength, currentDate)
                }
            }
        }

    }
}
