package com.ngrob.android.bluemoon.features.symptom.screens

import android.annotation.SuppressLint
import androidx.compose.foundation.background
import androidx.compose.foundation.clickable
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.ExperimentalLayoutApi
import androidx.compose.foundation.layout.fillMaxHeight
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.width
import androidx.compose.foundation.lazy.LazyColumn
import androidx.compose.foundation.lazy.LazyRow
import androidx.compose.foundation.lazy.items
import androidx.compose.foundation.lazy.itemsIndexed
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.material.Card
import androidx.compose.material.MaterialTheme
import androidx.compose.material.Scaffold
import androidx.compose.material.Text
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.runtime.Composable
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.runtime.rememberCoroutineScope
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.draw.clip
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.font.FontWeight
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.sp
import androidx.hilt.navigation.compose.hiltViewModel
import androidx.navigation.NavController
import com.ngrob.android.bluemoon.core.ui.component.Button
import com.ngrob.android.bluemoon.core.ui.theme.Gray900
import com.ngrob.android.bluemoon.core.ui.theme.White
import com.ngrob.android.bluemoon.features.calendar.components.CardEntryItem
import com.ngrob.android.bluemoon.features.symptom.components.MultipleChoiceOptionComponent
import com.ngrob.android.bluemoon.features.symptom.components.SingleChoiceOptionComponent
import kotlinx.coroutines.launch
import java.time.LocalDate

@Composable
fun SelectSymptomCard(
    item: CardEntryItem, modifier: Modifier = Modifier, onSelect: () -> Unit
) {
    Card(
        modifier
            .padding(20.dp)
            .width(120.dp)
            .clip(RoundedCornerShape(20.dp))
            .clickable {
                onSelect()
            }, shape = RoundedCornerShape(20.dp)
    ) {
        Column(
            horizontalAlignment = Alignment.CenterHorizontally,
            modifier = Modifier
                .background(color = item.color)
                .padding(10.dp)
        ) {
            Text(item.type.displayString, fontSize = 18.sp, fontWeight = FontWeight.W500)
            Text(
                stringResource(item.displayEmoji), fontSize = 40.sp
            )
        }
    }
}

@OptIn(ExperimentalMaterial3Api::class, ExperimentalLayoutApi::class)
@SuppressLint("UnusedMaterialScaffoldPaddingParameter")
@Composable
fun AddSymptomScreen(
    modifier: Modifier = Modifier,
    navController: NavController,
    viewModel: AddSymptomViewModel = hiltViewModel(),
    currentDate: LocalDate,
) {

    val uiState by viewModel.uiState.collectAsState()
    val coroutineScope = rememberCoroutineScope()
    Scaffold(modifier) {
        Column(Modifier.background(color = Gray900)) {
            Text(
                text = "Add data",
                style = MaterialTheme.typography.h2,
                modifier = Modifier.padding(20.dp, 5.dp, 10.dp, 0.dp)
            )
            Text(
                text = "Select what you want to track",
                style = MaterialTheme.typography.body2,
                modifier = Modifier.padding(20.dp, 5.dp, 10.dp, 0.dp)
            )
            LazyRow {
                items(
                    listOf(
                        CardEntryItem.Bleeding,
                        CardEntryItem.Sex,
//                        CardEntryItem.Medication,
//                        CardEntryItem.Mood
                    )
                ) {
                    SelectSymptomCard(
                        item = it,
                        onSelect = { viewModel.selectTrackingType(it.type) })
                }
            }
            Card(
                modifier = Modifier
                    .fillMaxSize()
                    .padding(0.dp, 5.dp, 0.dp, 0.dp),
                backgroundColor = White,
                elevation = 0.dp,
                shape = RoundedCornerShape(30.dp, 30.dp, 0.dp, 0.dp)
            ) {
                Column(
                    Modifier
                        .padding(15.dp)
                        .fillMaxHeight(1f),
                    verticalArrangement = Arrangement.SpaceBetween
                ) {
                    Text(uiState.currentSymptom.displayString, fontWeight = FontWeight.W800)
                    LazyColumn(modifier = Modifier.fillMaxHeight(0.8f)) {
                        itemsIndexed(uiState.symptomInputMap[uiState.currentSymptom]!!) { index, option ->
                            if (option is SingleChoiceOption) {
                                SingleChoiceOptionComponent(
                                    singleChoice = option,
                                    onAnswer = { selectedAnswer ->
                                        viewModel.answerQuestion(index, selectedAnswer)
                                    },
                                    onDeselect = { deselectedAnswer ->
                                        viewModel.deselectOption(index, deselectedAnswer)
                                    })
                            }
                            if (option is MultipleChoiceOption) {
                                MultipleChoiceOptionComponent(
                                    multipleChoice = option,
                                    onAnswer = { selectedAnswer ->
                                        viewModel.answerQuestion(index, selectedAnswer)
                                    },
                                    onDeselect = { deselectedAnswer ->
                                        viewModel.deselectOption(index, deselectedAnswer)
                                    })
                            }
                        }
                    }
                    Button(
                        label = "Save Data", enabled = uiState.saveButtonEnabled, onClick = {
                            coroutineScope.launch {
                                viewModel.saveData(currentDate)
                            }
                            navController.popBackStack()
                        }, primary = false, modifier = Modifier
                    )
                }

            }

        }
    }
}

