package com.ngrob.android.bluemoon.features.settings.screens

import android.content.Context
import android.content.Intent
import android.net.Uri
import android.util.Log
import android.widget.Toast
import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import com.ngrob.android.bluemoon.MainActivity
import com.ngrob.android.bluemoon.core.data.repository.BleedingRepository
import com.ngrob.android.bluemoon.core.data.repository.PillRepository
import com.ngrob.android.bluemoon.core.data.repository.SexRepository
import com.ngrob.android.bluemoon.core.data.repository.UserDataRepository
import com.ngrob.android.bluemoon.core.database.BluemoonDatabase
import com.ngrob.android.bluemoon.core.model.PillRecipe
import com.ngrob.android.bluemoon.core.model.ReminderSettings
import com.ngrob.android.bluemoon.core.onboarding.extractBleedings
import com.ngrob.android.bluemoon.core.reminder.BluemoonReminderNotificationScheduler
import dagger.hilt.android.lifecycle.HiltViewModel
import de.raphaelebner.roomdatabasebackup.core.RoomBackup
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.flow.SharingStarted
import kotlinx.coroutines.flow.collectLatest
import kotlinx.coroutines.flow.map
import kotlinx.coroutines.flow.stateIn
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import java.time.LocalDateTime
import java.time.format.DateTimeFormatter
import javax.inject.Inject
import kotlin.time.Duration.Companion.seconds
import com.ngrob.android.bluemoon.core.onboarding.extractFloData
import kotlinx.coroutines.flow.MutableSharedFlow
import kotlinx.coroutines.flow.SharedFlow


data class UserEditableSettings(
    val shouldTrackBirthControl: Boolean = false,
    val pillRecipe: PillRecipe,
    val reminderSettings: ReminderSettings,
)

@HiltViewModel
class SettingsViewModel @Inject constructor(
    private val bluemoonDatabase: BluemoonDatabase,
    private val bleedingRepository: BleedingRepository,
    private val sexRepository: SexRepository,
    private val pillRepository: PillRepository,
    private val userDataRepository: UserDataRepository,
    private val bluemoonReminderNotificationScheduler: BluemoonReminderNotificationScheduler
) : ViewModel() {


    val uiState = userDataRepository.userData.map {
        UserEditableSettings(
            shouldTrackBirthControl = it.trackBirthControl,
            pillRecipe = it.pillRecipe,
            reminderSettings = it.reminderSettings
        )
    }.stateIn(
        scope = viewModelScope,
        started = SharingStarted.WhileSubscribed(5.seconds.inWholeMilliseconds),
        initialValue = null
    )

    // For triggering the Snackbar after importing data
    private val _importResult = MutableSharedFlow<ImportResult>()
    val importResult: SharedFlow<ImportResult> = _importResult

    sealed class ImportResult {
        data object Success : ImportResult()
        data class Failure(val errorMessage: String) : ImportResult()
    }

    fun deleteAllData() {
        viewModelScope.launch {
            withContext(Dispatchers.IO) {
                bleedingRepository.deleteAllBleeding()
                sexRepository.deleteAllSex()
                pillRepository.deleteAllPills()
            }
        }
    }

    fun backupData(roomBackup: RoomBackup) {
        roomBackup.backupLocation(RoomBackup.BACKUP_FILE_LOCATION_CUSTOM_DIALOG)
            .database(bluemoonDatabase).customBackupFileName(
                "bluemoon-backup-${
                    LocalDateTime.now().format(
                        DateTimeFormatter.ISO_LOCAL_DATE_TIME
                    )
                }"
            )
            .apply {
                onCompleteListener { _, message, _ ->
                    Toast.makeText(
                        context,
                        message,
                        Toast.LENGTH_LONG
                    ).show()
                }
            }.backup()
    }

    fun restoreData(roomBackup: RoomBackup) {
        roomBackup.backupLocation(RoomBackup.BACKUP_FILE_LOCATION_CUSTOM_DIALOG)
            .database(bluemoonDatabase)
            .apply {
                onCompleteListener { success, message, _ ->
                    Toast.makeText(
                        context,
                        message,
                        Toast.LENGTH_LONG
                    ).show()
                    if (success) restartApp(Intent(context, MainActivity::class.java))
                }
            }.restore()
    }

    fun setTrackBirthControl(shouldTrackBirthControl: Boolean) {
        viewModelScope.launch {
            userDataRepository.setTrackBirthControl(shouldTrackBirthControl)
            userDataRepository.userData.collectLatest { userData ->
                bleedingRepository.getAllBleedingDates().collectLatest { bleedings ->
                    bluemoonReminderNotificationScheduler.scheduleAll(userData, bleedings)
                }
            }
        }
    }

    fun setPillRecipe(pillRecipe: PillRecipe) {
        viewModelScope.launch {
            userDataRepository.setPillRecipe(pillRecipe)
        }
    }

    fun setReminderSettings(reminderSettings: ReminderSettings) {
        viewModelScope.launch {
            userDataRepository.setReminderSettings(reminderSettings)
            userDataRepository.userData.collectLatest { userData ->
                bleedingRepository.getAllBleedingDates().collectLatest { bleedings ->
                    bluemoonReminderNotificationScheduler.scheduleAll(userData, bleedings)
                }
            }
        }
    }

    /**
     * Read a JSON file with user data from Flo, which the user previously exported from Flo.
     *
     * @param uri The URI of the JSON file.
     * @param context
     */
    fun readFloJsonFile(uri: Uri, context: Context){
        val gsonObject = extractFloData(uri, context)

        // Save the data to the database
        viewModelScope.launch {
            if (gsonObject == null) {
                _importResult.emit(ImportResult.Failure("The chosen file contains no readable data."))
                return@launch
            }

            withContext(Dispatchers.IO) {
                val cycleBleedings = try {
                    extractBleedings(gsonObject)
                } catch (e: Exception) {
                    Log.w("SettingsViewModel", "Exception occurred while extracting a List of Bleeding objects from FloData object.")
                    _importResult.emit(ImportResult.Failure("Unexpected file content structure."))
                    return@withContext
                }

                Log.d("Gson object", "Extracted $cycleBleedings")
                bleedingRepository.importBleedings(cycleBleedings)
                _importResult.emit(ImportResult.Success)
            }
        }
    }
}