package com.ngrob.android.bluemoon.features.settings.screens

import android.annotation.SuppressLint
import android.content.Intent
import android.net.Uri
import androidx.compose.foundation.background
import androidx.compose.foundation.border
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.foundation.text.ClickableText
import androidx.compose.foundation.verticalScroll
import androidx.compose.material.Card
import androidx.compose.material.MaterialTheme
import androidx.compose.material.Scaffold
import androidx.compose.material.Text
import androidx.compose.material3.AlertDialog
import androidx.compose.material3.SnackbarHost
import androidx.compose.material3.SnackbarHostState
import androidx.compose.material3.TextButton
import androidx.compose.runtime.Composable
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.text.SpanStyle
import androidx.compose.ui.text.buildAnnotatedString
import androidx.compose.ui.text.font.FontWeight
import androidx.compose.ui.text.style.TextDecoration
import androidx.compose.ui.text.withStyle
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.sp
import androidx.hilt.navigation.compose.hiltViewModel
import com.ngrob.android.bluemoon.BuildConfig
import com.ngrob.android.bluemoon.R
import com.ngrob.android.bluemoon.core.ui.theme.Gray600
import com.ngrob.android.bluemoon.core.ui.theme.Gray900
import com.ngrob.android.bluemoon.core.ui.theme.White
import com.ngrob.android.bluemoon.features.settings.components.ImportSection
import com.ngrob.android.bluemoon.features.settings.components.PillSection
import de.raphaelebner.roomdatabasebackup.core.RoomBackup
import com.ngrob.android.bluemoon.features.settings.components.RemindersParagraph

@SuppressLint("UnusedMaterialScaffoldPaddingParameter")
@Composable
fun SettingsScreen(
    modifier: Modifier = Modifier,
    viewModel: SettingsViewModel = hiltViewModel(),
    roomBackup: RoomBackup,
) {
    val openAlertDialog = remember { mutableStateOf(false) }
    val uiState by viewModel.uiState.collectAsState()

    val createdBy = buildAnnotatedString {
        withStyle(
            style = MaterialTheme.typography.body1.toSpanStyle()
                .copy(color = Gray600, fontWeight = FontWeight.W500, fontSize = 12.sp)
        ) {
            append("Created by Nils Grob\nLicensed under the GNU AGPLv3")
            addStringAnnotation(
                tag = "URL",
                annotation = "https://www.nilsgrob.ch",
                start = 11,
                end = 20
            )
            addStyle(
                style = SpanStyle(textDecoration = TextDecoration.Underline),
                start = 11,
                end = 20
            )
        }
    }
    val context = LocalContext.current

    val snackbarHostState = remember { SnackbarHostState() }

    LaunchedEffect(Unit) {
        viewModel.importResult.collect { result ->
            val message = when (result) {
                is SettingsViewModel.ImportResult.Success -> context.getString(R.string.import_successful)
                is SettingsViewModel.ImportResult.Failure -> context.getString(
                    R.string.import_failed,
                    result.errorMessage
                )
            }
            snackbarHostState.showSnackbar(message)
        }
    }

    Scaffold(
        modifier.fillMaxSize(),
        snackbarHost = { SnackbarHost(snackbarHostState) }) { paddingValues ->
        Column(Modifier.background(color = Gray900)) {
            Text(
                text = "Settings",
                style = MaterialTheme.typography.h2,
                modifier = Modifier.padding(20.dp, 15.dp, 15.dp, 0.dp)
            )
            Card(
                modifier = Modifier
                    .fillMaxSize()
                    .padding(0.dp, 20.dp, 0.dp, 0.dp),
                backgroundColor = White,
                elevation = 0.dp,
                shape = RoundedCornerShape(30.dp, 30.dp, 0.dp, 0.dp)
            ) {
                Column(modifier = Modifier.verticalScroll(rememberScrollState(0))) {
                    Spacer(Modifier.padding(10.dp))
                    if (uiState !== null) {
                        PillSection(
                            shouldTrackBirthControl = uiState!!.shouldTrackBirthControl,
                            onShouldTrackBirthControlChanged = { viewModel.setTrackBirthControl(it) },
                            recipe = uiState!!.pillRecipe,
                            onRecipeChange = { viewModel.setPillRecipe(it) },
                        )
                        Spacer(
                            Modifier
                                .padding(20.dp)
                                .height(1.dp)
                                .border(2.dp, Color.Gray)
                                .fillMaxWidth()
                        )
                        RemindersParagraph(
                            uiState!!.shouldTrackBirthControl,
                            onReminderSettingsChanged = { viewModel.setReminderSettings(it) },
                            reminderSettings = uiState!!.reminderSettings
                        )
                    }
                    Spacer(
                        Modifier
                            .padding(20.dp)
                            .height(1.dp)
                            .border(2.dp, Color.Gray)
                            .fillMaxWidth()
                    )
                    Row(
                        horizontalArrangement = Arrangement.SpaceBetween,
                        modifier = Modifier.fillMaxWidth()
                    ) {
                        Column {
                            Text(
                                modifier = Modifier.padding(vertical = 8.dp, horizontal = 20.dp),
                                text = "Backup",
                                fontWeight = FontWeight.W500,
                                fontSize = 18.sp,
                                color = Gray900
                            )
                            Text(
                                modifier = Modifier.padding(vertical = 2.dp, horizontal = 20.dp),
                                text = "Backup all your data\nto a file",
                                fontWeight = FontWeight.W500,
                                fontSize = 12.sp,
                                color = Gray900
                            )
                        }

                        TextButton(
                            enabled = true,
                            onClick = {
                                viewModel.backupData(roomBackup)
                            },
                            modifier = Modifier
                                .padding(horizontal = 20.dp)
                                .align(Alignment.Bottom)
                        ) {
                            Text(text = "Backup Data", color = Gray900)
                        }
                    }
                    Row(
                        horizontalArrangement = Arrangement.SpaceBetween,
                        modifier = Modifier.fillMaxWidth()
                    ) {
                        Column {
                            Text(
                                modifier = Modifier.padding(vertical = 8.dp, horizontal = 20.dp),
                                text = "Restore",
                                fontWeight = FontWeight.W500,
                                fontSize = 18.sp,
                                color = Gray900
                            )
                            Text(
                                modifier = Modifier.padding(vertical = 2.dp, horizontal = 20.dp),
                                text = "Restore data from a \nbackup file",
                                fontWeight = FontWeight.W500,
                                fontSize = 12.sp,
                                color = Gray900
                            )
                        }

                        TextButton(
                            enabled = true,
                            onClick = {
                                viewModel.restoreData(roomBackup)
                            },
                            modifier = Modifier
                                .padding(horizontal = 20.dp)
                                .align(Alignment.Bottom)
                        ) {
                            Text(text = "Restore Data", color = Gray900)
                        }
                    }
                    Spacer(
                        Modifier
                            .padding(20.dp)
                            .height(1.dp)
                            .border(2.dp, Color.Gray)
                            .fillMaxWidth()
                    )
                    Row(
                        horizontalArrangement = Arrangement.SpaceBetween,
                        modifier = Modifier.fillMaxWidth()
                    ) {
                        Column {
                            Text(
                                modifier = Modifier.padding(vertical = 8.dp, horizontal = 20.dp),
                                text = "Data Storage",
                                fontWeight = FontWeight.W500,
                                fontSize = 18.sp,
                                color = Gray900
                            )
                            Text(
                                modifier = Modifier.padding(vertical = 2.dp, horizontal = 20.dp),
                                text = "Delete all recorded data\non your device",
                                fontWeight = FontWeight.W500,
                                fontSize = 12.sp,
                                color = Gray900
                            )
                        }

                        TextButton(
                            enabled = true,
                            onClick = {
                                openAlertDialog.value = true
                            },
                            modifier = Modifier
                                .padding(horizontal = 20.dp)
                                .align(Alignment.Bottom)
                        ) {
                            Text(text = "Delete Data", color = Color.Red)
                        }
                    }
                    Spacer(
                        Modifier
                            .padding(20.dp)
                            .height(1.dp)
                            .border(2.dp, Color.Gray)
                            .fillMaxWidth()
                    )
                    if (uiState !== null) {
                        ImportSection(viewModel)
                    }
                    Spacer(
                        Modifier
                            .padding(20.dp)
                            .height(1.dp)
                            .border(2.dp, Color.Gray)
                            .fillMaxWidth()
                    )
                    Column {
                        Text(
                            modifier = Modifier.padding(vertical = 8.dp, horizontal = 20.dp),
                            text = "Version",
                            fontWeight = FontWeight.W500,
                            fontSize = 18.sp,
                            color = Gray900
                        )
                        Text(
                            modifier = Modifier.padding(vertical = 2.dp, horizontal = 20.dp),
                            text = BuildConfig.VERSION_NAME,
                            fontWeight = FontWeight.W500,
                            fontSize = 12.sp,
                            color = Gray900
                        )
                        ClickableText(
                            modifier = Modifier.padding(vertical = 2.dp, horizontal = 20.dp),
                            text = createdBy,
                            onClick = { offset ->
                                val clickedRange =
                                    createdBy.getStringAnnotations("URL", offset, offset)
                                clickedRange.firstOrNull()?.let { annotation ->
                                    // Handle the click action, for example, open a browser with the URL
                                    Intent(Intent.ACTION_VIEW, Uri.parse(annotation.item)).also {
                                        context.startActivity(it, null)
                                    }

                                }
                            })
                    }
                }
            }
        }
        when {
            openAlertDialog.value -> {
                AlertDialog(
                    title = {
                        Text(
                            text = "Delete all data", fontWeight = FontWeight.W500,
                            fontSize = 18.sp,
                            color = Gray900
                        )
                    },
                    text = {
                        Text(text = "Do you really want to delete all data?")
                    },
                    onDismissRequest = {
                        openAlertDialog.value = false
                    },
                    confirmButton = {
                        TextButton(
                            onClick = {
                                viewModel.deleteAllData()
                                openAlertDialog.value = false
                            }
                        ) {
                            Text("Delete", color = Color.Red)
                        }
                    },
                    dismissButton = {
                        TextButton(
                            onClick = {
                                openAlertDialog.value = false
                            }
                        ) {
                            Text("Cancel")
                        }
                    }
                )
            }
        }
    }
}


