package com.ngrob.android.bluemoon.features.settings.components

import androidx.compose.foundation.clickable
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.width
import androidx.compose.material.Switch
import androidx.compose.material.Text
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.OutlinedTextField
import androidx.compose.material3.OutlinedTextFieldDefaults
import androidx.compose.material3.TimePicker
import androidx.compose.material3.rememberTimePickerState
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.semantics.Role
import androidx.compose.ui.text.font.FontWeight
import androidx.compose.ui.tooling.preview.Preview
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.sp
import com.ngrob.android.bluemoon.core.model.ReminderSettings
import com.ngrob.android.bluemoon.core.ui.component.TimePickerDialog
import com.ngrob.android.bluemoon.core.ui.theme.BleedingRed
import com.ngrob.android.bluemoon.core.ui.theme.BluemoonTheme
import com.ngrob.android.bluemoon.core.ui.theme.Gray900
import com.ngrob.android.bluemoon.core.ui.theme.MoodGreen
import java.time.LocalTime

@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun RemindersParagraph(
    pillTrackingEnabled: Boolean,
    reminderSettings: ReminderSettings,
    onReminderSettingsChanged: (ReminderSettings) -> Unit
) {
    var timePickerDialogOpen by remember { mutableStateOf(false) }
    val timePickerState = rememberTimePickerState(
        reminderSettings.pillReminderTime.hour,
        reminderSettings.pillReminderTime.minute,
        true
    )

    if (timePickerDialogOpen) {
        TimePickerDialog(
            onCancel = { timePickerDialogOpen = false },
            onConfirm = {
                timePickerDialogOpen = false
                onReminderSettingsChanged(
                    reminderSettings.copy(
                        pillReminderTime = LocalTime.of(
                            timePickerState.hour,
                            timePickerState.minute
                        )
                    )
                )
            },
            content = {
                TimePicker(
                    state = timePickerState,
                )
            }
        )
    }
    Column {
        Text(
            modifier = Modifier.padding(vertical = 8.dp, horizontal = 20.dp),
            text = "Reminders",
            fontWeight = FontWeight.W500,
            fontSize = 18.sp,
            color = Gray900
        )
        Row(
            horizontalArrangement = Arrangement.SpaceBetween,
            verticalAlignment = Alignment.CenterVertically,
            modifier = Modifier.fillMaxWidth()
        ) {
            Text(
                modifier = Modifier.padding(vertical = 2.dp, horizontal = 20.dp),
                text = "Remind me in the case \nof an upcoming period",
                fontWeight = FontWeight.W500,
                fontSize = 12.sp,
                color = Gray900
            )
            Switch(
                modifier = Modifier
                    .padding(horizontal = 20.dp),
                checked = reminderSettings.remindForUpcomingPeriod,
                onCheckedChange = {
                    onReminderSettingsChanged(
                        reminderSettings.copy(remindForUpcomingPeriod = it)
                    )
                },
                colors = androidx.compose.material.SwitchDefaults.colors(
                    checkedThumbColor = MoodGreen,
                    checkedTrackColor = MoodGreen,
                    checkedTrackAlpha = 0.4f,
                    uncheckedThumbColor = BleedingRed,
                    uncheckedTrackAlpha = 0.4f,
                    uncheckedTrackColor = BleedingRed,
                )
            )
        }
        Row(
            horizontalArrangement = Arrangement.SpaceBetween,
            verticalAlignment = Alignment.CenterVertically,
            modifier = Modifier.fillMaxWidth()
        ) {
            Text(
                modifier = Modifier.padding(vertical = 2.dp, horizontal = 20.dp),
                text = "Remind me daily \nto track symptoms",
                fontWeight = FontWeight.W500,
                fontSize = 12.sp,
                color = Gray900
            )
            Switch(
                modifier = Modifier
                    .padding(horizontal = 20.dp),
                checked = reminderSettings.remindForTrackingSymptomsDaily,
                onCheckedChange = {
                    onReminderSettingsChanged(
                        reminderSettings.copy(remindForTrackingSymptomsDaily = it)
                    )
                },
                colors = androidx.compose.material.SwitchDefaults.colors(
                    checkedThumbColor = MoodGreen,
                    checkedTrackColor = MoodGreen,
                    checkedTrackAlpha = 0.4f,
                    uncheckedThumbColor = BleedingRed,
                    uncheckedTrackAlpha = 0.4f,
                    uncheckedTrackColor = BleedingRed,
                )
            )
        }
        Row(
            horizontalArrangement = Arrangement.SpaceBetween,
            verticalAlignment = Alignment.CenterVertically,
            modifier = Modifier.fillMaxWidth()
        ) {
            Text(
                modifier = Modifier.padding(vertical = 2.dp, horizontal = 20.dp),
                text = "Remind me to \ntake the pill",
                fontWeight = FontWeight.W500,
                fontSize = 12.sp,
                color = Gray900
            )
            Switch(
                modifier = Modifier
                    .padding(horizontal = 20.dp), checked = reminderSettings.remindForTakingPill,
                onCheckedChange = {
                    onReminderSettingsChanged(
                        reminderSettings.copy(remindForTakingPill = it)
                    )

                },
                enabled = pillTrackingEnabled,
                colors = androidx.compose.material.SwitchDefaults.colors(
                    checkedThumbColor = MoodGreen,
                    checkedTrackColor = MoodGreen,
                    checkedTrackAlpha = 0.4f,
                    uncheckedThumbColor = BleedingRed,
                    uncheckedTrackAlpha = 0.4f,
                    uncheckedTrackColor = BleedingRed,
                )
            )
        }
        if (reminderSettings.remindForTakingPill) {
            Row(
                horizontalArrangement = Arrangement.SpaceBetween,
                verticalAlignment = Alignment.CenterVertically,
                modifier = Modifier.fillMaxWidth()
            ) {
                Text(
                    modifier = Modifier.padding(vertical = 2.dp, horizontal = 20.dp),
                    text = "Time of\npill reminder",
                    fontWeight = FontWeight.W500,
                    fontSize = 12.sp,
                    color = Gray900
                )
                OutlinedTextField(
                    readOnly = true,
                    enabled = false,
                    value = LocalTime.of(timePickerState.hour, timePickerState.minute).toString(),
                    onValueChange = {},
                    modifier = Modifier
                        .padding(20.dp)
                        .width(120.dp)
                        .height(50.dp)
                        .clickable(enabled = true, role = Role.Button, onClick = {
                            timePickerDialogOpen = true

                        }),
                    colors = OutlinedTextFieldDefaults.colors(
                        disabledTextColor = MaterialTheme.colorScheme.onSurface,
                        disabledBorderColor = MaterialTheme.colorScheme.outline,
                        disabledLeadingIconColor = MaterialTheme.colorScheme.onSurfaceVariant,
                        disabledTrailingIconColor = MaterialTheme.colorScheme.onSurfaceVariant,
                        disabledLabelColor = MaterialTheme.colorScheme.onSurfaceVariant,
                        //For Icons
                        disabledPlaceholderColor = MaterialTheme.colorScheme.onSurfaceVariant,
                    )
                )
            }
        }
    }
}

@Preview(showBackground = true)
@Composable
fun RemindersParagraphPreview() {
    BluemoonTheme {
        RemindersParagraph(
            true,
            ReminderSettings(
                true,
                true,
                true,
                LocalTime.of(18, 0)
            ),
            onReminderSettingsChanged = {},
        )
    }
}