package com.ngrob.android.bluemoon.features.settings.components

import androidx.compose.foundation.clickable
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.width
import androidx.compose.material3.DatePicker
import androidx.compose.material3.DatePickerDialog
import androidx.compose.material3.DropdownMenuItem
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.ExposedDropdownMenuBox
import androidx.compose.material3.ExposedDropdownMenuDefaults
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.OutlinedTextField
import androidx.compose.material3.OutlinedTextFieldDefaults
import androidx.compose.material3.SelectableDates
import androidx.compose.material3.Text
import androidx.compose.material3.TextButton
import androidx.compose.material3.rememberDatePickerState
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.semantics.Role
import androidx.compose.ui.text.font.FontWeight
import androidx.compose.ui.text.style.TextAlign
import androidx.compose.ui.tooling.preview.Preview
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.sp
import com.ngrob.android.bluemoon.core.model.PillRecipe
import com.ngrob.android.bluemoon.core.ui.theme.BluemoonTheme
import com.ngrob.android.bluemoon.core.ui.theme.Gray900
import java.time.Instant
import java.time.LocalDate
import java.time.ZoneId
import java.time.format.DateTimeFormatter


@OptIn(ExperimentalMaterial3Api::class)
object PastOrPresentSelectableDates : SelectableDates {
    override fun isSelectableDate(utcTimeMillis: Long): Boolean {
        return utcTimeMillis <= System.currentTimeMillis()
    }

    override fun isSelectableYear(year: Int): Boolean {
        return year <= LocalDate.now().year
    }
}


@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun PillRecipe(
    recipe: PillRecipe,
    onRecipeChange: (recipe: PillRecipe) -> Unit,
) {
    val numberOfPills = recipe.numberOfPills
    val firstIntake = recipe.firstIntake

    val numberOfPillsOptions = arrayOf("21", "24", "28", "91")

    var selectedNumberOfPills by remember { mutableStateOf(numberOfPillsOptions.find { it == numberOfPills.toString() }!!) }
    var datePickerDialogOpen by remember { mutableStateOf(false) }
    var pilNumberMenuOpen by remember {
        mutableStateOf(false)
    }
    val datePickerState = rememberDatePickerState(
        initialSelectedDateMillis = firstIntake.atStartOfDay(ZoneId.systemDefault()).toInstant()
            .toEpochMilli(),
        initialDisplayedMonthMillis = null,
        selectableDates = PastOrPresentSelectableDates
    )
    if (datePickerDialogOpen) {
        DatePickerDialog(onDismissRequest = {}, dismissButton = {
            TextButton(onClick = {
                datePickerDialogOpen = false
            }) {
                Text(text = "Cancel")
            }

        }, confirmButton = {
            TextButton(onClick = {
                datePickerDialogOpen = false
                onRecipeChange(
                    PillRecipe(
                        numberOfPills = numberOfPills,
                        firstIntake = Instant.ofEpochMilli(datePickerState.selectedDateMillis!!)
                            .atZone(ZoneId.systemDefault()).toLocalDate()
                    )

                )
            }) {
                Text(text = "Confirm")
            }

        }, content = {
            DatePicker(state = datePickerState, title = {
                Text(
                    text = "First intake of Pill",
                    Modifier
                        .fillMaxWidth()
                        .padding(vertical = 5.dp), textAlign = TextAlign.Center
                )
            })

        })
    }
    Column(Modifier.fillMaxWidth(), horizontalAlignment = Alignment.End) {
        Text(
            modifier = Modifier
                .fillMaxWidth()
                .padding(horizontal = 20.dp, vertical = 4.dp),
            text = "Pill Recipe",
            fontWeight = FontWeight.W500,
            fontSize = 18.sp,
            color = Gray900
        )
        Row(
            verticalAlignment = Alignment.CenterVertically,
            horizontalArrangement = Arrangement.SpaceBetween,
            modifier = Modifier.fillMaxWidth()
        ) {
            Text(
                modifier = Modifier.padding(vertical = 2.dp, horizontal = 20.dp),
                text = "Number of Pills\nin pack",
                fontWeight = FontWeight.W500,
                fontSize = 12.sp,
                color = Gray900
            )
            ExposedDropdownMenuBox(
                expanded = pilNumberMenuOpen,
                onExpandedChange = {
                    pilNumberMenuOpen = !pilNumberMenuOpen
                }
            ) {
                OutlinedTextField(
                    value = selectedNumberOfPills,
                    onValueChange = {
                    },
                    readOnly = true,
                    enabled = false,
                    trailingIcon = { ExposedDropdownMenuDefaults.TrailingIcon(expanded = pilNumberMenuOpen) },
                    modifier = Modifier
                        .padding(horizontal = 20.dp)
                        .width(120.dp)
                        .height(50.dp)
                        .menuAnchor(),
                    colors = OutlinedTextFieldDefaults.colors(
                        disabledTextColor = MaterialTheme.colorScheme.onSurface,
                        disabledBorderColor = MaterialTheme.colorScheme.outline,
                        disabledLeadingIconColor = MaterialTheme.colorScheme.onSurfaceVariant,
                        disabledTrailingIconColor = MaterialTheme.colorScheme.onSurfaceVariant,
                        disabledLabelColor = MaterialTheme.colorScheme.onSurfaceVariant,
                        //For Icons
                        disabledPlaceholderColor = MaterialTheme.colorScheme.onSurfaceVariant,
                    )

                )

                ExposedDropdownMenu(
                    modifier = Modifier.width(120.dp),
                    expanded = pilNumberMenuOpen,
                    onDismissRequest = { pilNumberMenuOpen = false }
                ) {
                    numberOfPillsOptions.forEach { item ->
                        DropdownMenuItem(
                            text = { Text(text = item) },
                            onClick = {
                                selectedNumberOfPills = item
                                pilNumberMenuOpen = false
                                onRecipeChange(
                                    PillRecipe(
                                        numberOfPills = item.toInt(),
                                        firstIntake = firstIntake
                                    )
                                )

                            }
                        )
                    }


                }
            }

        }
        Row(
            verticalAlignment = Alignment.CenterVertically,
            horizontalArrangement = Arrangement.SpaceBetween,
            modifier = Modifier.fillMaxWidth()
        ) {
            Text(
                modifier = Modifier.padding(vertical = 2.dp, horizontal = 20.dp),
                text = "Date of first\nintake",
                fontWeight = FontWeight.W500,
                fontSize = 12.sp,
                color = Gray900
            )

            OutlinedTextField(
                readOnly = true,
                enabled = false,
                value = firstIntake.format(DateTimeFormatter.ISO_LOCAL_DATE),
                onValueChange = {},
                modifier = Modifier
                    .padding(20.dp)
                    .width(120.dp)
                    .height(50.dp)
                    .clickable(enabled = true, role = Role.Button, onClick = {
                        datePickerDialogOpen = true

                    }),
                colors = OutlinedTextFieldDefaults.colors(
                    disabledTextColor = MaterialTheme.colorScheme.onSurface,
                    disabledBorderColor = MaterialTheme.colorScheme.outline,
                    disabledLeadingIconColor = MaterialTheme.colorScheme.onSurfaceVariant,
                    disabledTrailingIconColor = MaterialTheme.colorScheme.onSurfaceVariant,
                    disabledLabelColor = MaterialTheme.colorScheme.onSurfaceVariant,
                    //For Icons
                    disabledPlaceholderColor = MaterialTheme.colorScheme.onSurfaceVariant,
                )


            )


        }
    }
}

@Preview(showBackground = true)
@Composable
fun PillRecipePreview() {
    BluemoonTheme {
        PillRecipe(
            recipe = PillRecipe(
                24,
                LocalDate.now()
            ),
            onRecipeChange = {}
        )
    }
}