package com.ngrob.android.bluemoon.features.edit_period.screens

import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import com.ngrob.android.bluemoon.MenstruationCycleTracker
import com.ngrob.android.bluemoon.core.data.repository.BleedingRepository
import dagger.hilt.android.lifecycle.HiltViewModel
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.asStateFlow
import kotlinx.coroutines.flow.update
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import java.time.Clock
import java.time.LocalDate
import java.time.YearMonth
import javax.inject.Inject

data class EditPeriodUiState(
    val bleedingDatesToDelete: List<LocalDate> = listOf(),
    val bleedingDays: List<LocalDate> = listOf(),
    val cycleTracker: MenstruationCycleTracker = MenstruationCycleTracker(listOf()),
    val currentMonth: YearMonth,
    val saveButtonEnabled: Boolean = true,
)

@HiltViewModel
class EditPeriodViewModel @Inject constructor(
    val bleedingRepository: BleedingRepository,
    val clock: Clock,
) : ViewModel() {

    private val _uiState = MutableStateFlow(EditPeriodUiState(currentMonth = YearMonth.now(clock)))
    val uiState = _uiState.asStateFlow()

    init {
        viewModelScope.launch {
            bleedingRepository.getAllBleedingDates().collect { bleedingDates ->
                _uiState.update { currentState ->
                    currentState.copy(
                        bleedingDays = bleedingDates,
                        cycleTracker = MenstruationCycleTracker(bleedingDates = bleedingDates)
                    )
                }
            }
        }
    }


    fun onBleedingDayClicked(date: LocalDate) {
        _uiState.update { currentState ->
            val newBleedingDaysToDelete = currentState.bleedingDatesToDelete.toMutableList()
            val newBleeding = currentState.bleedingDays.toMutableList()

            if (date in currentState.bleedingDays) {
                newBleeding.remove(date)
                newBleedingDaysToDelete.add(date)
            } else {
                newBleeding.add(date)
                newBleedingDaysToDelete.filter { it != date }
            }
            val newBleedingDays = newBleeding.toList()

            currentState.copy(
                bleedingDays = newBleedingDays, cycleTracker = MenstruationCycleTracker(
                    newBleedingDays
                ), bleedingDatesToDelete = newBleedingDaysToDelete.toList()
            )
        }

    }

    fun saveData() {
        viewModelScope.launch {
            val bleedingDays: List<LocalDate> =
                // Collect the latest value of bleedingDays flow on the IO dispatcher
                _uiState.value.bleedingDays
            val bleedingDatesToDelete: List<LocalDate> = _uiState.value.bleedingDatesToDelete
            withContext(Dispatchers.IO) {
                bleedingRepository.addBleedings(bleedingDays)
                bleedingRepository.deleteBleedings(bleedingDatesToDelete)
            }

        }
    }

}