package com.ngrob.android.bluemoon.features.edit_period.components

import androidx.compose.foundation.background
import androidx.compose.foundation.border
import androidx.compose.foundation.clickable
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.size
import androidx.compose.foundation.shape.CircleShape
import androidx.compose.material.Text
import androidx.compose.runtime.Composable
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.draw.clip
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.text.font.FontWeight
import androidx.compose.ui.tooling.preview.Preview
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.sp
import androidx.compose.ui.zIndex
import com.ngrob.android.bluemoon.MenstruationCycleTracker
import com.ngrob.android.bluemoon.core.ui.theme.BleedingRed
import com.ngrob.android.bluemoon.core.ui.theme.BluemoonTheme
import com.ngrob.android.bluemoon.core.ui.theme.Gray900
import com.ngrob.android.bluemoon.features.calendar.components.CustomWeekHeader
import com.ngrob.android.bluemoon.features.calendar.components.MonthHeader
import io.github.boguszpawlowski.composecalendar.CalendarState
import io.github.boguszpawlowski.composecalendar.StaticCalendar
import io.github.boguszpawlowski.composecalendar.day.DayState
import io.github.boguszpawlowski.composecalendar.rememberCalendarState
import io.github.boguszpawlowski.composecalendar.selection.EmptySelectionState
import java.time.LocalDate
import java.time.format.DateTimeFormatter
import java.time.format.FormatStyle


@Composable
fun CustomEditDay(
    modifier: Modifier =Modifier,
    state: DayState<EmptySelectionState>,
    isBleedingDay: Boolean,
    cycleDay: Long?,
    onDayClicked: (date: LocalDate) -> Unit
) {
    val selectionState = state.selectionState
    val date = state.date
    val dateFormatted = date.format(DateTimeFormatter.ofLocalizedDate(FormatStyle.MEDIUM))
    Box(modifier = modifier.padding(5.dp)) {
        Box(
            modifier = Modifier
                .padding(2.dp)
                .border(1.dp, Color.White, shape = CircleShape)
                .padding(5.dp)
                .clickable(
                    onClickLabel = if (isBleedingDay) "Remove bleeding day on $dateFormatted" else "Add bleeding day on $dateFormatted",
                    onClick = { selectionState.onDateSelected(date); onDayClicked(date) }),
            contentAlignment = Alignment.Center
        ) {
            Column(
                Modifier
                    .size(20.dp)
                    .clip(CircleShape)
                    .background(if (isBleedingDay) BleedingRed else Color.Transparent),
                horizontalAlignment = Alignment.CenterHorizontally,
                verticalArrangement = Arrangement.Center
            ) {
                Text(
                    text = date.dayOfMonth.toString(),
                    color = Color.White,
                    fontWeight = if (state.isCurrentDay) FontWeight.Bold else FontWeight.Normal,
                    fontSize = 12.sp
                )
            }
        }
        if (cycleDay != null) {
            Box(
                modifier = Modifier
                    .zIndex(1f)
                    .size(15.dp)
                    .clip(CircleShape)
                    .background(Color.White),
                contentAlignment = Alignment.Center

            ) {
                Text(
                    text = cycleDay.toString(),
                    fontSize = 8.sp,
                    color = Color.Black,
                    fontWeight = if (state.isCurrentDay) FontWeight.Bold else FontWeight.Normal,

                    )
            }
        }

    }

}

@Composable
fun EditCalendar(
    modifier: Modifier = Modifier,
    calendarState: CalendarState<EmptySelectionState>,
    bleedingDays: List<LocalDate>,
    cycleTracker: MenstruationCycleTracker,
    onDayClicked: (date: LocalDate) -> Unit,
) {

    StaticCalendar(
        modifier.background(color = Gray900),
        dayContent = { dayState ->
            val isBleedingDay = dayState.date in bleedingDays
            val cycleDay =
                if (isBleedingDay) cycleTracker.getCycleDayOfBleedingDate(dayState.date) else null

            CustomEditDay(modifier = Modifier.align(Alignment.Center), dayState, isBleedingDay, cycleDay, onDayClicked)
        },
        daysOfWeekHeader = { CustomWeekHeader(daysOfWeek = it) },
        showAdjacentMonths = false,
        monthHeader = { MonthHeader(monthState = it) },
        calendarState = calendarState,
        horizontalSwipeEnabled = false
    )
}

@Preview(showBackground = true)
@Composable
fun EditCalendarPreview() {
    BluemoonTheme {
        EditCalendar(
            calendarState = rememberCalendarState(),
            bleedingDays = listOf(LocalDate.now(), LocalDate.of(2023, 12, 4)),
            cycleTracker = MenstruationCycleTracker(
                bleedingDates = listOf(
                    LocalDate.now(),
                    LocalDate.of(2023, 12, 4)
                )
            ),
            onDayClicked = {}
        )
    }
}