package com.ngrob.android.bluemoon.features.dashboard.screens

import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import com.ngrob.android.bluemoon.MenstruationCycleTracker
import com.ngrob.android.bluemoon.core.data.repository.BleedingRepository
import com.ngrob.android.bluemoon.core.data.repository.PillRepository
import com.ngrob.android.bluemoon.core.data.repository.UserDataRepository
import com.ngrob.android.bluemoon.core.database.model.Pill
import com.ngrob.android.bluemoon.core.model.PillRecipe
import com.ngrob.android.bluemoon.core.model.UserData
import dagger.hilt.android.lifecycle.HiltViewModel
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.flow.SharingStarted
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.combine
import kotlinx.coroutines.flow.stateIn
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import java.time.Clock
import java.time.LocalDate
import java.time.temporal.ChronoUnit
import java.util.stream.Collectors
import java.util.stream.LongStream
import javax.inject.Inject
import kotlin.time.Duration.Companion.seconds


enum class MainButtonState(private val displayText: String) {
    PERIOD_STARTED("Period started"), PERIOD_ENDED("Period ended"), EDIT_PERIOD("Edit period");

    override fun toString(): String {
        return displayText
    }
}

data class DashboardUiState(
    val bleedingDays: List<LocalDate> = listOf(),
    val cycleTracker: MenstruationCycleTracker = MenstruationCycleTracker(listOf()),
    val mainButtonState: MainButtonState = MainButtonState.PERIOD_STARTED,
    val showPillAction: Boolean = false,
    val pillState: PillState? = null,
    val currentDate: LocalDate
)

data class PillState(
    val todayCount: Int,
    val pillTaken: Boolean,
    val pillRecipe: PillRecipe,
    val isIntakePaused: Boolean
)

@HiltViewModel
class DashboardViewModel @Inject constructor(
    private val bleedingRepository: BleedingRepository,
    private val pillRepository: PillRepository,
    userDataRepository: UserDataRepository,
    private val clock: Clock
) : ViewModel() {

    val uiState: StateFlow<DashboardUiState> = combine(
        bleedingRepository.getAllBleedingDates(),
        userDataRepository.userData,
        pillRepository.getPillOfDay(LocalDate.now(clock))
    ) { bleedingDates, userData, pill ->
        val newCycleTracker = MenstruationCycleTracker(bleedingDates)
        val recentCycle = newCycleTracker.getRecentCycle()
        val meanCycleLength = newCycleTracker.getPeriodLengthMean()
        val newButtonState =
            if (recentCycle == null || recentCycle.endDate.plusDays(meanCycleLength.toLong()) <= LocalDate.now(
                    clock
                )
            ) {
                MainButtonState.PERIOD_STARTED
            } else if (recentCycle.endDate <= LocalDate.now(clock)) {
                MainButtonState.EDIT_PERIOD
            } else {
                MainButtonState.PERIOD_ENDED
            }

        val pillState = getPillState(userData, pill)

        DashboardUiState(
            currentDate = LocalDate.now(clock),
            mainButtonState = newButtonState,
            cycleTracker = newCycleTracker,
            bleedingDays = bleedingDates,
            showPillAction = userData.trackBirthControl,
            pillState = pillState
        )
    }.stateIn(
        viewModelScope,
        SharingStarted.WhileSubscribed(5.seconds.inWholeMilliseconds),
        initialValue = DashboardUiState(currentDate = LocalDate.now(clock))
    )

    private fun getPillState(userData: UserData, pill: Pill?): PillState {
        val pillRecipe = userData.pillRecipe
        val isIntakePaused = (ChronoUnit.DAYS.between(pillRecipe.firstIntake, LocalDate.now(clock))
            .toInt() + 1) % 29 > pillRecipe.numberOfPills
        val todayCount = (ChronoUnit.DAYS.between(pillRecipe.firstIntake, LocalDate.now(clock))
            .toInt() + 1) % 29 % (pillRecipe.numberOfPills + 1)
        val pillTaken = pill != null

        return PillState(
            todayCount = todayCount,
            pillRecipe = pillRecipe,
            isIntakePaused = isIntakePaused,
            pillTaken = pillTaken
        )
    }

    fun startPeriod() {
        val meanPeriodLength = uiState.value.cycleTracker.getPeriodLengthMean()

        val bleedingDatesToAdd =
            LongStream.iterate(0) { i -> i + 1 }.limit(meanPeriodLength.toLong()).mapToObj { i ->
                LocalDate.now(clock).plusDays(
                    i
                )
            }.collect(Collectors.toList())

        viewModelScope.launch {
            withContext(Dispatchers.IO) {
                bleedingRepository.addBleedings(bleedingDatesToAdd)
            }
        }
    }

    fun takePill() {
        viewModelScope.launch {
            withContext(Dispatchers.IO) {
                pillRepository.takePill(LocalDate.now(clock), isSubstitute = false)
            }
        }
    }
}
