package com.ngrob.android.bluemoon.features.dashboard.screens

import android.annotation.SuppressLint
import androidx.compose.animation.AnimatedVisibility
import androidx.compose.foundation.background
import androidx.compose.foundation.gestures.Orientation
import androidx.compose.foundation.gestures.rememberScrollableState
import androidx.compose.foundation.gestures.scrollable
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.size
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.foundation.verticalScroll
import androidx.compose.material.Card
import androidx.compose.material.Text
import androidx.compose.material3.Button
import androidx.compose.material3.ButtonDefaults
import androidx.compose.material3.CardDefaults
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableFloatStateOf
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.draw.shadow
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.text.font.FontWeight
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.sp
import androidx.hilt.navigation.compose.hiltViewModel
import androidx.navigation.NavHostController
import com.ngrob.android.bluemoon.core.ui.theme.BleedingRed
import com.ngrob.android.bluemoon.core.ui.theme.Gray600
import com.ngrob.android.bluemoon.core.ui.theme.Gray900
import com.ngrob.android.bluemoon.core.ui.theme.MedicationBlue
import com.ngrob.android.bluemoon.core.ui.theme.White
import com.ngrob.android.bluemoon.features.dashboard.components.CycleCircle
import com.ngrob.android.bluemoon.features.dashboard.components.GreetingHeader
import com.ngrob.android.bluemoon.features.dashboard.components.PhaseBarData
import com.ngrob.android.bluemoon.features.dashboard.components.PillActionCard
import com.ngrob.android.bluemoon.features.dashboard.components.WeekCalendar
import io.github.boguszpawlowski.composecalendar.rememberWeekCalendarState
import java.time.format.DateTimeFormatter
import java.time.format.FormatStyle
import java.time.temporal.ChronoUnit
import kotlin.math.roundToInt

@SuppressLint("UnusedMaterialScaffoldPaddingParameter")
@Composable
fun DashboardScreen(
    modifier: Modifier = Modifier,
    nav: NavHostController,
    viewModel: DashboardViewModel = hiltViewModel(),
) {
    val uiState by viewModel.uiState.collectAsState()
    val bleedingDays = uiState.bleedingDays
    var cycleDayNumber = 0
    var currentPeriodLength = 0
    val recentCycle = uiState.cycleTracker.getRecentCycle()
    if (recentCycle != null) {
        cycleDayNumber =
            (ChronoUnit.DAYS.between(recentCycle.startDate, uiState.currentDate) + 1).toInt()
        currentPeriodLength =
            (ChronoUnit.DAYS.between(recentCycle.startDate, recentCycle.endDate) + 1).toInt()
    }

    val calendarState = rememberWeekCalendarState()

    var scrollOffset by remember { mutableFloatStateOf(0f) }

    var circleVisible by remember { mutableStateOf(true) }

    LaunchedEffect(scrollOffset) {
        circleVisible = scrollOffset >= 0
    }


    Column(modifier.fillMaxSize())
    {
        GreetingHeader(name = "", modifier = Modifier.padding(10.dp, 5.dp, 15.dp, 0.dp))
        WeekCalendar(calendarState = calendarState, bleedingDays = bleedingDays)
        Column(
            Modifier.scrollable(
                orientation = Orientation.Vertical,
                // Scrollable state: describes how to consume
                // scrolling delta and update offset
                state = rememberScrollableState { delta ->
                    scrollOffset += delta
                    delta
                })
        ) {
            AnimatedVisibility(
                circleVisible, modifier = Modifier.align(Alignment.CenterHorizontally)
            ) {
                CycleCircle(
                    modifier = Modifier
                        .size(200.dp)
                        .padding(8.dp)
                        .align(Alignment.CenterHorizontally),
                    progressBars = when {
                        currentPeriodLength > 0 -> listOf(
                            PhaseBarData(
                                BleedingRed,
                                currentPeriodLength / 28f,
                                (8 - currentPeriodLength) / 28f
                            ),
                            PhaseBarData(MedicationBlue, 12 / 28f, 0.0f),
                        )

                        else -> listOf()
                    },
                    currentDayIndicationAngle = (((cycleDayNumber / 28f) * 360) % 360).roundToInt(),
                    cycleDayNumber = cycleDayNumber,
                )
            }
            Button(
                enabled = true,
                onClick = {
                    if (uiState.mainButtonState != MainButtonState.PERIOD_STARTED) {
                        nav.navigate("edit_period")
                        return@Button
                    }
                    viewModel.startPeriod()
                },
                colors = ButtonDefaults.buttonColors(
                    disabledContainerColor = Gray600, containerColor = White
                ),
                modifier = Modifier
                    .align(Alignment.CenterHorizontally)
                    .padding(0.dp, 10.dp, 0.dp, 20.dp)
            ) {
                Text(
                    text = uiState.mainButtonState.toString(),
                    color = Gray900,
                    fontWeight = FontWeight.Bold
                )
            }

            Card(
                modifier = Modifier.fillMaxSize(),
                backgroundColor = White,
                elevation = 0.dp,
                shape = RoundedCornerShape(30.dp, 30.dp, 0.dp, 0.dp)
            ) {
                Column {
                    Text(
                        modifier = Modifier.padding(vertical = 8.dp, horizontal = 20.dp),
                        text = uiState.currentDate
                            .format(DateTimeFormatter.ofLocalizedDate(FormatStyle.FULL)),
                        fontWeight = FontWeight.W500,
                        fontSize = 18.sp,
                        color = Gray900
                    )
                    Column(Modifier.verticalScroll(rememberScrollState(0))) {
                        if (recentCycle != null) {
                            androidx.compose.material3.Card(
                                colors = CardDefaults.elevatedCardColors(containerColor = Color.White),
                                modifier =
                                    Modifier
                                        .height(120.dp)
                                        .padding(vertical = 8.dp, horizontal = 20.dp)
                                        .fillMaxWidth()
                                        .shadow(
                                            2.dp, shape = RoundedCornerShape(12.dp)
                                        ),
                            ) {
                                Box(modifier = Modifier.padding(6.dp)) {
                                    Column(
                                        Modifier.fillMaxSize(),
                                    ) {
                                        Text(
                                            text = "Cycle",
                                            fontSize = 20.sp,
                                            fontWeight = FontWeight.W500,
                                        )
                                        Column {
                                            Text(
                                                text = "You're in the ${if (cycleDayNumber - 14 < 0) "first" else "second"} phase of your cycle",
                                                fontSize = 16.sp,
                                                color = Gray900
                                            )
                                            Text(
                                                text = "The chance to get pregnant is ${if (cycleDayNumber in 11..20) "high" else "low"}",
                                                fontSize = 16.sp,
                                                color = Gray900
                                            )
                                        }

                                    }
                                }
                            }
                        }
                        if (uiState.showPillAction) {
                            PillActionCard(
                                takePill = { viewModel.takePill() },
                                todayCount = uiState.pillState!!.todayCount,
                                totalPillNumber = uiState.pillState!!.pillRecipe.numberOfPills,
                                isIntakePaused = uiState.pillState!!.isIntakePaused,
                                pillTaken = uiState.pillState!!.pillTaken,
                                currentDate = uiState.currentDate
                            )
                        }
                    }

                }
            }
        }
    }
}


